/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2018 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blocks;

import net.minecraft.block.material.Material;
import net.minecraft.block.properties.PropertyBool;
import net.minecraft.block.properties.PropertyDirection;
import net.minecraft.block.state.BlockStateContainer;
import net.minecraft.block.state.IBlockState;
import net.minecraft.client.resources.I18n;
import net.minecraft.client.util.ITooltipFlag;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemStack;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumBlockRenderType;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumHand;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.text.TextFormatting;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;
import prospector.shootingstar.ShootingStar;
import prospector.shootingstar.model.ModelCompound;
import reborncore.api.ToolManager;
import reborncore.common.BaseTileBlock;
import reborncore.common.blocks.BlockWrenchEventHandler;
import reborncore.common.items.WrenchHelper;
import techreborn.lib.ModInfo;
import techreborn.tiles.TileAlarm;
import techreborn.utils.TechRebornCreativeTab;

import javax.annotation.Nullable;
import java.util.List;

public class BlockAlarm extends BaseTileBlock {
	public static PropertyDirection FACING;
	public static PropertyBool ACTIVE;
	private AxisAlignedBB[] bbs;

	public BlockAlarm() {
		super(Material.field_151576_e);
		func_149663_c("techreborn.alarm");
		this.func_180632_j(this.field_176227_L.func_177621_b().func_177226_a(FACING, EnumFacing.NORTH).func_177226_a(ACTIVE, false));
		this.bbs = GenBoundingBoxes(0.19, 0.81);
		func_149647_a(TechRebornCreativeTab.instance);
		ShootingStar.registerModel(new ModelCompound(ModInfo.MOD_ID, this, "machines/lighting"));
		BlockWrenchEventHandler.wrenableBlocks.add(this);
	}

	private static AxisAlignedBB[] GenBoundingBoxes(double depth, double width) {
		AxisAlignedBB[] dimm = {
			new AxisAlignedBB(width, 1.0 - depth, width, 1.0 - width, 1.0D, 1.0 - width),
			new AxisAlignedBB(width, 0.0D, width, 1.0 - width, depth, 1.0 - width),
			new AxisAlignedBB(width, width, 1.0 - depth, 1.0 - width, 1.0 - width, 1.0D),
			new AxisAlignedBB(width, width, 0.0D, 1.0 - width, 1.0 - width, depth),
			new AxisAlignedBB(1.0 - depth, width, width, 1.0D, 1.0 - width, 1.0 - width),
			new AxisAlignedBB(0.0D, width, width, depth, 1.0 - width, 1.0 - width),
		};
		return dimm;
	}

	public static boolean isActive(IBlockState state) {
		return state.func_177229_b(ACTIVE);
	}

	public static EnumFacing getFacing(IBlockState state) {
		return (EnumFacing) state.func_177229_b(FACING);
	}

	public static void setFacing(EnumFacing facing, World world, BlockPos pos) {
		world.func_175656_a(pos, world.func_180495_p(pos).func_177226_a(FACING, facing));
	}

	public static void setActive(boolean active, World world, BlockPos pos) {
		EnumFacing facing = world.func_180495_p(pos).func_177229_b(FACING);
		IBlockState state = world.func_180495_p(pos).func_177226_a(ACTIVE, active).func_177226_a(FACING, facing);
		world.func_180501_a(pos, state, 3);
	}

	@Override
	protected BlockStateContainer func_180661_e() {
		FACING = PropertyDirection.func_177714_a("facing");
		ACTIVE = PropertyBool.func_177716_a("active");
		return new BlockStateContainer(this, FACING, ACTIVE);
	}

	@Override
	public boolean func_180639_a(World worldIn, BlockPos pos, IBlockState state, EntityPlayer playerIn,
	                                EnumHand hand, EnumFacing side, float hitX, float hitY, float hitZ) {
		ItemStack stack = playerIn.func_184586_b(EnumHand.MAIN_HAND);
		TileEntity tileEntity = worldIn.func_175625_s(pos);

		// We extended BaseTileBlock. Thus we should always have tile entity. I hope.
		if (tileEntity == null) {
			return false;
		}

		if (!stack.func_190926_b() && ToolManager.INSTANCE.canHandleTool(stack)) {
			if (WrenchHelper.handleWrench(stack, worldIn, pos, playerIn, side)) {
				return true;
			}
		}

		if (!worldIn.field_72995_K && playerIn.func_70093_af()) {
			((TileAlarm) tileEntity).rightClick();
			return true;

		}

		return super.func_180639_a(worldIn, pos, state, playerIn, hand, side, hitX, hitY, hitZ);
	}

	@Override
	public int func_176201_c(IBlockState state) {
		int facingInt = state.func_177229_b(FACING).func_176745_a();
		int activeInt = state.func_177229_b(ACTIVE) ? 8 : 0;
		return facingInt + activeInt;
	}

	@Override
	public IBlockState func_176203_a(int meta) {
		Boolean active = (meta & 8) == 8;
		EnumFacing facing = EnumFacing.func_82600_a(meta & 7);
		return this.func_176223_P().func_177226_a(FACING, facing).func_177226_a(ACTIVE, active);
	}

	@Nullable
	@Override
	public TileEntity func_149915_a(World worldIn, int meta) {
		return new TileAlarm();
	}

	@Override
	public IBlockState getStateForPlacement(World world, BlockPos pos, EnumFacing facing, float hitX, float hitY,
	                                        float hitZ, int meta, EntityLivingBase placer, EnumHand hand) {
		return this.func_176223_P().func_177226_a(FACING, facing);
	}

	@Override
	public EnumBlockRenderType func_149645_b(IBlockState state) {
		return EnumBlockRenderType.MODEL;
	}

	@Override
	public boolean func_149662_c(IBlockState state) {
		return false;
	}

	@Override
	public boolean func_149730_j(IBlockState state) {
		return false;
	}

	@Override
	public boolean func_149686_d(IBlockState state) {
		return false;
	}

	@Override
	public AxisAlignedBB func_185496_a(IBlockState state, IBlockAccess source, BlockPos pos) {
		return this.bbs[getFacing(state).func_176745_a()];
	}

	@Override
	public void func_190948_a(final ItemStack stack, final World world, final List<String> tooltip, ITooltipFlag flag) {
		tooltip.add(TextFormatting.GRAY + I18n.func_135052_a("techreborn.tooltip.alarm"));
	}

}
