/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2018 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blocks.cable;

import net.minecraft.block.BlockContainer;
import net.minecraft.block.material.Material;
import net.minecraft.block.properties.IProperty;
import net.minecraft.block.properties.PropertyBool;
import net.minecraft.block.properties.PropertyEnum;
import net.minecraft.block.state.BlockStateContainer;
import net.minecraft.block.state.IBlockState;
import net.minecraft.creativetab.CreativeTabs;
import net.minecraft.entity.Entity;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemStack;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.*;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.RayTraceResult;
import net.minecraft.world.ChunkCache;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;
import net.minecraft.world.chunk.Chunk;
import net.minecraftforge.energy.CapabilityEnergy;
import reborncore.api.ToolManager;
import reborncore.common.RebornCoreConfig;
import reborncore.common.blocks.BlockWrenchEventHandler;
import reborncore.common.items.WrenchHelper;
import reborncore.common.registration.RebornRegistry;
import reborncore.common.registration.impl.ConfigRegistry;
import techreborn.init.ModBlocks;
import techreborn.init.ModSounds;
import techreborn.lib.ModInfo;
import techreborn.tiles.cable.TileCable;
import techreborn.utils.TechRebornCreativeTab;
import techreborn.utils.damageSources.ElectrialShockSource;

import javax.annotation.Nullable;
import java.security.InvalidParameterException;

/**
 * Created by modmuss50 on 19/05/2017.
 */
@RebornRegistry(modID = ModInfo.MOD_ID)
public class BlockCable extends BlockContainer {

	public static final PropertyBool EAST = PropertyBool.func_177716_a("east");
	public static final PropertyBool WEST = PropertyBool.func_177716_a("west");
	public static final PropertyBool NORTH = PropertyBool.func_177716_a("north");
	public static final PropertyBool SOUTH = PropertyBool.func_177716_a("south");
	public static final PropertyBool UP = PropertyBool.func_177716_a("up");
	public static final PropertyBool DOWN = PropertyBool.func_177716_a("down");
	public static final IProperty<EnumCableType> TYPE = PropertyEnum.func_177709_a("type", EnumCableType.class);

	@ConfigRegistry(config = "misc", category = "cable", key = "uninsulatedElectrocutionDamage", comment = "When true an uninsulated cable will cause damage to entities")
	public static boolean uninsulatedElectrocutionDamage = true;

	@ConfigRegistry(config = "misc", category = "cable", key = "uninsulatedElectrocutionSound", comment = "When true an uninsulated cable will create a spark sound when an entity touches it")
	public static boolean uninsulatedElectrocutionSound = true;

	@ConfigRegistry(config = "misc", category = "cable", key = "uninsulatedElectrocutionParticles", comment = "When true an uninsulated cable will create a spark when an entity touches it")
	public static boolean uninsulatedElectrocutionParticles = true;

	public BlockCable() {
		super(Material.field_151576_e);
		func_149711_c(1F);
		func_149752_b(8F);
		func_149647_a(TechRebornCreativeTab.instance);
		func_180632_j(func_176223_P().func_177226_a(EAST, false).func_177226_a(WEST, false).func_177226_a(NORTH, false).func_177226_a(SOUTH, false).func_177226_a(UP, false).func_177226_a(DOWN, false).func_177226_a(TYPE, EnumCableType.COPPER));
		BlockWrenchEventHandler.wrenableBlocks.add(this);
	}

	public static ItemStack getCableByName(String name, int count) {
		for (int i = 0; i < EnumCableType.values().length; i++) {
			if (EnumCableType.values()[i].func_176610_l().equalsIgnoreCase(name)) {
				return new ItemStack(ModBlocks.CABLE,
					count, i);
			}
		}
		throw new InvalidParameterException("The cable " + name + " could not be found.");
	}

	public static ItemStack getCableByName(String name) {
		return getCableByName(name, 1);
	}
	
	//see for more info https://www.reddit.com/r/feedthebeast/comments/5mxwq9/psa_mod_devs_do_you_call_worldgettileentity_from/
	public TileEntity getTileEntitySafely(IBlockAccess blockAccess, BlockPos pos) {
		if (blockAccess instanceof ChunkCache) {
			return ((ChunkCache) blockAccess).func_190300_a(pos, Chunk.EnumCreateEntityType.CHECK);
		} else {
			return blockAccess.func_175625_s(pos);
		}
	}

	public IProperty<Boolean> getProperty(EnumFacing facing) {
		switch (facing) {
			case EAST:
				return EAST;
			case WEST:
				return WEST;
			case NORTH:
				return NORTH;
			case SOUTH:
				return SOUTH;
			case UP:
				return UP;
			case DOWN:
				return DOWN;
			default:
				return EAST;
		}
	}
	
	// BlockContainer
	@Override
	public EnumBlockRenderType func_149645_b(IBlockState state) {
		return EnumBlockRenderType.MODEL;
	}
	
	@Nullable
	@Override
	public TileEntity func_149915_a(World worldIn, int meta) {
		return new TileCable(func_176203_a(meta).func_177229_b(TYPE));
	}

	// Block
	@Override
	public boolean func_180639_a(World worldIn, BlockPos pos, IBlockState state, EntityPlayer playerIn, EnumHand hand, EnumFacing side, float hitX, float hitY, float hitZ) {
		ItemStack stack = playerIn.func_184586_b(EnumHand.MAIN_HAND);
		TileEntity tileEntity = worldIn.func_175625_s(pos);

		// We should always have tile entity. I hope.
		if (tileEntity == null) {
			return false;
		}
		
		if (!stack.func_190926_b() && ToolManager.INSTANCE.canHandleTool(stack)) {
			if (WrenchHelper.handleWrench(stack, worldIn, pos, playerIn, side)) {
				return true;
			}
		}
		return super.func_180639_a(worldIn, pos, state, playerIn, hand, side, hitX, hitY, hitZ);
	}
	
	@Override
	public void getDrops(NonNullList<ItemStack> drops, IBlockAccess world, BlockPos pos, IBlockState state, int fortune) {	
		if (RebornCoreConfig.wrenchRequired){
			if (state.func_177229_b(TYPE) == EnumCableType.ICOPPER) {
				drops.add(getCableByName("copper", 1));				
			}
			else if (state.func_177229_b(TYPE) == EnumCableType.IGOLD) {
				drops.add(getCableByName("gold", 1));			
			}
			else if (state.func_177229_b(TYPE) == EnumCableType.IHV) {
				drops.add(getCableByName("hv", 1));
			}
			else {
				super.getDrops(drops, world, pos, state, fortune);
			}
		}
		else {
			super.getDrops(drops, world, pos, state, fortune);
		}
	}

	@Override
	public ItemStack getPickBlock(IBlockState state, RayTraceResult target, World world, BlockPos pos, EntityPlayer player) {
		return state.func_177229_b(TYPE).getStack();
	}

	@Override
	public int func_180651_a(IBlockState state) {
		return func_176201_c(state);
	}

	@Override
	public boolean func_149662_c(IBlockState state) {
		return false;
	}

	@Override
	public BlockRenderLayer func_180664_k() {
		return BlockRenderLayer.CUTOUT;
	}

	@Override
	public boolean func_176225_a(IBlockState blockState, IBlockAccess blockAccess, BlockPos pos, EnumFacing side) {
		if (blockState.func_177229_b(TYPE) == EnumCableType.GLASSFIBER)
			return false;
		else
			return true;
	}

	@Override
	public boolean func_149730_j(IBlockState state) {
		return false;
	}

	@Override
	protected BlockStateContainer func_180661_e() {
		return new BlockStateContainer(this, EAST, WEST, NORTH, SOUTH, UP, DOWN, TYPE);
	}

	@Override
	public IBlockState getStateForPlacement(World world, BlockPos pos, EnumFacing facing, float hitX, float hitY, float hitZ, int meta, EntityLivingBase placer, EnumHand hand) {
		return func_176203_a(meta);
	}

	@Override
	public void func_149666_a(CreativeTabs tab, NonNullList<ItemStack> list) {
		for (EnumCableType cableType : EnumCableType.values()) {
			list.add(new ItemStack(this, 1, cableType.ordinal()));
		}
	}

	@Override
	public int func_176201_c(IBlockState state) {
		return state.func_177229_b(TYPE).ordinal();
	}

	@Override
	public IBlockState func_176203_a(int meta) {
		return func_176223_P().func_177226_a(TYPE, EnumCableType.values()[meta]);
	}

	@Override
	public boolean func_149686_d(IBlockState state) {
		return false;
	}

	@Override
	public AxisAlignedBB func_185496_a(IBlockState state, IBlockAccess source, BlockPos pos) {
		state = state.func_185899_b(source, pos);
		float minSize = 0.3125F;
		float maxSize =  0.6875F;
		int thinkness = (int) state.func_177229_b(TYPE).cableThickness;
		if(thinkness == 6){
			minSize = 0.35F;
			maxSize = 0.65F;
		}
		float minX = state.func_177229_b(WEST) ? 0.0F : minSize;
		float minY = state.func_177229_b(DOWN) ? 0.0F : minSize;
		float minZ = state.func_177229_b(NORTH) ? 0.0F : minSize;
		float maxX = state.func_177229_b(EAST) ? 1.0F : maxSize;
		float maxY = state.func_177229_b(UP) ? 1.0F : maxSize;
		float maxZ = state.func_177229_b(SOUTH) ? 1.0F : maxSize;
		return new AxisAlignedBB((double) minX, (double) minY, (double) minZ, (double) maxX, (double) maxY, (double) maxZ);
	}

	@Override
	public IBlockState func_176221_a(IBlockState state, IBlockAccess worldIn, BlockPos pos) {
		IBlockState actualState = state;
		for (EnumFacing facing : EnumFacing.values()) {
			TileEntity tileEntity = getTileEntitySafely(worldIn, pos.func_177972_a(facing));
			if (tileEntity != null) {
				actualState = actualState.func_177226_a(getProperty(facing), tileEntity.hasCapability(CapabilityEnergy.ENERGY, facing.func_176734_d()));
			}
		}
		return actualState;
	}

	@Override
	public void func_180634_a(World worldIn, BlockPos pos, IBlockState state, Entity entity) {
		super.func_180634_a(worldIn, pos, state, entity);
		if (state.func_177229_b(TYPE).canKill && entity instanceof EntityLivingBase) {
			TileEntity tileEntity = worldIn.func_175625_s(pos);
			if (tileEntity != null && tileEntity instanceof TileCable) {
				TileCable tileCable = (TileCable) tileEntity;
				if (tileCable.power != 0) {
					if (uninsulatedElectrocutionDamage) {
						if (state.func_177229_b(TYPE) == EnumCableType.HV) {
							entity.func_70015_d(1);
						}
						entity.func_70097_a(new ElectrialShockSource(), 1F);
					}
					if (uninsulatedElectrocutionSound) {
						worldIn.func_184148_a(null, entity.field_70165_t, entity.field_70163_u,
							entity.field_70161_v, ModSounds.CABLE_SHOCK,
							SoundCategory.BLOCKS, 0.6F, 1F);
					}
					if (uninsulatedElectrocutionParticles) {
						worldIn.func_175688_a(EnumParticleTypes.CRIT, entity.field_70165_t, entity.field_70163_u, entity.field_70161_v, 0,
							0, 0);
					}
				}
			}
		}
	}
}
