/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2018 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blocks.tier1;

import com.google.common.collect.Lists;
import net.minecraft.block.state.BlockStateContainer;
import net.minecraft.block.state.IBlockState;
import net.minecraft.client.resources.I18n;
import net.minecraft.creativetab.CreativeTabs;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.init.Blocks;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumHand;
import net.minecraft.util.NonNullList;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.RayTraceResult;
import net.minecraft.util.text.TextComponentString;
import net.minecraft.util.text.TextFormatting;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;
import prospector.shootingstar.ShootingStar;
import prospector.shootingstar.model.ModelCompound;
import reborncore.api.IToolDrop;
import reborncore.api.ToolManager;
import reborncore.api.tile.IMachineGuiHandler;
import reborncore.common.blocks.BlockMachineBase;
import reborncore.common.blocks.PropertyString;
import reborncore.common.util.ArrayUtils;
import reborncore.common.util.ChatUtils;
import reborncore.common.util.StringUtils;
import techreborn.utils.TechRebornCreativeTab;
import techreborn.init.ModBlocks;
import techreborn.lib.MessageIDs;
import techreborn.lib.ModInfo;
import techreborn.tiles.tier1.TilePlayerDectector;

import java.util.List;
import java.util.Random;

public class BlockPlayerDetector extends BlockMachineBase {

	public static final String[] types = new String[] { "all", "others", "you" };
	static List<String> typeNamesList = Lists.newArrayList(ArrayUtils.arrayToLowercase(types));
	public PropertyString TYPE;

	public BlockPlayerDetector() {
		super(true);
		func_149647_a(TechRebornCreativeTab.instance);
		this.func_180632_j(this.func_176203_a(0));
		for (int i = 0; i < types.length; i++) {
			ShootingStar.registerModel(new ModelCompound(ModInfo.MOD_ID, this, i, "machines/tier1_machines").setInvVariant("type=" + types[i]));
		}
	}

	// BlockMachineBase
	@Override
	public TileEntity func_149915_a(final World world, final int meta) {
		return new TilePlayerDectector();
	}
	
	@Override
	public void func_180633_a(World worldIn, BlockPos pos, IBlockState state, EntityLivingBase placer,
	                            ItemStack stack) {
		super.func_180633_a(worldIn, pos, state, placer, stack);
		TileEntity tile = worldIn.func_175625_s(pos);
		if (tile instanceof TilePlayerDectector) {
			((TilePlayerDectector) tile).owenerUdid = placer.func_110124_au().toString();
		}
	}
	
	@Override
	public boolean func_180639_a(World worldIn, BlockPos pos, IBlockState state, EntityPlayer playerIn,
	                                EnumHand hand, EnumFacing side, float hitX, float hitY, float hitZ) {
		ItemStack stack = playerIn.func_184586_b(EnumHand.MAIN_HAND);
		TileEntity tileEntity = worldIn.func_175625_s(pos);
			
		if (tileEntity == null) {
			return super.func_180639_a(worldIn, pos, state, playerIn, hand, side, hitX, hitY, hitZ);
		}
		
		String type = state.func_177229_b(TYPE);
		String newType = type;
		TextFormatting color = TextFormatting.GREEN;
		
		if (!stack.func_190926_b() && ToolManager.INSTANCE.canHandleTool(stack)) {
			if (ToolManager.INSTANCE.handleTool(stack, pos, worldIn, playerIn, side, false)) {
				if (playerIn.func_70093_af()) {
					if (tileEntity instanceof IToolDrop) {
						ItemStack drop = ((IToolDrop) tileEntity).getToolDrop(playerIn);
						if (drop == null) {
							return false;
						}
						if (!drop.func_190926_b()) {
							func_180635_a(worldIn, pos, drop);
						}
						if (!worldIn.field_72995_K) {
							worldIn.func_180501_a(pos, Blocks.field_150350_a.func_176223_P(), 2);
						}
						return true;
					}
				} else {
					if (type.equals("all")) {
						newType = "others";
						color = TextFormatting.RED;
					} else if (type.equals("others")) {
						newType = "you";
						color = TextFormatting.BLUE;
					} else if (type.equals("you")) {
						newType = "all";
					}
					worldIn.func_175656_a(pos, state.func_177226_a(TYPE, newType));
				}
			}
		}

		if (worldIn.field_72995_K) {
			ChatUtils.sendNoSpamMessages(MessageIDs.playerDetectorID, new TextComponentString(
				TextFormatting.GRAY + I18n.func_135052_a("techreborn.message.detects") + " " + color
					+ StringUtils.toFirstCapital(newType)));
		}
		return true;
	}
	
	@Override
	protected BlockStateContainer func_180661_e() {
		TYPE = new PropertyString("type", types);
		return new BlockStateContainer(this, TYPE);
	}

	@Override
	public IBlockState func_176203_a(int meta) {
		if (meta > types.length) {
			meta = 0;
		}
		return func_176194_O().func_177621_b().func_177226_a(TYPE, typeNamesList.get(meta));
	}

	@Override
	public int func_176201_c(IBlockState state) {
		return typeNamesList.indexOf(state.func_177229_b(TYPE));
	}

	@Override
	public IMachineGuiHandler getGui() {
		return null;
	}
	
	// Block
	@Override
	public ItemStack getPickBlock(IBlockState state, RayTraceResult target, World world, BlockPos pos, EntityPlayer player) {
		return new ItemStack(ModBlocks.PLAYER_DETECTOR, 1, typeNamesList.indexOf(state.func_177229_b(TYPE)));
	}

	@Override
	public Item func_180660_a(IBlockState state, Random rand, int fortune) {
		return Item.func_150898_a(this);
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void func_149666_a(CreativeTabs creativeTabs, NonNullList<ItemStack> list) {
		for (int meta = 0; meta < types.length; meta++) {
			list.add(new ItemStack(this, 1, meta));
		}
	}

	@Override
	public boolean canConnectRedstone(IBlockState state, IBlockAccess world, BlockPos pos, EnumFacing side) {
		return true;
	}

	@Override
	public boolean func_149744_f(IBlockState state) {
		return true;
	}

	@Override
	public int func_180656_a(IBlockState blockState, IBlockAccess blockAccess, BlockPos pos, EnumFacing side) {
		TileEntity entity = blockAccess.func_175625_s(pos);
		if (entity instanceof TilePlayerDectector) {
			return ((TilePlayerDectector) entity).isProvidingPower() ? 15 : 0;
		}
		return 0;
	}

	@Override
	public int func_176211_b(IBlockState blockState, IBlockAccess blockAccess, BlockPos pos, EnumFacing side) {
		TileEntity entity = blockAccess.func_175625_s(pos);
		if (entity instanceof TilePlayerDectector) {
			return ((TilePlayerDectector) entity).isProvidingPower() ? 15 : 0;
		}
		return 0;
	}
}
