/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2018 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.blocks.transformers;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterators;
import net.minecraft.block.Block;
import net.minecraft.block.material.Material;
import net.minecraft.block.properties.PropertyDirection;
import net.minecraft.block.state.BlockStateContainer;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemStack;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumHand;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;
import prospector.shootingstar.ShootingStar;
import prospector.shootingstar.model.ModelCompound;
import reborncore.api.ToolManager;
import reborncore.common.BaseTileBlock;
import reborncore.common.blocks.BlockWrenchEventHandler;
import reborncore.common.items.WrenchHelper;
import techreborn.lib.ModInfo;
import techreborn.utils.TechRebornCreativeTab;

import java.util.Iterator;
import java.util.Random;

/**
 * Created by Rushmead
 */
public abstract class BlockTransformer extends BaseTileBlock {
	public static PropertyDirection FACING = PropertyDirection.func_177712_a("facing", Facings.ALL);
	public String name;

	public BlockTransformer(String name) {
		super(Material.field_151573_f);
		func_149711_c(2f);
		func_149647_a(TechRebornCreativeTab.instance);
		this.func_180632_j(this.field_176227_L.func_177621_b().func_177226_a(FACING, EnumFacing.NORTH));
		this.name = name;
		ShootingStar.registerModel(new ModelCompound(ModInfo.MOD_ID, this, "machines/energy"));
		BlockWrenchEventHandler.wrenableBlocks.add(this);
	}

	public void setFacing(EnumFacing facing, World world, BlockPos pos) {
		world.func_175656_a(pos, world.func_180495_p(pos).func_177226_a(FACING, facing));
	}
	
	public EnumFacing getFacing(IBlockState state) {
		return state.func_177229_b(FACING);
	}

	public EnumFacing getSideFromint(int i) {
		if (i == 0) {
			return EnumFacing.NORTH;
		} else if (i == 1) {
			return EnumFacing.SOUTH;
		} else if (i == 2) {
			return EnumFacing.EAST;
		} else if (i == 3) {
			return EnumFacing.WEST;
		} else if (i == 4) {
			return EnumFacing.UP;
		} else if (i == 5) {
			return EnumFacing.DOWN;
		}
		return EnumFacing.NORTH;
	}

	public int getSideFromEnum(EnumFacing facing) {
		if (facing == EnumFacing.NORTH) {
			return 0;
		} else if (facing == EnumFacing.SOUTH) {
			return 1;
		} else if (facing == EnumFacing.EAST) {
			return 2;
		} else if (facing == EnumFacing.WEST) {
			return 3;
		} else if (facing == EnumFacing.UP) {
			return 4;
		} else if (facing == EnumFacing.DOWN) {
			return 5;
		}
		return 0;
	}

	private void setDefaultFacing(World worldIn, BlockPos pos, IBlockState state) {
		if (!worldIn.field_72995_K) {
			IBlockState state0 = worldIn.func_180495_p(pos.func_177978_c());
			IBlockState state1 = worldIn.func_180495_p(pos.func_177968_d());
			IBlockState state2 = worldIn.func_180495_p(pos.func_177976_e());
			IBlockState state3 = worldIn.func_180495_p(pos.func_177974_f());
			EnumFacing enumfacing = (EnumFacing) state.func_177229_b(FACING);
			
			if (enumfacing == EnumFacing.NORTH && state0.func_185913_b() && !state1.func_185913_b()) {
				enumfacing = EnumFacing.SOUTH;
			} else if (enumfacing == EnumFacing.SOUTH && state1.func_185913_b() && !state0.func_185913_b()) {
				enumfacing = EnumFacing.NORTH;
			} else if (enumfacing == EnumFacing.WEST && state2.func_185913_b() && !state3.func_185913_b()) {
				enumfacing = EnumFacing.EAST;
			} else if (enumfacing == EnumFacing.EAST && state3.func_185913_b() && !state2.func_185913_b()) {
				enumfacing = EnumFacing.WEST;
			}
			
			worldIn.func_180501_a(pos, state.func_177226_a(FACING, enumfacing), 2);
		}
	}
	
	// Block
	@Override
	protected BlockStateContainer func_180661_e() {
		FACING = PropertyDirection.func_177712_a("facing", Facings.ALL);
		return new BlockStateContainer(this, FACING);
	}

	@Override
	public void func_176213_c(World worldIn, BlockPos pos, IBlockState state) {
		super.func_176213_c(worldIn, pos, state);
		this.setDefaultFacing(worldIn, pos, state);
	}

	@Override
	public void func_180633_a(World worldIn, BlockPos pos, IBlockState state, EntityLivingBase placer,
	                            ItemStack stack) {
		super.func_180633_a(worldIn, pos, state, placer, stack);
		EnumFacing facing = placer.func_174811_aO().func_176734_d();
		if (placer.field_70125_A < -50) {
			facing = EnumFacing.DOWN;
		} else if (placer.field_70125_A > 50) {
			facing = EnumFacing.UP;
		}
		setFacing(facing, worldIn, pos);
	}
	
	@Override
	public boolean func_180639_a(World worldIn, BlockPos pos, IBlockState state, EntityPlayer playerIn, EnumHand hand,
	                                EnumFacing side, float hitX, float hitY, float hitZ) {
		ItemStack stack = playerIn.func_184586_b(EnumHand.MAIN_HAND);
		TileEntity tileEntity = worldIn.func_175625_s(pos);

		// We extended BlockTileBase. Thus we should always have tile entity. I hope.
		if (tileEntity == null) {
			return false;
		}
	
		if (!stack.func_190926_b() && ToolManager.INSTANCE.canHandleTool(stack)) {
			if (WrenchHelper.handleWrench(stack, worldIn, pos, playerIn, side)) {
				return true;
			}
		}

		return super.func_180639_a(worldIn, pos, state, playerIn, hand, side, hitX, hitY, hitZ);
	}
	
	@Override
	public boolean rotateBlock(World world, BlockPos pos, EnumFacing side) {
		IBlockState state = world.func_180495_p(pos);
		Block block = state.func_177230_c();
		if (block instanceof BlockTransformer) {
			EnumFacing facing = state.func_177229_b(BlockTransformer.FACING);
			if (facing.func_176734_d() == side) {
				facing = side;
			} else {
				facing = side.func_176734_d();
			}
			world.func_175656_a(pos, state.func_177226_a(BlockTransformer.FACING, facing));
			return true;
		}

		return false;
	}
	
	@Override
	public int func_176201_c(IBlockState state) {
		int facingInt = getSideFromEnum(state.func_177229_b(FACING));
		return facingInt;
	}

	@Override
	public IBlockState func_176203_a(int meta) {
		EnumFacing facing = getSideFromint(meta);
		return this.func_176223_P().func_177226_a(FACING, facing);
	}

	public enum Facings implements Predicate<EnumFacing>, Iterable<EnumFacing> {
		ALL;

		public EnumFacing[] facings() {
			return new EnumFacing[] { EnumFacing.NORTH, EnumFacing.EAST, EnumFacing.SOUTH, EnumFacing.WEST,
				EnumFacing.UP, EnumFacing.DOWN };
		}

		public EnumFacing random(Random rand) {
			EnumFacing[] aenumfacing = this.facings();
			return aenumfacing[rand.nextInt(aenumfacing.length)];
		}

		@Override
		public boolean apply(EnumFacing p_apply_1_) {
			return p_apply_1_ != null;
		}

		@Override
		public Iterator<EnumFacing> iterator() {
			return Iterators.forArray(this.facings());
		}
	}
}
