/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2018 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.items;

import net.minecraft.block.Block;
import net.minecraft.block.BlockStaticLiquid;
import net.minecraft.block.material.Material;
import net.minecraft.block.state.IBlockState;
import net.minecraft.creativetab.CreativeTabs;
import net.minecraft.entity.Entity;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.ActionResult;
import net.minecraft.util.EnumActionResult;
import net.minecraft.util.EnumHand;
import net.minecraft.util.NonNullList;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.RayTraceResult;
import net.minecraft.world.World;
import net.minecraftforge.common.capabilities.ICapabilityProvider;
import net.minecraftforge.fluids.*;
import net.minecraftforge.fluids.capability.IFluidHandlerItem;
import net.minecraftforge.fluids.capability.templates.FluidHandlerItemStack;
import org.apache.commons.lang3.Validate;
import reborncore.common.util.StringUtils;
import techreborn.utils.TechRebornCreativeTab;
import techreborn.events.TRRecipeHandler;
import techreborn.init.ModItems;

/**
 * Created by modmuss50 on 17/05/2016.
 */
public class DynamicCell extends Item {

	public static final int CAPACITY = Fluid.BUCKET_VOLUME;

	public DynamicCell() {
		super();
		func_77637_a(TechRebornCreativeTab.instance);
		func_77655_b("techreborn.cell");
		func_77625_d(64);
		TRRecipeHandler.hideEntry(this);
	}

	@Override
	public void func_77663_a(ItemStack stack, World worldIn, Entity entityIn, int itemSlot, boolean isSelected) {
		//Clearing tag because ItemUtils.isItemEqual doesn't handle tags ForgeCaps and display
		//And breaks ability to use in recipes
		//TODO: Property ItemUtils.isItemEquals tags equality handling?
		if (stack.func_77942_o()) {
			NBTTagCompound tag = stack.func_77978_p();
			if (tag.func_186856_d() != 1 || tag.func_74764_b("Fluid")) {
				NBTTagCompound clearTag = new NBTTagCompound();
				clearTag.func_74782_a("Fluid", tag.func_74775_l("Fluid"));
				stack.func_77982_d(clearTag);
			}
		}
	}

	@Override
	public ActionResult<ItemStack> func_77659_a(World worldIn, EntityPlayer playerIn, EnumHand hand) {
		ItemStack stack = playerIn.func_184586_b(hand);
		if (!worldIn.field_72995_K) {
			RayTraceResult result = func_77621_a(worldIn, playerIn, true);

			if (result != null && result.field_72313_a == RayTraceResult.Type.BLOCK) {
				BlockPos pos = result.func_178782_a();
				IBlockState state = worldIn.func_180495_p(pos);
				Block block = state.func_177230_c();

				if (block instanceof IFluidBlock) {
					IFluidBlock fluidBlock = (IFluidBlock) block;

					if (fluidBlock.canDrain(worldIn, pos)) {
						FluidStack fluid = fluidBlock.drain(worldIn, pos, false);

						if (fluid != null && fluid.amount == DynamicCell.CAPACITY) {
							if (tryAddCellToInventory(playerIn, stack, fluid.getFluid())) {
								fluidBlock.drain(worldIn, pos, true);
								return ActionResult.newResult(EnumActionResult.SUCCESS, stack);
							}
						}

					}

				} else if (block instanceof BlockStaticLiquid) {
					Fluid fluid = state.func_185904_a() == Material.field_151587_i ? FluidRegistry.LAVA : FluidRegistry.WATER;

					if (tryAddCellToInventory(playerIn, stack, fluid)) {
						if (fluid != FluidRegistry.WATER)
							worldIn.func_175698_g(pos);
						return ActionResult.newResult(EnumActionResult.SUCCESS, stack);
					}

				} else {
					 ItemStack usedCell = stack.func_77946_l();
					 usedCell.func_190920_e(1);
					 IFluidHandlerItem fluidHandler = FluidUtil.getFluidHandler(usedCell);
					 if (fluidHandler != null) {
						 FluidStack fluid = fluidHandler.drain(DynamicCell.CAPACITY, false);
						 if (fluid != null){
							 if(FluidUtil.tryPlaceFluid(playerIn, worldIn, pos.func_177972_a(result.field_178784_b), fluidHandler, fluid)){
								 stack.func_190918_g(1);
								 playerIn.field_71071_by.func_70441_a(getEmptyCell(1));
								 return ActionResult.newResult(EnumActionResult.SUCCESS, stack);
							 }
							 return ActionResult.newResult(EnumActionResult.FAIL, stack);
						 }
						 return ActionResult.newResult(EnumActionResult.FAIL, stack);				 
					 }
					 return ActionResult.newResult(EnumActionResult.FAIL, stack);   
				}
			}
		}
		return ActionResult.newResult(EnumActionResult.FAIL, stack);
	}

	public boolean tryAddCellToInventory(EntityPlayer player, ItemStack stack, Fluid fluid) {
		if (player.field_71071_by.func_70441_a(DynamicCell.getCellWithFluid(fluid))) {
			stack.func_190918_g(1);
			return true;
		}
		return false;
	}

	@Override
	public void func_150895_a(CreativeTabs tab, NonNullList<ItemStack> subItems) {
		if (!func_194125_a(tab)) {
			return;
		}
		subItems.add(getEmptyCell(1));
		for (Fluid fluid : FluidRegistry.getRegisteredFluids().values()) {
			subItems.add(getCellWithFluid(fluid));
		}
	}

	@Override
	public String func_77653_i(ItemStack stack) {
		FluidStack fluidStack = getFluidHandler(stack).getFluid();
		if (fluidStack == null)
			return super.func_77653_i(stack);
		return StringUtils.t("item.techreborn.cell.fluid.name").replaceAll("\\$fluid\\$", fluidStack.getLocalizedName());
	}

	@Override
	public ICapabilityProvider initCapabilities(ItemStack stack, NBTTagCompound nbt) {
		return getFluidHandler(stack);
	}

	public static FluidHandler getFluidHandler(ItemStack stack) {
		return new FluidHandler(stack, CAPACITY);
	}

	public static ItemStack getCellWithFluid(Fluid fluid, int stackSize) {
		Validate.notNull(fluid);
		ItemStack stack = new ItemStack(ModItems.CELL);
		getFluidHandler(stack).fill(new FluidStack(fluid, CAPACITY), true);
		stack.func_190920_e(stackSize);
		return stack;
	}

	public static ItemStack getEmptyCell(int amount) {
		return new ItemStack(ModItems.CELL, amount);
	}

	public static ItemStack getCellWithFluid(Fluid fluid) {
		return getCellWithFluid(fluid, 1);
	}

	public static class FluidHandler extends FluidHandlerItemStack {

		public FluidHandler(ItemStack container, int capacity) {
			super(container, capacity);

			//backwards compatibility
			if (container.func_77942_o() && container.func_77978_p().func_74764_b("FluidName")) {
				FluidStack stack = FluidStack.loadFluidStackFromNBT(container.func_77978_p());
				if (stack != null) {
					container.func_77982_d(new NBTTagCompound());
					fill(stack, true);
				}
			}

		}

		@Override
		public int fill(FluidStack resource, boolean doFill) {
			if(resource == null)
				return 0;
			//Done to allow mods that try to move max int of fluid, allows the cells to work with thermal tanks.
			if(resource.amount > capacity){
				resource.amount = capacity;
			}
			if (resource.amount != capacity)
				return 0;
			return super.fill(resource, doFill);
		}

		@Override
		public FluidStack drain(int maxDrain, boolean doDrain) {
			//Done to allow mods that try to move max int of fluid, allows the cells to work with thermal tanks.
			if(maxDrain > capacity){
				maxDrain = capacity;
			}
			if (maxDrain != capacity)
				return null;
			return super.drain(maxDrain, doDrain);
		}

		@Override
		public ItemStack getContainer() {
			ItemStack cell;
			if (container.func_77942_o() && container.func_77978_p().func_74764_b(FLUID_NBT_KEY)) {
				cell = super.getContainer();
			}
			else {
				cell = new ItemStack(ModItems.CELL, 1);
			}
			return cell;
		}
		
		@Override
		protected void setContainerToEmpty() {
			container = new ItemStack(ModItems.CELL, 1);
		}
	}
}
