/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2018 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.items;

import net.minecraft.creativetab.CreativeTabs;
import net.minecraft.inventory.Container;
import net.minecraft.item.ItemStack;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.NonNullList;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;
import reborncore.api.tile.IUpgrade;
import reborncore.common.powerSystem.TilePowerAcceptor;
import reborncore.common.recipes.IUpgradeHandler;
import reborncore.common.registration.RebornRegistry;
import reborncore.common.registration.impl.ConfigRegistry;
import reborncore.common.tile.TileLegacyMachineBase;
import techreborn.init.ModItems;
import techreborn.lib.ModInfo;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.security.InvalidParameterException;

@RebornRegistry(modID = ModInfo.MOD_ID)
public class ItemUpgrades extends ItemTR implements IUpgrade {

	@ConfigRegistry(config = "items", category = "upgrades", key = "overclocker_speed", comment = "Overclocker upgrade speed multipiler")
	public static double overclockerSpeed = 0.25;

	@ConfigRegistry(config = "items", category = "upgrades", key = "overclocker_power", comment = "Overclocker upgrade power multipiler")
	public static double overclockerPower = 0.75;

	@ConfigRegistry(config = "items", category = "upgrades", key = "energy_storage", comment = "Energy storage upgrade extra power")
	public static double energyStoragePower = 40_000;

	public static final String[] types = new String[] { "overclock", "transformer", "energy_storage"};

	public ItemUpgrades() {
		func_77655_b("techreborn.upgrade");
		func_77627_a(true);
		func_77625_d(16);
	}

	public static ItemStack getUpgradeByName(String name, int count) {
		for (int i = 0; i < types.length; i++) {
			if (types[i].equalsIgnoreCase(name)) {
				return new ItemStack(ModItems.UPGRADES, count, i);
			}
		}
		throw new InvalidParameterException("The upgrade " + name + " could not be found.");
	}

	public static ItemStack getUpgradeByName(String name) {
		return getUpgradeByName(name, 1);
	}

	// Item
	@Override
	public String func_77667_c(ItemStack itemStack) {
		int meta = itemStack.func_77952_i();
		if (meta < 0 || meta >= types.length) {
			meta = 0;
		}

		return super.func_77658_a() + "." + types[meta];
	}

	@Override
	public void func_150895_a(CreativeTabs creativeTabs, NonNullList<ItemStack> list) {
		if (!func_194125_a(creativeTabs)) {
			return;
		}
		for (int meta = 0; meta < types.length; ++meta) {
			list.add(new ItemStack(this, 1, meta));
		}
	}

	// IUpgrade
	@Override
	public void process(
		@Nonnull
			TileLegacyMachineBase machineBase,
		@Nullable
			IUpgradeHandler handler,
		@Nonnull
			ItemStack stack) {

		if (stack.func_77952_i() == 0) {
			handler.addSpeedMulti(overclockerSpeed);
			handler.addPowerMulti(overclockerPower);
			if(machineBase instanceof TilePowerAcceptor){
				TilePowerAcceptor powerAcceptor = (TilePowerAcceptor) machineBase;
				powerAcceptor.extraPowerInput += powerAcceptor.getMaxInput();
				powerAcceptor.extraPowerStoage += powerAcceptor.getBaseMaxPower();
			}
		}
		if (machineBase instanceof TilePowerAcceptor) {
			if (stack.func_77952_i() == 2) {
				TilePowerAcceptor acceptor = (TilePowerAcceptor) machineBase;
				acceptor.extraPowerStoage += energyStoragePower;
			}
			if (stack.func_77952_i() == 1) {
				TilePowerAcceptor acceptor = (TilePowerAcceptor) machineBase;
				acceptor.extraTeir += 1;
			}
		}
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void handleRightClick(TileEntity tile, ItemStack stack, Container container, int slotID) {

	}
}
