/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2018 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.tiles;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.NetworkManager;
import net.minecraft.network.play.server.SPacketUpdateTileEntity;
import reborncore.api.IListInfoProvider;
import reborncore.api.IToolDrop;
import reborncore.api.tile.IInventoryProvider;
import reborncore.common.registration.RebornRegistry;
import reborncore.common.registration.impl.ConfigRegistry;
import reborncore.common.tile.TileLegacyMachineBase;
import reborncore.common.util.FluidUtils;
import reborncore.common.util.Inventory;
import reborncore.common.util.Tank;
import reborncore.client.containerBuilder.IContainerProvider;
import reborncore.client.containerBuilder.builder.BuiltContainer;
import reborncore.client.containerBuilder.builder.ContainerBuilder;
import techreborn.init.ModBlocks;
import techreborn.lib.ModInfo;

import javax.annotation.Nullable;
import java.util.List;

@RebornRegistry(modID = ModInfo.MOD_ID)
public class TileQuantumTank extends TileLegacyMachineBase
	implements IInventoryProvider, IToolDrop, IListInfoProvider, IContainerProvider {

	@ConfigRegistry(config = "machines", category = "quantum_tank", key = "QuantumTankMaxStorage", comment = "Maximum amount of millibuckets a Quantum Tank can store")
	public static int maxStorage = Integer.MAX_VALUE;

	public Tank tank = new Tank("TileQuantumTank", maxStorage, this);
	public Inventory inventory = new Inventory(3, "TileQuantumTank", 64, this);

	public void readFromNBTWithoutCoords(final NBTTagCompound tagCompound) {
		tank.readFromNBT(tagCompound);
	}
	
	public NBTTagCompound writeToNBTWithoutCoords(final NBTTagCompound tagCompound) {
		tank.writeToNBT(tagCompound);
		return tagCompound;
	}
	
	public ItemStack getDropWithNBT() {
		final NBTTagCompound tileEntity = new NBTTagCompound();
		final ItemStack dropStack = new ItemStack(ModBlocks.QUANTUM_TANK, 1);
		this.writeToNBTWithoutCoords(tileEntity);
		dropStack.func_77982_d(new NBTTagCompound());
		dropStack.func_77978_p().func_74782_a("tileEntity", tileEntity);
		return dropStack;
	}
	
	// TileLegacyMachineBase
	@Override
	public void func_73660_a() {
		super.func_73660_a();
		if (!field_145850_b.field_72995_K) {
			if (FluidUtils.drainContainers(tank, inventory, 0, 1)
				|| FluidUtils.fillContainers(tank, inventory, 0, 1, tank.getFluidType()))
				this.syncWithAll();
		}
		tank.compareAndUpdate();
	}
	
	@Override
	public boolean canBeUpgraded() {
		return false;
	}
	
	@Override
	public void func_145839_a(final NBTTagCompound tagCompound) {
		super.func_145839_a(tagCompound);
		readFromNBTWithoutCoords(tagCompound);
	}

	@Override
	public NBTTagCompound func_189515_b(final NBTTagCompound tagCompound) {
		super.func_189515_b(tagCompound);
		writeToNBTWithoutCoords(tagCompound);
		return tagCompound;
	}

	@Override
	public void onDataPacket(final NetworkManager net, final SPacketUpdateTileEntity packet) {
		field_145850_b.func_147458_c(field_174879_c.func_177958_n(), field_174879_c.func_177956_o(), field_174879_c.func_177952_p(), field_174879_c.func_177958_n(), field_174879_c.func_177956_o(), field_174879_c.func_177952_p());
		func_145839_a(packet.func_148857_g());
	}

	// IInventoryProvider
	@Override
	public Inventory getInventory() {
		return this.inventory;
	}
	
	// IToolDrop
	@Override
	public ItemStack getToolDrop(final EntityPlayer entityPlayer) {
		return this.getDropWithNBT();
	}
	
	// IListInfoProvider
	@Override
	public void addInfo(final List<String> info, final boolean isRealTile) {
		if (isRealTile) {
			if (this.tank.getFluid() != null) {
				info.add(this.tank.getFluidAmount() + " of " + this.tank.getFluidType().getName());
			} else {
				info.add("Empty");
			}
		}
		info.add("Capacity " + this.tank.getCapacity() + " mb");
	}

	// IContainerProvider
	@Override
	public BuiltContainer createContainer(final EntityPlayer player) {
		return new ContainerBuilder("quantumtank").player(player.field_71071_by).inventory().hotbar()
			.addInventory().tile(this).fluidSlot(0, 80, 17).outputSlot(1, 80, 53).addInventory()
			.create(this);
	}

	@Nullable
	@Override
	public Tank getTank() {
		return tank;
	}
}
