/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2018 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.tiles;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.NetworkManager;
import net.minecraft.network.play.server.SPacketUpdateTileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraftforge.common.capabilities.Capability;
import net.minecraftforge.items.CapabilityItemHandler;
import net.minecraftforge.items.wrapper.InvWrapper;
import powercrystals.minefactoryreloaded.api.IDeepStorageUnit;
import reborncore.api.IListInfoProvider;
import reborncore.api.IToolDrop;
import reborncore.api.tile.IInventoryProvider;
import reborncore.common.tile.TileLegacyMachineBase;
import reborncore.common.util.Inventory;
import reborncore.common.util.ItemUtils;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nullable;

public class TileTechStorageBase extends TileLegacyMachineBase
		implements IInventoryProvider, IToolDrop, IListInfoProvider, IDeepStorageUnit {

	public final int maxCapacity;
	public final Inventory inventory;
	public InvWrapper invWrapper;
	public ItemStack storedItem;

	public TileTechStorageBase(String name, int maxCapacity) {
		this.maxCapacity = maxCapacity;
		storedItem = ItemStack.field_190927_a;
		inventory = new Inventory(3, name, maxCapacity, this);
	}

	public void readFromNBTWithoutCoords(NBTTagCompound tagCompound) {

		storedItem = ItemStack.field_190927_a;

		if (tagCompound.func_74764_b("storedStack")) {
			storedItem = new ItemStack((NBTTagCompound) tagCompound.func_74781_a("storedStack"));
		}

		if (!storedItem.func_190926_b()) {
			storedItem.func_190920_e(Math.min(tagCompound.func_74762_e("storedQuantity"), this.maxCapacity));
		}

		inventory.readFromNBT(tagCompound);
	}

	public NBTTagCompound writeToNBTWithoutCoords(NBTTagCompound tagCompound) {
		if (!storedItem.func_190926_b()) {
			ItemStack temp = storedItem.func_77946_l();
			if (storedItem.func_190916_E() > storedItem.func_77976_d()) {
				temp.func_190920_e(storedItem.func_77976_d());
			}
			tagCompound.func_74782_a("storedStack", temp.func_77955_b(new NBTTagCompound()));
			tagCompound.func_74768_a("storedQuantity", Math.min(storedItem.func_190916_E(), maxCapacity));
		} else {
			tagCompound.func_74768_a("storedQuantity", 0);
		}
		inventory.writeToNBT(tagCompound);
		return tagCompound;
	}

	public ItemStack getDropWithNBT() {
		NBTTagCompound tileEntity = new NBTTagCompound();
		ItemStack dropStack = new ItemStack(func_145838_q(), 1);
		writeToNBTWithoutCoords(tileEntity);
		dropStack.func_77982_d(new NBTTagCompound());
		dropStack.func_77978_p().func_74782_a("tileEntity", tileEntity);
		storedItem.func_190920_e(0);
		func_70299_a(1, ItemStack.field_190927_a);
		syncWithAll();

		return dropStack;
	}

	public InvWrapper getInvWrapper() {
		if (invWrapper == null)
			invWrapper = new InvWrapper(this);
		return invWrapper;
	}

	public int getStoredCount() {
		return storedItem.func_190916_E();
	}

	public List<ItemStack> getContentDrops() {
		ArrayList<ItemStack> stacks = new ArrayList<>();

		if (!getStoredItemType().func_190926_b()) {
			if (!func_70301_a(1).func_190926_b()) {
				stacks.add(func_70301_a(1));
			}
			int size = storedItem.func_77976_d();
			for (int i = 0; i < getStoredCount() / size; i++) {
				ItemStack droped = storedItem.func_77946_l();
				droped.func_190920_e(size);
				stacks.add(droped);
			}
			if (getStoredCount() % size != 0) {
				ItemStack droped = storedItem.func_77946_l();
				droped.func_190920_e(getStoredCount() % size);
				stacks.add(droped);
			}
		}

		return stacks;
	}

	// TileLegacyMachineBase
	@Override
	public void func_73660_a() {
		super.func_73660_a();
		if (!field_145850_b.field_72995_K) {
			ItemStack outputStack = ItemStack.field_190927_a;
			if (!func_70301_a(1).func_190926_b()) {
				outputStack = func_70301_a(1);
			}
			if (!func_70301_a(0).func_190926_b()
					&& (storedItem.func_190916_E() + outputStack.func_190916_E()) < maxCapacity) {
				ItemStack inputStack = func_70301_a(0);
				if (getStoredItemType().func_190926_b()
						|| (storedItem.func_190926_b() && ItemUtils.isItemEqual(inputStack, outputStack, true, true))) {

					storedItem = inputStack;
					func_70299_a(0, ItemStack.field_190927_a);
				} else if (ItemUtils.isItemEqual(getStoredItemType(), inputStack, true, true)) {
					int reminder = maxCapacity - storedItem.func_190916_E() - outputStack.func_190916_E();
					if (inputStack.func_190916_E() <= reminder) {
						setStoredItemCount(inputStack.func_190916_E());
						func_70299_a(0, ItemStack.field_190927_a);
					} else {
						setStoredItemCount(maxCapacity - outputStack.func_190916_E());
						func_70301_a(0).func_190918_g(reminder);
					}
				}
				func_70296_d();
				syncWithAll();
			}

			if (!storedItem.func_190926_b()) {
				if (outputStack.func_190926_b()) {

					ItemStack delivered = storedItem.func_77946_l();
					delivered.func_190920_e(Math.min(storedItem.func_190916_E(), delivered.func_77976_d()));
					storedItem.func_190918_g(delivered.func_190916_E());

					if (storedItem.func_190926_b()) {
						storedItem = ItemStack.field_190927_a;
					}

					func_70299_a(1, delivered);
					func_70296_d();
					syncWithAll();
				} else if (ItemUtils.isItemEqual(storedItem, outputStack, true, true)
						&& outputStack.func_190916_E() < outputStack.func_77976_d()) {

					int wanted = Math.min(storedItem.func_190916_E(),
							outputStack.func_77976_d() - outputStack.func_190916_E());
					outputStack.func_190920_e(outputStack.func_190916_E() + wanted);
					storedItem.func_190918_g(wanted);

					if (storedItem.func_190926_b()) {
						storedItem = ItemStack.field_190927_a;
					}
					func_70296_d();
					syncWithAll();
				}
			}
		}
	}

	@Override
	public boolean canBeUpgraded() {
		return false;
	}

	@Override
	public void onDataPacket(NetworkManager net, SPacketUpdateTileEntity packet) {
		field_145850_b.func_147458_c(field_174879_c.func_177958_n(), field_174879_c.func_177956_o(), field_174879_c.func_177952_p(), field_174879_c.func_177958_n(), field_174879_c.func_177956_o(), field_174879_c.func_177952_p());
		func_145839_a(packet.func_148857_g());
	}

	@Override
	public void func_145839_a(NBTTagCompound tagCompound) {
		super.func_145839_a(tagCompound);
		readFromNBTWithoutCoords(tagCompound);
	}

	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound tagCompound) {
		super.func_189515_b(tagCompound);
		writeToNBTWithoutCoords(tagCompound);
		return tagCompound;
	}

	@Override
	public <T> T getCapability(Capability<T> capability, EnumFacing facing) {
		if (capability == CapabilityItemHandler.ITEM_HANDLER_CAPABILITY) {
			return CapabilityItemHandler.ITEM_HANDLER_CAPABILITY.cast(getInvWrapper());
		}
		return super.getCapability(capability, facing);
	}

	@Override
	public boolean hasCapability(Capability<?> capability, @Nullable EnumFacing facing) {
		return capability == CapabilityItemHandler.ITEM_HANDLER_CAPABILITY || super.hasCapability(capability, facing);
	}

	// IInventoryProvider
	@Override
	public Inventory getInventory() {
		return inventory;
	}

	// IToolDrop
	@Override
	public ItemStack getToolDrop(EntityPlayer entityPlayer) {
		return getDropWithNBT();
	}

	// IListInfoProvider
	@Override
	public void addInfo(List<String> info, boolean isRealTile) {
		if (isRealTile) {
			int size = 0;
			String name = "of nothing";
			if (!storedItem.func_190926_b()) {
				name = storedItem.func_82833_r();
				size += storedItem.func_190916_E();
			}
			if (!func_70301_a(1).func_190926_b()) {
				name = func_70301_a(1).func_82833_r();
				size += func_70301_a(1).func_190916_E();
			}
			info.add(size + " " + name);
		}
	}

	// IDeepStorageUnit
	@Override
	public ItemStack getStoredItemType() {
		return storedItem.func_190926_b() ? func_70301_a(1) : storedItem;
	}

	@Override
	public void setStoredItemCount(int amount) {
		storedItem.func_190917_f(amount);
		func_70296_d();
	}

	@Override
	public void setStoredItemType(ItemStack type, int amount) {
		storedItem = type;
		storedItem.func_190920_e(amount);
		func_70296_d();
	}

	@Override
	public int getMaxStoredCount() {
		return maxCapacity;
	}
}
