/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2018 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.tiles.multiblock;

import net.minecraft.block.material.Material;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.network.NetworkManager;
import net.minecraft.network.play.server.SPacketUpdateTileEntity;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.math.BlockPos;
import reborncore.common.multiblock.IMultiblockPart;
import reborncore.common.recipes.RecipeCrafter;
import reborncore.common.registration.RebornRegistry;
import reborncore.common.registration.impl.ConfigRegistry;
import reborncore.common.util.Inventory;
import techreborn.api.Reference;
import techreborn.api.recipe.ITileRecipeHandler;
import techreborn.api.recipe.machines.BlastFurnaceRecipe;
import techreborn.blocks.BlockMachineCasing;
import reborncore.client.containerBuilder.IContainerProvider;
import reborncore.client.containerBuilder.builder.BuiltContainer;
import reborncore.client.containerBuilder.builder.ContainerBuilder;
import techreborn.init.ModBlocks;
import techreborn.lib.ModInfo;
import techreborn.multiblocks.MultiBlockCasing;
import techreborn.tiles.TileGenericMachine;
import techreborn.tiles.TileMachineCasing;

@RebornRegistry(modID = ModInfo.MOD_ID)
public class TileIndustrialBlastFurnace extends TileGenericMachine implements IContainerProvider, ITileRecipeHandler<BlastFurnaceRecipe>  {
	
	@ConfigRegistry(config = "machines", category = "industrial_furnace", key = "IndustrialFurnaceMaxInput", comment = "Industrial Blast Furnace Max Input (Value in EU)")
	public static int maxInput = 128;
	@ConfigRegistry(config = "machines", category = "industrial_furnace", key = "IndustrialFurnaceMaxEnergy", comment = "Industrial Blast Furnace Max Energy (Value in EU)")
	public static int maxEnergy = 40_000;

	public MultiblockChecker multiblockChecker;
	private int cachedHeat;

	public TileIndustrialBlastFurnace() {
		super("IndustrialBlastFurnace", maxInput, maxEnergy, ModBlocks.INDUSTRIAL_BLAST_FURNACE, 4);
		final int[] inputs = new int[] { 0, 1 };
		final int[] outputs = new int[] { 2, 3 };
		this.inventory = new Inventory(5, "TileIndustrialBlastFurnace", 64, this);
		this.crafter = new RecipeCrafter(Reference.BLAST_FURNACE_RECIPE, this, 2, 2, this.inventory, inputs, outputs);
	}
	
	public int getHeat() {
		if (!getMutliBlock()){
			return 0;
		}
		
		// Bottom center of multiblock
		final BlockPos location = field_174879_c.func_177967_a(getFacing().func_176734_d(), 2);
		final TileEntity tileEntity = field_145850_b.func_175625_s(location);

		if (tileEntity instanceof TileMachineCasing) {
			if (((TileMachineCasing) tileEntity).isConnected()
					&& ((TileMachineCasing) tileEntity).getMultiblockController().isAssembled()) {
				final MultiBlockCasing casing = ((TileMachineCasing) tileEntity).getMultiblockController();

				int heat = 0;

				// Bottom center shouldn't have any tile entities below it
				if (field_145850_b.func_180495_p(new BlockPos(location.func_177958_n(), location.func_177956_o() - 1, location.func_177952_p()))
						.func_177230_c() == tileEntity.func_145838_q()) {
					return 0;
				}

				for (final IMultiblockPart part : casing.connectedParts) {
					final BlockMachineCasing casing1 = (BlockMachineCasing) field_145850_b.func_180495_p(part.func_174877_v()).func_177230_c();
					heat += casing1.getHeatFromState(field_145850_b.func_180495_p(part.func_174877_v()));
				}

				if (field_145850_b.func_180495_p(location.func_177967_a(EnumFacing.UP, 1)).func_177230_c().func_149739_a().equals("tile.lava")
						&& field_145850_b.func_180495_p(location.func_177967_a(EnumFacing.UP, 2)).func_177230_c().func_149739_a().equals("tile.lava")) {
					heat += 500;
				}
				return heat;
			}
		}

		return 0;
	}
	
	public boolean getMutliBlock() {
		final boolean layer0 = multiblockChecker.checkRectY(1, 1, MultiblockChecker.CASING_ANY, MultiblockChecker.ZERO_OFFSET);
		final boolean layer1 = multiblockChecker.checkRingY(1, 1, MultiblockChecker.CASING_ANY, new BlockPos(0, 1, 0));
		final boolean layer2 = multiblockChecker.checkRingY(1, 1, MultiblockChecker.CASING_ANY, new BlockPos(0, 2, 0));
		final boolean layer3 = multiblockChecker.checkRectY(1, 1, MultiblockChecker.CASING_ANY, new BlockPos(0, 3, 0));
		final Material centerBlock1 = multiblockChecker.getBlock(0, 1, 0).func_185904_a();
		final Material centerBlock2 = multiblockChecker.getBlock(0, 2, 0).func_185904_a();
		final boolean center1 = (centerBlock1 == Material.field_151579_a || centerBlock1 == Material.field_151587_i);
		final boolean center2 = (centerBlock2 == Material.field_151579_a || centerBlock2 == Material.field_151587_i);
		return layer0 && layer1 && layer2 && layer3 && center1 && center2;
	}
	
	public void setHeat(final int heat) {
		cachedHeat = heat;
	}

	public int getCachedHeat() {
		return cachedHeat;
	}
	
	// TileGenericMachine
	@Override
	public void func_73660_a() {
		if (multiblockChecker == null) {
			final BlockPos downCenter = field_174879_c.func_177967_a(getFacing().func_176734_d(), 2);
			multiblockChecker = new MultiblockChecker(field_145850_b, downCenter);
		}
		
		if (!field_145850_b.field_72995_K && getMutliBlock()){ 
			super.func_73660_a();
		}		
	}

	// TileLegacyMachineBase
	@Override
	public void onDataPacket(final NetworkManager net, final SPacketUpdateTileEntity packet) {
		field_145850_b.func_147458_c(field_174879_c.func_177958_n(), field_174879_c.func_177956_o(), field_174879_c.func_177952_p(), field_174879_c.func_177958_n(), field_174879_c.func_177956_o(), field_174879_c.func_177952_p());
		func_145839_a(packet.func_148857_g());
	}
	
	// IContainerProvider
	@Override
	public BuiltContainer createContainer(final EntityPlayer player) {
		return new ContainerBuilder("blastfurnace").player(player.field_71071_by).inventory().hotbar().addInventory()
				.tile(this).slot(0, 50, 27).slot(1, 50, 47).outputSlot(2, 93, 37).outputSlot(3, 113, 37)
				.energySlot(4, 8, 72).syncEnergyValue().syncCrafterValue()
				.syncIntegerValue(this::getHeat, this::setHeat).addInventory().create(this);
	}
	
	// ITileRecipeHandler
	@Override
	public boolean canCraft(final TileEntity tile, final BlastFurnaceRecipe recipe) {
		if (tile instanceof TileIndustrialBlastFurnace) {
			final TileIndustrialBlastFurnace blastFurnace = (TileIndustrialBlastFurnace) tile;
			return blastFurnace.getHeat() >= recipe.neededHeat;
		}
		return false;
	}

	@Override
	public boolean onCraft(final TileEntity tile, final BlastFurnaceRecipe recipe) {
		return true;
	}
}
