/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2018 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.tiles.tier1;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.Container;
import net.minecraft.inventory.IInventory;
import net.minecraft.inventory.InventoryCrafting;
import net.minecraft.item.ItemStack;
import net.minecraft.item.crafting.CraftingManager;
import net.minecraft.item.crafting.IRecipe;
import net.minecraft.item.crafting.Ingredient;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.SoundCategory;
import org.apache.commons.lang3.tuple.Pair;
import reborncore.api.IToolDrop;
import reborncore.api.tile.IInventoryProvider;
import reborncore.common.powerSystem.TilePowerAcceptor;
import reborncore.common.registration.RebornRegistry;
import reborncore.common.registration.impl.ConfigRegistry;
import reborncore.common.util.Inventory;
import reborncore.common.util.ItemUtils;
import reborncore.client.containerBuilder.IContainerProvider;
import reborncore.client.containerBuilder.builder.BuiltContainer;
import reborncore.client.containerBuilder.builder.ContainerBuilder;
import techreborn.init.ModBlocks;
import techreborn.init.ModSounds;
import techreborn.lib.ModInfo;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.List;

/**
 * Created by modmuss50 on 20/06/2017.
 */
@RebornRegistry(modID = ModInfo.MOD_ID)
public class TileAutoCraftingTable extends TilePowerAcceptor
		implements IToolDrop, IInventoryProvider, IContainerProvider {

	@ConfigRegistry(config = "machines", category = "autocrafter", key = "AutoCrafterInput", comment = "AutoCrafting Table Max Input (Value in EU)")
	public static int maxInput = 32;
	@ConfigRegistry(config = "machines", category = "autocrafter", key = "AutoCrafterMaxEnergy", comment = "AutoCrafting Table Max Energy (Value in EU)")
	public static int maxEnergy = 10_000;

	public Inventory inventory = new Inventory(11, "TileAutoCraftingTable", 64, this);
	public int progress;
	public int maxProgress = 120;
	public int euTick = 10;

	InventoryCrafting inventoryCrafting = null;
	IRecipe lastCustomRecipe = null;
	IRecipe lastRecipe = null;

	public boolean locked = true;

	public TileAutoCraftingTable() {
		super();
	}

	@Nullable
	public IRecipe getIRecipe() {
		InventoryCrafting crafting = getCraftingInventory();
		if (!crafting.func_191420_l()) {
			if (lastRecipe != null) {
				if (lastRecipe.func_77569_a(crafting, field_145850_b)) {
					return lastRecipe;
				}
			}
			for (IRecipe testRecipe : CraftingManager.field_193380_a) {
				if (testRecipe.func_77569_a(crafting, field_145850_b)) {
					lastRecipe = testRecipe;
					return testRecipe;
				}
			}
		}
		return null;
	}

	public InventoryCrafting getCraftingInventory() {
		if (inventoryCrafting == null) {
			inventoryCrafting = new InventoryCrafting(new Container() {
				@Override
				public boolean func_75145_c(EntityPlayer playerIn) {
					return false;
				}
			}, 3, 3);
		}
		for (int i = 0; i < 9; i++) {
			inventoryCrafting.func_70299_a(i, inventory.func_70301_a(i));
		}
		return inventoryCrafting;
	}

	public boolean canMake(IRecipe recipe) {
		if (recipe != null && recipe.func_194133_a(3, 3)) {
			boolean missingOutput = false;
			int[] stacksInSlots = new int[9];
			for (int i = 0; i < 9; i++) {
				stacksInSlots[i] = inventory.func_70301_a(i).func_190916_E();
			}
			for (Ingredient ingredient : recipe.func_192400_c()) {
				if (ingredient != Ingredient.field_193370_a) {
					boolean foundIngredient = false;
					for (int i = 0; i < 9; i++) {
						ItemStack stack = inventory.func_70301_a(i);
						int requiredSize = locked ? 1 : 0;
						if (stack.func_77976_d() == 1) {
							requiredSize = 0;
						}
						if (stacksInSlots[i] > requiredSize) {
							if (ingredient.apply(stack)) {
								if (stack.func_77973_b().func_77668_q() != null) {
									if (!hasRoomForExtraItem(stack.func_77973_b().getContainerItem(stack))) {
										continue;
									}
								}
								foundIngredient = true;
								stacksInSlots[i]--;
								break;
							}
						}
					}
					if (!foundIngredient) {
						missingOutput = true;
					}
				}
			}
			if (!missingOutput) {
				if (hasOutputSpace(recipe.func_77571_b(), 9)) {
					return true;
				}
			}
			return false;
		}
		return false;
	}

	boolean hasRoomForExtraItem(ItemStack stack) {
		ItemStack extraOutputSlot = func_70301_a(10);
		if (extraOutputSlot.func_190926_b()) {
			return true;
		}
		return hasOutputSpace(stack, 10);
	}

	public boolean hasOutputSpace(ItemStack output, int slot) {
		ItemStack stack = inventory.func_70301_a(slot);
		if (stack.func_190926_b()) {
			return true;
		}
		if (ItemUtils.isItemEqual(stack, output, true, true)) {
			if (stack.func_77976_d() > stack.func_190916_E() + output.func_190916_E()) {
				return true;
			}
		}
		return false;
	}

	public boolean make(IRecipe recipe) {
		if (recipe == null || !canMake(recipe)) {
			return false;
		}
		for (int i = 0; i < recipe.func_192400_c().size(); i++) {
			Ingredient ingredient = recipe.func_192400_c().get(i);
			// Looks for the best slot to take it from
			ItemStack bestSlot = inventory.func_70301_a(i);
			if (ingredient.apply(bestSlot)) {
				handleContainerItem(bestSlot);
				bestSlot.func_190918_g(1);
			} else {
				for (int j = 0; j < 9; j++) {
					ItemStack stack = inventory.func_70301_a(j);
					if (ingredient.apply(stack)) {
						handleContainerItem(stack);
						stack.func_190918_g(1); // TODO is this right? or do I need
											// to use it as an actull
											// crafting grid
						break;
					}
				}
			}
		}
		ItemStack output = inventory.func_70301_a(9);
		// TODO fire forge recipe event
		ItemStack ouputStack = recipe.func_77572_b(getCraftingInventory());
		if (output.func_190926_b()) {
			inventory.func_70299_a(9, ouputStack.func_77946_l());
		} else {
			// TODO use ouputStack in someway?
			output.func_190917_f(recipe.func_77571_b().func_190916_E());
		}
		return true;
	}

	private void handleContainerItem(ItemStack stack) {
		if (stack.func_77973_b().hasContainerItem(stack)) {
			ItemStack containerItem = stack.func_77973_b().getContainerItem(stack);
			ItemStack extraOutputSlot = func_70301_a(10);
			if (hasOutputSpace(containerItem, 10)) {
				if (extraOutputSlot.func_190926_b()) {
					func_70299_a(10, containerItem.func_77946_l());
				} else if (ItemUtils.isItemEqual(extraOutputSlot, containerItem, true, true)
						&& extraOutputSlot.func_77976_d() < extraOutputSlot.func_190916_E() + containerItem.func_190916_E()) {
					extraOutputSlot.func_190917_f(1);
				}
			}
		}
	}

	public boolean hasIngredient(Ingredient ingredient) {
		for (int i = 0; i < 9; i++) {
			ItemStack stack = inventory.func_70301_a(i);
			if (ingredient.apply(stack)) {
				return true;
			}
		}
		return false;
	}

	public boolean isItemValidForRecipeSlot(IRecipe recipe, ItemStack stack, int slotID) {
		if (recipe == null) {
			return true;
		}
		int bestSlot = findBestSlotForStack(recipe, stack);
		if (bestSlot != -1) {
			return bestSlot == slotID;
		}
		return true;
	}

	public int findBestSlotForStack(IRecipe recipe, ItemStack stack) {
		if (recipe == null) {
			return -1;
		}
		List<Integer> possibleSlots = new ArrayList<>();
		for (int i = 0; i < recipe.func_192400_c().size(); i++) {
			ItemStack stackInSlot = inventory.func_70301_a(i);
			Ingredient ingredient = recipe.func_192400_c().get(i);
			if (ingredient != Ingredient.field_193370_a && ingredient.apply(stack)) {
				if (stackInSlot.func_190926_b()) {
					possibleSlots.add(i);
				} else if (stackInSlot.func_77973_b() == stack.func_77973_b()
						&& stackInSlot.func_77952_i() == stack.func_77952_i()) {
					if (stackInSlot.func_77976_d() >= stackInSlot.func_190916_E() + stack.func_190916_E()) {
						possibleSlots.add(i);
					}
				}
			}
		}
		// Slot, count
		Pair<Integer, Integer> smallestCount = null;
		for (Integer slot : possibleSlots) {
			ItemStack slotStack = inventory.func_70301_a(slot);
			if (slotStack.func_190926_b()) {
				return slot;
			}
			if (smallestCount == null) {
				smallestCount = Pair.of(slot, slotStack.func_190916_E());
			} else if (smallestCount.getRight() >= slotStack.func_190916_E()) {
				smallestCount = Pair.of(slot, slotStack.func_190916_E());
			}
		}
		if (smallestCount != null) {
			return smallestCount.getLeft();
		}
		return -1;
	}

	public int getProgress() {
		return progress;
	}

	public void setProgress(int progress) {
		this.progress = progress;
	}

	public int getMaxProgress() {
		if (maxProgress == 0) {
			maxProgress = 1;
		}
		return maxProgress;
	}

	public void setMaxProgress(int maxProgress) {
		this.maxProgress = maxProgress;
	}

	// TilePowerAcceptor
	@Override
	public void func_73660_a() {
		super.func_73660_a();
		if (field_145850_b.field_72995_K) {
			return;
		}
		IRecipe recipe = getIRecipe();
		if (recipe != null) {
			if (progress >= maxProgress) {
				if (make(recipe)) {
					progress = 0;
				}
			} else {
				if (canMake(recipe)) {
					if (canUseEnergy(euTick)) {
						progress++;
						if (progress == 1) {
							field_145850_b.func_184148_a(null, field_174879_c.func_177958_n(), field_174879_c.func_177956_o(), field_174879_c.func_177952_p(), ModSounds.AUTO_CRAFTING,
									SoundCategory.BLOCKS, 0.3F, 0.8F);
						}
						useEnergy(euTick);
					}
				} else {
					progress = 0;
				}
			}
		}
		if (recipe == null) {
			progress = 0;
		}
	}

	// Easyest way to sync back to the client
	public int getLockedInt() {
		return locked ? 1 : 0;
	}

	public void setLockedInt(int lockedInt) {
		locked = lockedInt == 1;
	}

	@Override
	public double getBaseMaxPower() {
		return maxEnergy;
	}

	@Override
	public double getBaseMaxOutput() {
		return 0;
	}

	@Override
	public double getBaseMaxInput() {
		return maxInput;
	}

	@Override
	public boolean canAcceptEnergy(EnumFacing enumFacing) {
		return true;
	}

	@Override
	public boolean canProvideEnergy(EnumFacing enumFacing) {
		return false;
	}

	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound tag) {
		tag.func_74757_a("locked", locked);
		return super.func_189515_b(tag);
	}

	@Override
	public void func_145839_a(NBTTagCompound tag) {
		if (tag.func_74764_b("locked")) {
			locked = tag.func_74767_n("locked");
		}
		super.func_145839_a(tag);
	}

	// TileLegacyMachineBase
	@Override
	public boolean canBeUpgraded() {
		return false;
	}

	@Override
	public boolean func_94041_b(int index, ItemStack stack) {
		int bestSlot = findBestSlotForStack(getIRecipe(), stack);
		if (bestSlot != -1) {
			return index == bestSlot;
		}
		return super.func_94041_b(index, stack);
	}

	@Override
	public int[] func_180463_a(EnumFacing side) {
		return new int[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 };
	}

	@Override
	public boolean func_180462_a(int index, ItemStack stack, EnumFacing direction) {
		if (index > 8) {
			return false;
		}
		int bestSlot = findBestSlotForStack(getIRecipe(), stack);
		if (bestSlot != -1) {
			return index == bestSlot;
		}
		return true;
	}

	@Override
	public boolean func_180461_b(int index, ItemStack stack, EnumFacing direction) {
		if (index > 8) {
			return true;
		}
		return false;
	}

	// This machine doesnt have a facing
	@Override
	public EnumFacing getFacingEnum() {
		return EnumFacing.NORTH;
	}

	// IToolDrop
	@Override
	public ItemStack getToolDrop(EntityPlayer playerIn) {
		return new ItemStack(ModBlocks.AUTO_CRAFTING_TABLE, 1);
	}

	// IInventoryProvider
	@Override
	public IInventory getInventory() {
		return inventory;
	}

	// IContainerProvider
	@Override
	public BuiltContainer createContainer(EntityPlayer player) {
		return new ContainerBuilder("autocraftingtable").player(player.field_71071_by).inventory().hotbar().addInventory()
				.tile(this).slot(0, 28, 25).slot(1, 46, 25).slot(2, 64, 25).slot(3, 28, 43).slot(4, 46, 43)
				.slot(5, 64, 43).slot(6, 28, 61).slot(7, 46, 61).slot(8, 64, 61).outputSlot(9, 145, 42)
				.outputSlot(10, 145, 70).syncEnergyValue().syncIntegerValue(this::getProgress, this::setProgress)
				.syncIntegerValue(this::getMaxProgress, this::setMaxProgress)
				.syncIntegerValue(this::getLockedInt, this::setLockedInt).addInventory().create(this);
	}

	@Override
	public boolean hasSlotConfig() {
		return false;
	}
}
