/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2018 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.tiles.tier1;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.Container;
import net.minecraft.inventory.InventoryCrafting;
import net.minecraft.item.ItemStack;
import net.minecraft.item.crafting.IRecipe;
import net.minecraft.item.crafting.Ingredient;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.EnumFacing;
import org.apache.commons.lang3.tuple.Pair;
import reborncore.api.IToolDrop;
import reborncore.api.tile.IInventoryProvider;
import reborncore.common.powerSystem.TilePowerAcceptor;
import reborncore.common.registration.RebornRegistry;
import reborncore.common.registration.impl.ConfigRegistry;
import reborncore.common.util.Inventory;
import reborncore.common.util.ItemUtils;
import techreborn.api.RollingMachineRecipe;
import reborncore.client.containerBuilder.IContainerProvider;
import reborncore.client.containerBuilder.builder.BuiltContainer;
import reborncore.client.containerBuilder.builder.ContainerBuilder;
import techreborn.init.ModBlocks;
import techreborn.lib.ModInfo;

import javax.annotation.Nonnull;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

//TODO add tick and power bars.

@RebornRegistry(modID = ModInfo.MOD_ID)
public class TileRollingMachine extends TilePowerAcceptor
	implements IToolDrop, IInventoryProvider, IContainerProvider {

	@ConfigRegistry(config = "machines", category = "rolling_machine", key = "RollingMachineMaxInput", comment = "Rolling Machine Max Input (Value in EU)")
	public static int maxInput = 32;
	@ConfigRegistry(config = "machines", category = "rolling_machine", key = "RollingMachineEnergyPerTick", comment = "Rolling Machine Energy Per Tick (Value in EU)")
	public static int energyPerTick = 5;
	@ConfigRegistry(config = "machines", category = "rolling_machine", key = "RollingMachineEnergyRunTime", comment = "Rolling Machine Run Time")
	public static int runTime = 250;
	@ConfigRegistry(config = "machines", category = "rolling_machine", key = "RollingMachineMaxEnergy", comment = "Rolling Machine Max Energy (Value in EU)")
	public static int maxEnergy = 10000;

	public int[] craftingSlots = new int[] { 0, 1, 2, 3, 4, 5, 6, 7, 8 };
	private InventoryCrafting craftCache;
	public Inventory inventory = new Inventory(12, "TileRollingMachine", 64, this);
	public boolean isRunning;
	public int tickTime;
	@Nonnull
	public ItemStack currentRecipeOutput;
	public IRecipe currentRecipe;
	private int outputSlot;
	public boolean locked = false;
	public int balanceSlot = 0;

	public TileRollingMachine() {
		super();
		outputSlot = 9;
	}

	@Override
	public double getBaseMaxPower() {
		return maxEnergy;
	}

	@Override
	public boolean canAcceptEnergy(final EnumFacing direction) {
		return true;
	}

	@Override
	public boolean canProvideEnergy(final EnumFacing direction) {
		return false;
	}

	@Override
	public double getBaseMaxOutput() {
		return 0;
	}

	@Override
	public double getBaseMaxInput() {
		return maxInput;
	}

	@Override
	public void func_73660_a() {
		super.func_73660_a();
		if (field_145850_b.field_72995_K) {
			return;
		}
		charge(10);

		InventoryCrafting craftMatrix = getCraftingMatrix();
		currentRecipe = RollingMachineRecipe.instance.findMatchingRecipe(craftMatrix, field_145850_b);
		if (currentRecipe != null) {
			if (field_145850_b.func_82737_E() % 2 == 0) {
				Optional<InventoryCrafting> balanceResult = balanceRecipe(craftMatrix);
				if (balanceResult.isPresent()) {
					craftMatrix = balanceResult.get();
				}
			}
			currentRecipeOutput = currentRecipe.func_77572_b(craftMatrix);
		} else {
			currentRecipeOutput = ItemStack.field_190927_a;
		}

		if (!currentRecipeOutput.func_190926_b() && canMake(craftMatrix)) {
			if (tickTime >= Math.max((int) (runTime * (1.0 - getSpeedMultiplier())), 1)) {
				currentRecipeOutput = RollingMachineRecipe.instance.findMatchingRecipeOutput(craftMatrix, field_145850_b);
				if (!currentRecipeOutput.func_190926_b()) {
					boolean hasCrafted = false;
					if (inventory.func_70301_a(outputSlot).func_190926_b()) {
						inventory.func_70299_a(outputSlot, currentRecipeOutput);
						tickTime = 0;
						hasCrafted = true;
					} else {
						if (inventory.func_70301_a(outputSlot).func_190916_E()
								+ currentRecipeOutput.func_190916_E() <= currentRecipeOutput.func_77976_d()) {
							final ItemStack stack = inventory.func_70301_a(outputSlot);
							stack.func_190920_e(stack.func_190916_E() + currentRecipeOutput.func_190916_E());
							inventory.func_70299_a(outputSlot, stack);
							tickTime = 0;
							hasCrafted = true;
						}
					}
					if (hasCrafted) {
						for (int i = 0; i < craftMatrix.func_70302_i_(); i++) {
							inventory.func_70298_a(i, 1);
						}
						currentRecipeOutput = ItemStack.field_190927_a;
						currentRecipe = null;
					}
				}
			}
		} else {
			tickTime = 0;
		}
		if (!currentRecipeOutput.func_190926_b()) {
			if (canUseEnergy(getEuPerTick(energyPerTick))
					&& tickTime < Math.max((int) (runTime * (1.0 - getSpeedMultiplier())), 1)
					&& canMake(craftMatrix)) {
				useEnergy(getEuPerTick(energyPerTick));
				tickTime++;
			}
		}
		if (currentRecipeOutput.func_190926_b()) {
			tickTime = 0;
			currentRecipe = null;
		}

	}

	public Optional<InventoryCrafting> balanceRecipe(InventoryCrafting craftCache) {
		if (currentRecipe == null) {
			return Optional.empty();
		}
		if (field_145850_b.field_72995_K) {
			return Optional.empty();
		}
		if (!locked) {
			return Optional.empty();
		}
		if (craftCache.func_191420_l()) {
			return Optional.empty();
		}
		balanceSlot++;
		if (balanceSlot > craftCache.func_70302_i_()) {
			balanceSlot = 0;
		}
		//Find the best slot for each item in a recipe, and move it if needed
		ItemStack sourceStack = inventory.func_70301_a(balanceSlot);
		if (sourceStack.func_190926_b()) {
			return Optional.empty();
		}
		List<Integer> possibleSlots = new ArrayList<>();
		for (int s = 0; s < currentRecipe.func_192400_c().size(); s++) {
			ItemStack stackInSlot = inventory.func_70301_a(s);
			Ingredient ingredient = currentRecipe.func_192400_c().get(s);
			if (ingredient != Ingredient.field_193370_a && ingredient.apply(sourceStack)) {
				if (stackInSlot.func_190926_b()) {
					possibleSlots.add(s);
				} else if (stackInSlot.func_77973_b() == sourceStack.func_77973_b() && stackInSlot.func_77952_i() == sourceStack.func_77952_i()) {
					possibleSlots.add(s);
				}
			}
		}

		//Slot, count
		Pair<Integer, Integer> bestSlot = null;
		for (Integer slot : possibleSlots) {
			ItemStack slotStack = inventory.func_70301_a(slot);
			if (slotStack.func_190926_b()) {
				bestSlot = Pair.of(slot, 0);
			}
			if (bestSlot == null) {
				bestSlot = Pair.of(slot, slotStack.func_190916_E());
			} else if (bestSlot.getRight() >= slotStack.func_190916_E()) {
				bestSlot = Pair.of(slot, slotStack.func_190916_E());
			}
		}
		if (bestSlot == null
			|| bestSlot.getLeft() == balanceSlot
			|| bestSlot.getRight() == sourceStack.func_190916_E()
			|| inventory.func_70301_a(bestSlot.getLeft()).func_190926_b()
			|| !ItemUtils.isItemEqual(sourceStack, inventory.func_70301_a(bestSlot.getLeft()), true, true, true)) {
			return Optional.empty();
		}
		sourceStack.func_190918_g(1);
		inventory.func_70301_a(bestSlot.getLeft()).func_190917_f(1);
		inventory.hasChanged = true;

		return Optional.of(getCraftingMatrix());
	}

	private InventoryCrafting getCraftingMatrix() {
		if (craftCache == null) {
			craftCache = new InventoryCrafting(new RollingTileContainer(), 3, 3);
		}
		if (inventory.hasChanged) {
			for (int i = 0; i < 9; i++) {
				craftCache.func_70299_a(i, inventory.func_70301_a(i).func_77946_l());
			}
			inventory.hasChanged = false;
		}
		return craftCache;
	}

	public boolean canMake(InventoryCrafting craftMatrix) {
		ItemStack stack = RollingMachineRecipe.instance.findMatchingRecipeOutput(craftMatrix, this.field_145850_b);
		if (locked) {
			for (int i = 0; i < craftMatrix.func_70302_i_(); i++) {
				ItemStack stack1 = craftMatrix.func_70301_a(i);
				if (!stack1.func_190926_b() && stack1.func_190916_E() < 2) {
					return false;
				}
			}
		}
		if (stack.func_190926_b()) {
			return false;
		}
		ItemStack output = func_70301_a(outputSlot);
		if (output.func_190926_b()) {
			return true;
		}
		return ItemUtils.isItemEqual(stack, output, true, true);
	}

	@Override
	public ItemStack getToolDrop(final EntityPlayer entityPlayer) {
		return new ItemStack(ModBlocks.ROLLING_MACHINE, 1);
	}

	@Override
	public void func_145839_a(final NBTTagCompound tagCompound) {
		super.func_145839_a(tagCompound);
		this.isRunning = tagCompound.func_74767_n("isRunning");
		this.tickTime = tagCompound.func_74762_e("tickTime");
		this.locked = tagCompound.func_74767_n("locked");
	}

	@Override
	public NBTTagCompound func_189515_b(final NBTTagCompound tagCompound) {
		super.func_189515_b(tagCompound);
		tagCompound.func_74757_a("isRunning", this.isRunning);
		tagCompound.func_74768_a("tickTime", this.tickTime);
		tagCompound.func_74757_a("locked", locked);
		return tagCompound;
	}

	@Override
	public void func_145843_s() {
		super.func_145843_s();
	}

	@Override
	public void onChunkUnload() {
		super.onChunkUnload();
	}

	@Override
	public Inventory getInventory() {
		return inventory;
	}

	public int getBurnTime() {
		return tickTime;
	}

	public void setBurnTime(final int burnTime) {
		this.tickTime = burnTime;
	}

	public int getBurnTimeRemainingScaled(final int scale) {
		if (tickTime == 0 || Math.max((int) (runTime* (1.0 - getSpeedMultiplier())), 1) == 0) {
			return 0;
		}
		return tickTime * scale / Math.max((int) (runTime* (1.0 - getSpeedMultiplier())), 1);
	}

	@Override
	public BuiltContainer createContainer(final EntityPlayer player) {
		return new ContainerBuilder("rollingmachine").player(player.field_71071_by)
			.inventory().hotbar()
			.addInventory().tile(this)
			.slot(0, 30, 22).slot(1, 48, 22).slot(2, 66, 22)
			.slot(3, 30, 40).slot(4, 48, 40).slot(5, 66, 40)
			.slot(6, 30, 58).slot(7, 48, 58).slot(8, 66, 58)
			.onCraft(inv -> this.inventory.func_70299_a(1, RollingMachineRecipe.instance.findMatchingRecipeOutput(getCraftingMatrix(), this.field_145850_b)))
			.outputSlot(9, 124, 40)
			.energySlot(10, 8, 70)
			.syncEnergyValue().syncIntegerValue(this::getBurnTime, this::setBurnTime).syncIntegerValue(this::getLockedInt, this::setLockedInt).addInventory().create(this);
	}

	//Easyest way to sync back to the client
	public int getLockedInt() {
		return locked ? 1 : 0;
	}

	public void setLockedInt(int lockedInt) {
		locked = lockedInt == 1;
	}

	public int getProgressScaled(final int scale) {
		if (tickTime != 0 && Math.max((int) (runTime* (1.0 - getSpeedMultiplier())), 1) != 0) {
			return tickTime * scale / Math.max((int) (runTime* (1.0 - getSpeedMultiplier())), 1);
		}
		return 0;
	}

	private static class RollingTileContainer extends Container {

		@Override
		public boolean func_75145_c(final EntityPlayer entityplayer) {
			return true;
		}

	}

	@Override
	public boolean canBeUpgraded() {
		return true;
	}
}
