/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2018 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package techreborn.world;

import net.minecraft.block.Block;
import net.minecraft.block.state.IBlockState;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;
import net.minecraft.world.gen.feature.WorldGenerator;
import net.minecraftforge.common.IPlantable;
import techreborn.Core;
import techreborn.blocks.BlockRubberLeaves;
import techreborn.blocks.BlockRubberLog;
import techreborn.init.ModBlocks;

import java.util.Random;

public class RubberTreeGenerator extends WorldGenerator {

	boolean isWorldGen = true;

	public RubberTreeGenerator() {
		super();
	}

	public RubberTreeGenerator(boolean isWorldGen) {
		super(!isWorldGen);
		this.isWorldGen = isWorldGen;
	}

	@Override
	public boolean func_180709_b(World worldIn, Random rand, BlockPos position) {
		int retries = rand.nextInt(3) + 2;
		for (int c = 0; c < retries; c++) {
			int x = position.func_177958_n() + 8 + rand.nextInt(16);
			int z = position.func_177952_p() + 8 + rand.nextInt(16);
			int y = worldIn.func_189649_b(x, z) - 1;
			while (worldIn.func_175623_d(new BlockPos(x, y, z)) && y > 0) {
				y--;
			}
			if (!growTree(worldIn, rand, x, y + 1, z)) {
				retries--;
			}
		}

		return false;
	}

	public boolean growTree(World world, Random rand, int x, int y, int z) {
		int treeHeight = rand.nextInt(5) + Core.worldGen.config.rubberTreeConfig.treeBaseHeight;
		int worldHeight = world.func_72800_K();
		if (y >= 1 && y + treeHeight + 1 <= worldHeight) {
			int xOffset;
			int yOffset;
			int zOffset;
			IBlockState baseSate = world.func_180495_p(new BlockPos(x, y - 1, z));
			Block baseBlock = baseSate.func_177230_c();
			boolean hasPlacedBlock = false;
			if (baseBlock != null && baseBlock.canSustainPlant(baseSate, world, new BlockPos(x, y - 1, z),
				EnumFacing.UP, (IPlantable) ModBlocks.RUBBER_SAPLING) && y < worldHeight - treeHeight - 1) {
				for (yOffset = y; yOffset <= y + 1 + treeHeight; ++yOffset) {
					byte radius = 1;
					if (yOffset == y) {
						radius = 0;
					}
					if (yOffset >= y + 1 + treeHeight - 2) {
						radius = 2;
					}
					if (yOffset >= 0 & yOffset < worldHeight) {
						for (xOffset = x - radius; xOffset <= x + radius; ++xOffset) {
							for (zOffset = z - radius; zOffset <= z + radius; ++zOffset) {
								IBlockState state = world.func_180495_p(new BlockPos(xOffset, yOffset, zOffset));
								Block block = state.func_177230_c();

								if (block != null
									&& !(block.isLeaves(state, world, new BlockPos(xOffset, yOffset, zOffset))
									|| block.isAir(state, world, new BlockPos(xOffset, yOffset, zOffset))
									|| block.canBeReplacedByLeaves(state, world,
									new BlockPos(xOffset, yOffset, zOffset)))) {
									return false;
								}
							}
						}
					} else {
						return false;
					}
				}

				BlockPos treeBase = new BlockPos(x, y, z);
				BlockPos treeRoot = treeBase.func_177977_b();
				IBlockState state = world.func_180495_p(treeRoot);
				state.func_177230_c().onPlantGrow(state, world, treeRoot, treeBase);
				for (yOffset = y - 3 + treeHeight; yOffset <= y + treeHeight; ++yOffset) {
					int var12 = yOffset - (y + treeHeight), center = 1 - var12 / 2;
					for (xOffset = x - center; xOffset <= x + center; ++xOffset) {
						int xPos = xOffset - x, t = xPos >> 15;
						xPos = (xPos + t) ^ t;
						for (zOffset = z - center; zOffset <= z + center; ++zOffset) {
							int zPos = zOffset - z;
							zPos = (zPos + (t = zPos >> 31)) ^ t;
							IBlockState state1 = world.func_180495_p(new BlockPos(xOffset, yOffset, zOffset));
							Block block = state1.func_177230_c();
							if (((xPos != center | zPos != center) || rand.nextInt(2) != 0 && var12 != 0)
								&& (block == null
								|| block.isLeaves(state1, world, new BlockPos(xOffset, yOffset, zOffset))
								|| block.isAir(state1, world, new BlockPos(xOffset, yOffset, zOffset))
								|| block.canBeReplacedByLeaves(state1, world,
								new BlockPos(xOffset, yOffset, zOffset)))) {
								this.func_175903_a(world, new BlockPos(xOffset, yOffset, zOffset),
									ModBlocks.RUBBER_LEAVES.func_176223_P().func_177226_a(BlockRubberLeaves.field_176237_a, true).func_177226_a(BlockRubberLeaves.field_176236_b, false));
								hasPlacedBlock = true;
							}
						}
					}
				}

				BlockPos topLogPos = null;
				for (yOffset = 0; yOffset < treeHeight; ++yOffset) {
					BlockPos blockpos = new BlockPos(x, y + yOffset, z);
					IBlockState state1 = world.func_180495_p(blockpos);
					Block block = state1.func_177230_c();
					if (block == null || block.isAir(state1, world, blockpos) || block.isLeaves(state1, world, blockpos)
						|| block.func_176200_f(world, blockpos)) {
						IBlockState newState = ModBlocks.RUBBER_LOG.func_176223_P();
						boolean isAddingSap = false;
						if (rand.nextInt(Core.worldGen.config.rubberTreeConfig.sapRarity) == 0) {
							newState = newState.func_177226_a(BlockRubberLog.HAS_SAP, true)
								.func_177226_a(BlockRubberLog.SAP_SIDE, EnumFacing.func_176731_b(rand.nextInt(4)));
							isAddingSap = true;
						}
						if (isAddingSap) {
							world.func_180501_a(blockpos, newState, 2);
						} else {
							this.func_175903_a(world, blockpos, newState);
						}
						hasPlacedBlock = true;
						topLogPos = blockpos;
					}
				}
				if (topLogPos != null) {
					for (int i = 0; i < Core.worldGen.config.rubberTreeConfig.spireHeight; i++) {
						BlockPos spikePos = topLogPos.func_177981_b(i);
						this.func_175903_a(world, spikePos, ModBlocks.RUBBER_LEAVES.func_176223_P()
							.func_177226_a(BlockRubberLeaves.field_176237_a, true).func_177226_a(BlockRubberLeaves.field_176236_b, false));
					}
				}
			}
			return hasPlacedBlock;
		}
		return false;
	}
}
