/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.gui.widget;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import me.shedaniel.clothconfig2.ClothConfigInitializer;
import me.shedaniel.clothconfig2.api.ScissorsHandler;
import me.shedaniel.clothconfig2.api.ScrollingContainer;
import me.shedaniel.clothconfig2.gui.widget.DynamicNewSmoothScrollingEntryListWidget;
import me.shedaniel.math.Point;
import me.shedaniel.math.Rectangle;
import me.shedaniel.math.impl.PointHelper;
import me.shedaniel.rei.RoughlyEnoughItemsCore;
import me.shedaniel.rei.api.*;
import me.shedaniel.rei.api.widgets.Tooltip;
import me.shedaniel.rei.gui.ContainerScreenOverlay;
import me.shedaniel.rei.gui.config.EntryPanelOrdering;
import me.shedaniel.rei.impl.OptimalEntryStack;
import me.shedaniel.rei.impl.ScreenHelper;
import me.shedaniel.rei.impl.SearchArgument;
import me.shedaniel.rei.utils.CollectionUtils;
import net.minecraft.class_1159;
import net.minecraft.class_1269;
import net.minecraft.class_1761;
import net.minecraft.class_1792;
import net.minecraft.class_2561;
import net.minecraft.class_2585;
import net.minecraft.class_2588;
import net.minecraft.class_289;
import net.minecraft.class_310;
import net.minecraft.class_3532;
import net.minecraft.class_4587;
import net.minecraft.class_4597;
import net.minecraft.class_746;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.*;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@ApiStatus.Internal
public class EntryListWidget extends WidgetWithBounds {
    
    static final Supplier<Boolean> RENDER_ENCHANTMENT_GLINT = ConfigObject.getInstance()::doesRenderEntryEnchantmentGlint;
    static final Comparator<? super EntryStack> ENTRY_NAME_COMPARER = Comparator.comparing(stack -> stack.asFormatStrippedText().getString());
    static final Comparator<? super EntryStack> ENTRY_GROUP_COMPARER = Comparator.comparingInt(stack -> {
        if (stack.getType() == EntryStack.Type.ITEM) {
            class_1761 group = stack.getItem().method_7859();
            if (group != null)
                return group.method_7741();
        }
        return Integer.MAX_VALUE;
    });
    private static final int SIZE = 18;
    private static final boolean LAZY = true;
    private static int page;
    protected final ScrollingContainer scrolling = new ScrollingContainer() {
        @Override
        public Rectangle getBounds() {
            return EntryListWidget.this.getBounds();
        }
        
        @Override
        public int getMaxScrollHeight() {
            return class_3532.method_15386((allStacks.size() + blockedCount) / (innerBounds.width / (float) entrySize())) * entrySize();
        }
    };
    protected int blockedCount;
    private boolean debugTime;
    private double lastAverageDebugTime;
    private double averageDebugTime;
    private double lastTotalDebugTime;
    private double totalDebugTime;
    private float totalDebugTimeDelta;
    private Rectangle bounds, innerBounds;
    private List<EntryStack> allStacks = null;
    private List<EntryListEntry> entries = Collections.emptyList();
    private List<Widget> renders = Collections.emptyList();
    private List<Widget> widgets = Collections.emptyList();
    private List<SearchArgument.SearchArguments> lastSearchArguments = Collections.emptyList();
    private String lastSearchTerm = null;
    
    public static int entrySize() {
        return class_3532.method_15384(SIZE * ConfigObject.getInstance().getEntrySize());
    }
    
    static boolean notSteppingOnExclusionZones(int left, int top, Rectangle listArea) {
        class_310 instance = class_310.method_1551();
        for (OverlayDecider decider : DisplayHelper.getInstance().getSortedOverlayDeciders(instance.field_1755.getClass())) {
            class_1269 fit = decider.isInZone(left, top);
            if (fit == class_1269.field_5814)
                return false;
            class_1269 fit2 = decider.isInZone(left + 18, top + 18);
            if (fit2 == class_1269.field_5814)
                return false;
            if (fit == class_1269.field_5812 && fit2 == class_1269.field_5812)
                return true;
        }
        return true;
    }
    
    private static Rectangle updateInnerBounds(Rectangle bounds) {
        if (ConfigObject.getInstance().isEntryListWidgetScrolled()) {
            int width = Math.max(class_3532.method_15375((bounds.width - 2 - 6) / (float) entrySize()), 1);
            if (ConfigObject.getInstance().isLeftHandSidePanel())
                return new Rectangle((int) (bounds.getCenterX() - width * (entrySize() / 2f) + 3), bounds.y, width * entrySize(), bounds.height);
            return new Rectangle((int) (bounds.getCenterX() - width * (entrySize() / 2f) - 3), bounds.y, width * entrySize(), bounds.height);
        }
        int width = Math.max(class_3532.method_15375((bounds.width - 2) / (float) entrySize()), 1);
        int height = Math.max(class_3532.method_15375((bounds.height - 2) / (float) entrySize()), 1);
        return new Rectangle((int) (bounds.getCenterX() - width * (entrySize() / 2f)), (int) (bounds.getCenterY() - height * (entrySize() / 2f)), width * entrySize(), height * entrySize());
    }
    
    @Override
    public boolean method_25401(double double_1, double double_2, double double_3) {
        if (ConfigObject.getInstance().isEntryListWidgetScrolled() && bounds.contains(double_1, double_2)) {
            scrolling.offset(ClothConfigInitializer.getScrollStep() * -double_3, true);
            return true;
        }
        return super.method_25401(double_1, double_2, double_3);
    }
    
    @NotNull
    @Override
    public Rectangle getBounds() {
        return bounds;
    }
    
    public int getPage() {
        return page;
    }
    
    public void setPage(int page) {
        EntryListWidget.page = page;
    }
    
    public void previousPage() {
        page--;
    }
    
    public void nextPage() {
        page++;
    }
    
    public int getTotalPages() {
        if (ConfigObject.getInstance().isEntryListWidgetScrolled())
            return 1;
        return class_3532.method_15386(allStacks.size() / (float) entries.size());
    }
    
    public static void renderEntries(boolean debugTime, int[] size, long[] time, boolean fastEntryRendering, class_4587 matrices, int mouseX, int mouseY, float delta, List<EntryListEntryWidget> entries) {
        if (entries.isEmpty()) return;
        EntryListEntryWidget firstWidget = entries.get(0);
        EntryStack first = firstWidget.getCurrentEntry();
        if (fastEntryRendering && first instanceof OptimalEntryStack) {
            OptimalEntryStack firstStack = (OptimalEntryStack) first;
            firstStack.optimisedRenderStart(matrices, delta);
            long l = debugTime ? System.nanoTime() : 0;
            class_4597.class_4598 immediate = class_310.method_1551().method_22940().method_23000();
            for (EntryListEntryWidget listEntry : entries) {
                EntryStack currentEntry = listEntry.getCurrentEntry();
                currentEntry.setZ(100);
                listEntry.drawBackground(matrices, mouseX, mouseY, delta);
                ((OptimalEntryStack) currentEntry).optimisedRenderBase(matrices, immediate, listEntry.getInnerBounds(), mouseX, mouseY, delta);
                if (debugTime && !currentEntry.isEmpty()) size[0]++;
            }
            immediate.method_22993();
            for (EntryListEntryWidget listEntry : entries) {
                EntryStack currentEntry = listEntry.getCurrentEntry();
                ((OptimalEntryStack) currentEntry).optimisedRenderOverlay(matrices, listEntry.getInnerBounds(), mouseX, mouseY, delta);
                if (listEntry.containsMouse(mouseX, mouseY)) {
                    listEntry.queueTooltip(matrices, mouseX, mouseY, delta);
                    listEntry.drawHighlighted(matrices, mouseX, mouseY, delta);
                }
            }
            if (debugTime) time[0] += (System.nanoTime() - l);
            firstStack.optimisedRenderEnd(matrices, delta);
        } else {
            for (EntryListEntryWidget entry : entries) {
                if (entry.getCurrentEntry().isEmpty())
                    continue;
                if (debugTime) {
                    size[0]++;
                    long l = System.nanoTime();
                    entry.method_25394(matrices, mouseX, mouseY, delta);
                    time[0] += (System.nanoTime() - l);
                } else entry.method_25394(matrices, mouseX, mouseY, delta);
            }
        }
    }
    
    @Override
    public void method_25394(class_4587 matrices, int mouseX, int mouseY, float delta) {
        int[] size = {0};
        long[] time = {0};
        long totalTimeStart = debugTime ? System.nanoTime() : 0;
        boolean fastEntryRendering = ConfigObject.getInstance().doesFastEntryRendering();
        if (ConfigObject.getInstance().isEntryListWidgetScrolled()) {
            for (EntryListEntry entry : entries)
                entry.clearStacks();
            ScissorsHandler.INSTANCE.scissor(bounds);
            
            int skip = Math.max(0, class_3532.method_15357(scrolling.scrollAmount / (float) entrySize()));
            int nextIndex = skip * innerBounds.width / entrySize();
            int[] i = {nextIndex};
            blockedCount = 0;
            
            Stream<EntryListEntry> entryStream = this.entries.stream().skip(nextIndex).filter(entry -> {
                entry.getBounds().y = (int) (entry.backupY - scrolling.scrollAmount);
                if (entry.getBounds().y > bounds.getMaxY()) return false;
                if (notSteppingOnExclusionZones(entry.getBounds().x, entry.getBounds().y, innerBounds)) {
                    EntryStack stack = allStacks.get(i[0]++);
                    if (!stack.isEmpty()) {
                        entry.entry(stack);
                        return true;
                    }
                } else {
                    blockedCount++;
                }
                return false;
            }).limit(Math.max(0, allStacks.size() - i[0]));
            
            if (fastEntryRendering) {
                entryStream.collect(Collectors.groupingBy(entryListEntry -> OptimalEntryStack.groupingHashFrom(entryListEntry.getCurrentEntry()))).forEach((integer, entries) -> {
                    renderEntries(debugTime, size, time, fastEntryRendering, matrices, mouseX, mouseY, delta, (List) entries);
                });
            } else {
                renderEntries(debugTime, size, time, fastEntryRendering, matrices, mouseX, mouseY, delta, entryStream.collect(Collectors.toList()));
            }
            
            updatePosition(delta);
            ScissorsHandler.INSTANCE.removeLastScissor();
            scrolling.renderScrollBar(0, 1, REIHelper.getInstance().isDarkThemeEnabled() ? 0.8f : 1f);
        } else {
            for (Widget widget : renders) {
                widget.method_25394(matrices, mouseX, mouseY, delta);
            }
            if (fastEntryRendering) {
                entries.stream().collect(Collectors.groupingBy(entryListEntry -> OptimalEntryStack.groupingHashFrom(entryListEntry.getCurrentEntry()))).forEach((integer, entries) -> {
                    renderEntries(debugTime, size, time, fastEntryRendering, matrices, mouseX, mouseY, delta, (List) entries);
                });
            } else {
                renderEntries(debugTime, size, time, fastEntryRendering, matrices, mouseX, mouseY, delta, (List) entries);
            }
        }
        
        if (debugTime) {
            long totalTime = System.nanoTime() - totalTimeStart;
            averageDebugTime += (time[0] / (double) size[0]) * delta;
            totalDebugTime += totalTime / 1000000d * delta;
            totalDebugTimeDelta += delta;
            if (totalDebugTimeDelta >= 20) {
                lastAverageDebugTime = averageDebugTime / totalDebugTimeDelta;
                lastTotalDebugTime = totalDebugTime / totalDebugTimeDelta;
                averageDebugTime = 0;
                totalDebugTime = 0;
                totalDebugTimeDelta = 0;
            } else if (lastAverageDebugTime == 0) {
                lastAverageDebugTime = time[0] / (double) size[0];
                totalDebugTime = totalTime / 1000000d;
            }
            int z = getZ();
            setZ(500);
            class_2561 debugText = new class_2585(String.format("%d entries, avg. %.0fns, ttl. %.2fms, %s fps", size[0], lastAverageDebugTime, lastTotalDebugTime, minecraft.field_1770.split(" ")[0]));
            int stringWidth = font.method_27525(debugText);
            method_25296(matrices, Math.min(bounds.x, minecraft.field_1755.field_22789 - stringWidth - 2), bounds.y, bounds.x + stringWidth + 2, bounds.y + font.field_2000 + 2, -16777216, -16777216);
            class_4597.class_4598 immediate = class_4597.method_22991(class_289.method_1348().method_1349());
            matrices.method_22903();
            matrices.method_22904(0.0D, 0.0D, getZ());
            class_1159 matrix = matrices.method_23760().method_23761();
            font.method_22942(debugText.method_30937(), Math.min(bounds.x + 2, minecraft.field_1755.field_22789 - stringWidth), bounds.y + 2, -1, false, matrix, immediate, false, 0, 15728880);
            immediate.method_22993();
            setZ(z);
            matrices.method_22909();
        }
        
        if (containsMouse(mouseX, mouseY) && ClientHelper.getInstance().isCheating() && !minecraft.field_1724.field_7514.method_7399().method_7960() && RoughlyEnoughItemsCore.canDeleteItems()) {
            EntryStack stack = EntryStack.create(minecraft.field_1724.field_7514.method_7399().method_7972());
            if (stack.getType() == EntryStack.Type.FLUID) {
                class_1792 bucketItem = stack.getFluid().method_15774();
                if (bucketItem != null) {
                    stack = EntryStack.create(bucketItem);
                }
            }
            for (Widget child : method_25396()) {
                if (child.containsMouse(mouseX, mouseY) && child instanceof EntryWidget) {
                    if (((EntryWidget) child).cancelDeleteItems(stack)) {
                        return;
                    }
                }
            }
            Tooltip.create(new class_2588("text.rei.delete_items")).queue();
        }
    }
    
    private int getScrollbarMinX() {
        if (ConfigObject.getInstance().isLeftHandSidePanel())
            return bounds.x + 1;
        return bounds.getMaxX() - 7;
    }
    
    @Override
    public boolean method_25403(double mouseX, double mouseY, int button, double dx, double dy) {
        if (scrolling.mouseDragged(mouseX, mouseY, button, dx, dy, ConfigObject.getInstance().doesSnapToRows(), entrySize()))
            return true;
        return super.method_25403(mouseX, mouseY, button, dx, dy);
    }
    
    private void updatePosition(float delta) {
        if (ConfigObject.getInstance().doesSnapToRows() && scrolling.scrollTarget >= 0 && scrolling.scrollTarget <= scrolling.getMaxScroll()) {
            double nearestRow = Math.round(scrolling.scrollTarget / (double) entrySize()) * (double) entrySize();
            if (!DynamicNewSmoothScrollingEntryListWidget.Precision.almostEquals(scrolling.scrollTarget, nearestRow, DynamicNewSmoothScrollingEntryListWidget.Precision.FLOAT_EPSILON))
                scrolling.scrollTarget += (nearestRow - scrolling.scrollTarget) * Math.min(delta / 2.0, 1.0);
            else
                scrolling.scrollTarget = nearestRow;
        }
        scrolling.updatePosition(delta);
    }
    
    @Override
    public boolean method_25404(int int_1, int int_2, int int_3) {
        if (containsMouse(PointHelper.ofMouse()))
            for (Widget widget : widgets)
                if (widget.method_25404(int_1, int_2, int_3))
                    return true;
        return false;
    }
    
    public void updateArea(@Nullable String searchTerm) {
        this.bounds = ScreenHelper.getItemListArea(ScreenHelper.getLastOverlay().getBounds());
        FavoritesListWidget favoritesListWidget = ContainerScreenOverlay.getFavoritesListWidget();
        if (favoritesListWidget != null)
            favoritesListWidget.updateFavoritesBounds(searchTerm);
        if (searchTerm != null)
            updateSearch(searchTerm, true);
        else if (allStacks == null || (favoritesListWidget != null && favoritesListWidget.favorites == null))
            updateSearch("", true);
        else
            updateEntriesPosition();
    }
    
    public void updateEntriesPosition() {
        this.innerBounds = updateInnerBounds(bounds);
        if (!ConfigObject.getInstance().isEntryListWidgetScrolled()) {
            this.renders = Lists.newArrayList();
            page = Math.max(page, 0);
            List<EntryListEntry> entries = Lists.newArrayList();
            int width = innerBounds.width / entrySize();
            int height = innerBounds.height / entrySize();
            for (int currentY = 0; currentY < height; currentY++) {
                for (int currentX = 0; currentX < width; currentX++) {
                    if (notSteppingOnExclusionZones(currentX * entrySize() + innerBounds.x, currentY * entrySize() + innerBounds.y, innerBounds)) {
                        entries.add((EntryListEntry) new EntryListEntry(currentX * entrySize() + innerBounds.x, currentY * entrySize() + innerBounds.y).noBackground());
                    }
                }
            }
            page = Math.max(Math.min(page, getTotalPages() - 1), 0);
            List<EntryStack> subList = allStacks.stream().skip(Math.max(0, page * entries.size())).limit(Math.max(0, entries.size() - Math.max(0, -page * entries.size()))).collect(Collectors.toList());
            for (int i = 0; i < subList.size(); i++) {
                EntryStack stack = subList.get(i);
                entries.get(i + Math.max(0, -page * entries.size())).clearStacks().entry(stack);
            }
            this.entries = entries;
            this.widgets = Lists.newArrayList(renders);
            this.widgets.addAll(entries);
        } else {
            page = 0;
            int width = innerBounds.width / entrySize();
            int pageHeight = innerBounds.height / entrySize();
            int slotsToPrepare = Math.max(allStacks.size() * 3, width * pageHeight * 3);
            int currentX = 0;
            int currentY = 0;
            List<EntryListEntry> entries = Lists.newArrayList();
            for (int i = 0; i < slotsToPrepare; i++) {
                int xPos = currentX * entrySize() + innerBounds.x;
                int yPos = currentY * entrySize() + innerBounds.y;
                entries.add((EntryListEntry) new EntryListEntry(xPos, yPos).noBackground());
                currentX++;
                if (currentX >= width) {
                    currentX = 0;
                    currentY++;
                }
            }
            this.entries = entries;
            this.widgets = Lists.newArrayList(renders);
            this.widgets.addAll(entries);
        }
        FavoritesListWidget favoritesListWidget = ContainerScreenOverlay.getFavoritesListWidget();
        if (favoritesListWidget != null)
            favoritesListWidget.updateEntriesPosition();
    }
    
    @ApiStatus.Internal
    public List<EntryStack> getAllStacks() {
        return allStacks;
    }
    
    public void updateSearch(String searchTerm) {
        updateSearch(searchTerm, true);
    }
    
    public void updateSearch(String searchTerm, boolean ignoreLastSearch) {
        long started = System.nanoTime();
        if (ignoreLastSearch || this.lastSearchTerm == null || !this.lastSearchTerm.equals(searchTerm)) {
            this.lastSearchTerm = searchTerm;
            this.lastSearchArguments = SearchArgument.processSearchTerm(searchTerm);
            List<EntryStack> list = Lists.newArrayList();
            boolean checkCraftable = ConfigManager.getInstance().isCraftableOnlyEnabled() && !ScreenHelper.inventoryStacks.isEmpty();
            Set<Integer> workingItems = checkCraftable ? Sets.newHashSet() : null;
            if (checkCraftable)
                workingItems.addAll(CollectionUtils.map(RecipeHelper.getInstance().findCraftableEntriesByItems(CollectionUtils.map(ScreenHelper.inventoryStacks, EntryStack::create)), EntryStack::hashIgnoreAmount));
            List<EntryStack> stacks = EntryRegistry.getInstance().getPreFilteredList();
            if (stacks instanceof CopyOnWriteArrayList && !stacks.isEmpty()) {
                if (ConfigObject.getInstance().shouldAsyncSearch()) {
                    int size = ConfigObject.getInstance().getNumberAsyncSearch();
                    List<CompletableFuture<List<EntryStack>>> completableFutures = Lists.newArrayList();
                    for (int i = 0; i < stacks.size(); i += size) {
                        int[] start = {i};
                        completableFutures.add(CompletableFuture.supplyAsync(() -> {
                            int end = Math.min(stacks.size(), start[0] + size);
                            List<EntryStack> filtered = Lists.newArrayList();
                            for (; start[0] < end; start[0]++) {
                                EntryStack stack = stacks.get(start[0]);
                                if (canLastSearchTermsBeAppliedTo(stack)) {
                                    if (workingItems != null && !workingItems.contains(stack.hashIgnoreAmount()))
                                        continue;
                                    filtered.add(stack.copy().setting(EntryStack.Settings.RENDER_COUNTS, EntryStack.Settings.FALSE).setting(EntryStack.Settings.Item.RENDER_ENCHANTMENT_GLINT, RENDER_ENCHANTMENT_GLINT));
                                }
                            }
                            return filtered;
                        }));
                    }
                    try {
                        CompletableFuture.allOf(completableFutures.toArray(new CompletableFuture[0])).get(10, TimeUnit.SECONDS);
                    } catch (InterruptedException | ExecutionException | TimeoutException e) {
                        e.printStackTrace();
                    }
                    for (CompletableFuture<List<EntryStack>> future : completableFutures) {
                        List<EntryStack> now = future.getNow(null);
                        if (now != null)
                            list.addAll(now);
                    }
                } else {
                    for (EntryStack stack : stacks) {
                        if (canLastSearchTermsBeAppliedTo(stack)) {
                            if (workingItems != null && !workingItems.contains(stack.hashIgnoreAmount()))
                                continue;
                            list.add(stack.copy().setting(EntryStack.Settings.RENDER_COUNTS, EntryStack.Settings.FALSE).setting(EntryStack.Settings.Item.RENDER_ENCHANTMENT_GLINT, RENDER_ENCHANTMENT_GLINT));
                        }
                    }
                }
            }
            EntryPanelOrdering ordering = ConfigObject.getInstance().getItemListOrdering();
            if (ordering == EntryPanelOrdering.NAME)
                list.sort(ENTRY_NAME_COMPARER);
            if (ordering == EntryPanelOrdering.GROUPS)
                list.sort(ENTRY_GROUP_COMPARER);
            if (!ConfigObject.getInstance().isItemListAscending())
                Collections.reverse(list);
            allStacks = list;
        }
        debugTime = ConfigObject.getInstance().doDebugRenderTimeRequired();
        FavoritesListWidget favoritesListWidget = ContainerScreenOverlay.getFavoritesListWidget();
        if (favoritesListWidget != null)
            favoritesListWidget.updateSearch(this, searchTerm);
        long ended = System.nanoTime();
        long time = ended - started;
        if (ConfigObject.getInstance().doDebugSearchTimeRequired())
            RoughlyEnoughItemsCore.LOGGER.info("Search Used: %.2fms", time * 1e-6);
        updateEntriesPosition();
    }
    
    public boolean canLastSearchTermsBeAppliedTo(EntryStack stack) {
        return lastSearchArguments.isEmpty() || SearchArgument.canSearchTermsBeAppliedTo(stack, lastSearchArguments);
    }
    
    @Override
    public List<? extends Widget> method_25396() {
        return widgets;
    }
    
    @Override
    public boolean method_25402(double double_1, double double_2, int int_1) {
        if (ConfigObject.getInstance().isEntryListWidgetScrolled()) {
            if (scrolling.updateDraggingState(double_1, double_2, int_1))
                return true;
        }
        for (Widget widget : method_25396())
            if (widget.method_25402(double_1, double_2, int_1))
                return true;
        return false;
    }
    
    @Override
    public boolean method_25406(double mouseX, double mouseY, int button) {
        if (containsMouse(mouseX, mouseY)) {
            for (Widget widget : method_25396())
                if (widget.method_25406(mouseX, mouseY, button))
                    return true;
            class_746 player = minecraft.field_1724;
            if (ClientHelper.getInstance().isCheating() && player != null && player.field_7514 != null && !player.field_7514.method_7399().method_7960() && RoughlyEnoughItemsCore.canDeleteItems()) {
                ClientHelper.getInstance().sendDeletePacket();
                return true;
            }
            if (player != null && player.field_7514 != null && !player.field_7514.method_7399().method_7960() && RoughlyEnoughItemsCore.hasPermissionToUsePackets())
                return false;
        }
        return false;
    }
    
    private class EntryListEntry extends EntryListEntryWidget {
        private EntryListEntry(int x, int y) {
            super(new Point(x, y));
        }
        
        @Override
        public boolean containsMouse(double mouseX, double mouseY) {
            return super.containsMouse(mouseX, mouseY) && bounds.contains(mouseX, mouseY);
        }
    }
}
