/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import me.shedaniel.cloth.api.client.events.v0.ClothClientHooks;
import me.shedaniel.math.Point;
import me.shedaniel.math.Rectangle;
import me.shedaniel.math.api.Executor;
import me.shedaniel.rei.RoughlyEnoughItemsCore;
import me.shedaniel.rei.RoughlyEnoughItemsState;
import me.shedaniel.rei.api.ConfigManager;
import me.shedaniel.rei.api.ConfigObject;
import me.shedaniel.rei.api.REIHelper;
import me.shedaniel.rei.api.REIOverlay;
import me.shedaniel.rei.api.widgets.Tooltip;
import me.shedaniel.rei.gui.ContainerScreenOverlay;
import me.shedaniel.rei.gui.OverlaySearchField;
import me.shedaniel.rei.gui.RecipeScreen;
import me.shedaniel.rei.gui.WarningAndErrorScreen;
import me.shedaniel.rei.gui.config.SearchFieldLocation;
import me.shedaniel.rei.gui.widget.TextFieldWidget;
import net.fabricmc.api.ClientModInitializer;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.fabricmc.fabric.api.event.client.ClientTickCallback;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_1041;
import net.minecraft.class_1269;
import net.minecraft.class_1799;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_437;
import net.minecraft.class_4587;
import net.minecraft.class_465;
import org.apache.logging.log4j.util.TriConsumer;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;

@ApiStatus.Internal
@Environment(EnvType.CLIENT)
public class ScreenHelper implements ClientModInitializer, REIHelper {
    private static final class_2960 DISPLAY_TEXTURE = new class_2960("roughlyenoughitems", "textures/gui/display.png");
    private static final class_2960 DISPLAY_TEXTURE_DARK = new class_2960("roughlyenoughitems", "textures/gui/display_dark.png");
    private OverlaySearchField searchField;
    @ApiStatus.Internal
    public static List<class_1799> inventoryStacks = Lists.newArrayList();
    private static ContainerScreenOverlay overlay;
    private static class_465<?> previousContainerScreen = null;
    private static LinkedHashSet<RecipeScreen> lastRecipeScreen = Sets.newLinkedHashSetWithExpectedSize(5);
    private static ScreenHelper instance;
    
    /**
     * @return the instance of screen helper
     * @see REIHelper#getInstance()
     */
    @ApiStatus.Internal
    public static ScreenHelper getInstance() {
        return instance;
    }
    
    @Override
    public void queueTooltip(@Nullable Tooltip tooltip) {
        if (overlay != null && tooltip != null) {
            overlay.addTooltip(tooltip);
        }
    }
    
    @Override
    public TextFieldWidget getSearchTextField() {
        return searchField;
    }
    
    @Override
    public List<class_1799> getInventoryStacks() {
        return inventoryStacks;
    }
    
    public static OverlaySearchField getSearchField() {
        return (OverlaySearchField) getInstance().getSearchTextField();
    }
    
    @ApiStatus.Internal
    public static void setSearchField(OverlaySearchField searchField) {
        getInstance().searchField = searchField;
    }
    
    public static void storeRecipeScreen(RecipeScreen screen) {
        while (lastRecipeScreen.size() >= 5)
            lastRecipeScreen.remove(Iterables.get(lastRecipeScreen, 0));
        lastRecipeScreen.add(screen);
    }
    
    public static boolean hasLastRecipeScreen() {
        return !lastRecipeScreen.isEmpty();
    }
    
    public static class_437 getLastRecipeScreen() {
        RecipeScreen screen = Iterables.getLast(lastRecipeScreen);
        lastRecipeScreen.remove(screen);
        screen.recalculateCategoryPage();
        return (class_437) screen;
    }
    
    @ApiStatus.Internal
    public static void clearLastRecipeScreenData() {
        lastRecipeScreen.clear();
    }
    
    public static boolean isOverlayVisible() {
        return ConfigObject.getInstance().isOverlayVisible();
    }
    
    public static void toggleOverlayVisible() {
        ConfigObject.getInstance().setOverlayVisible(!ConfigObject.getInstance().isOverlayVisible());
        ConfigManager.getInstance().saveConfig();
    }
    
    public static Optional<ContainerScreenOverlay> getOptionalOverlay() {
        return Optional.ofNullable(overlay);
    }
    
    @Override
    public Optional<REIOverlay> getOverlay() {
        return Optional.ofNullable(overlay);
    }
    
    public static ContainerScreenOverlay getLastOverlay(boolean reset, boolean setPage) {
        if (overlay == null || reset) {
            overlay = new ContainerScreenOverlay();
            overlay.init();
            getSearchField().setFocused(false);
        }
        return overlay;
    }
    
    public static ContainerScreenOverlay getLastOverlay() {
        return getLastOverlay(false, false);
    }
    
    /**
     * @see REIHelper#getPreviousContainerScreen()
     */
    @Deprecated
    @ApiStatus.ScheduledForRemoval
    public static class_465<?> getLastHandledScreen() {
        return previousContainerScreen;
    }
    
    @Override
    public class_465<?> getPreviousContainerScreen() {
        return previousContainerScreen;
    }
    
    public static void setPreviousContainerScreen(class_465<?> previousContainerScreen) {
        ScreenHelper.previousContainerScreen = previousContainerScreen;
    }
    
    public static void drawHoveringWidget(class_4587 matrices, int x, int y, TriConsumer<class_4587, Point, Float> consumer, int width, int height, float delta) {
        class_1041 window = class_310.method_1551().method_22683();
        drawHoveringWidget(matrices, window.method_4486(), window.method_4502(), x, y, consumer, width, height, delta);
    }
    
    public static void drawHoveringWidget(class_4587 matrices, int screenWidth, int screenHeight, int x, int y, TriConsumer<class_4587, Point, Float> consumer, int width, int height, float delta) {
        int actualX = Math.max(x + 12, 6);
        int actualY = Math.min(y - height / 2, screenHeight - height - 6);
        if (actualX + width > screenWidth)
            actualX -= 24 + width;
        if (actualY < 6)
            actualY += 24;
        consumer.accept(matrices, new Point(actualX, actualY), delta);
    }
    
    /**
     * @deprecated Please switch to {@link REIHelper#isDarkThemeEnabled()}
     */
    @Deprecated
    @ApiStatus.Internal
    @ApiStatus.ScheduledForRemoval
    public static boolean isDarkModeEnabled() {
        return ConfigObject.getInstance().isUsingDarkTheme();
    }
    
    @Override
    public boolean isDarkThemeEnabled() {
        return isDarkModeEnabled();
    }
    
    @Override
    public class_2960 getDefaultDisplayTexture() {
        return isDarkThemeEnabled() ? DISPLAY_TEXTURE_DARK : DISPLAY_TEXTURE;
    }
    
    public ScreenHelper() {
        ScreenHelper.instance = this;
        RoughlyEnoughItemsCore.attachInstance(instance, REIHelper.class);
    }
    
    public static Rectangle getItemListArea(Rectangle bounds) {
        return new Rectangle(bounds.x, bounds.y + 2 + (ConfigObject.getInstance().getSearchFieldLocation() == SearchFieldLocation.TOP_SIDE ? 24 : 0) + (ConfigObject.getInstance().isEntryListWidgetScrolled() ? 0 : 22), bounds.width, bounds.height - (ConfigObject.getInstance().getSearchFieldLocation() != SearchFieldLocation.CENTER ? 27 + 22 : 27) + (!ConfigObject.getInstance().isEntryListWidgetScrolled() ? 0 : 22));
    }
    
    public static Rectangle getFavoritesListArea(Rectangle bounds) {
        int offset = 6 + (!ConfigObject.getInstance().isLowerConfigButton() ? 25 : 0) + (ConfigObject.getInstance().doesShowUtilsButtons() ? 25 : 0);
        return new Rectangle(bounds.x, bounds.y + 2 + offset, bounds.width, bounds.height - 5 - offset);
    }
    
    @Override
    public void onInitializeClient() {
        ClothClientHooks.SCREEN_INIT_PRE.register((client, screen, screenHooks) -> {
            if ((!RoughlyEnoughItemsState.getErrors().isEmpty() || !RoughlyEnoughItemsState.getWarnings().isEmpty()) && !(screen instanceof WarningAndErrorScreen)) {
                WarningAndErrorScreen warningAndErrorScreen = WarningAndErrorScreen.INSTANCE.method_15332();
                warningAndErrorScreen.setParent(screen);
                try {
                    if (client.field_1755 != null) client.field_1755.method_25432();
                } catch (Throwable ignored) {
                }
                client.field_1755 = null;
                client.method_1507(warningAndErrorScreen);
            } else if (previousContainerScreen != screen && screen instanceof class_465)
                previousContainerScreen = (class_465<?>) screen;
            return class_1269.field_5811;
        });
        boolean loaded = FabricLoader.getInstance().isModLoaded("fabric-events-lifecycle-v0");
        if (!loaded) {
            RoughlyEnoughItemsState.error("Fabric API is not installed!", "https://www.curseforge.com/minecraft/mc-mods/fabric-api/files/all");
            return;
        }
        Executor.run(() -> () -> {
            ClientTickCallback.EVENT.register(minecraftClient -> {
                if (isOverlayVisible() && getSearchField() != null)
                    getSearchField().method_16896();
            });
        });
    }
}
