/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import me.shedaniel.cloth.api.client.events.v0.ClothClientHooks;
import me.shedaniel.math.Point;
import me.shedaniel.math.Rectangle;
import me.shedaniel.math.api.Executor;
import me.shedaniel.rei.api.*;
import me.shedaniel.rei.api.fluid.FluidSupportProvider;
import me.shedaniel.rei.api.fractions.Fraction;
import me.shedaniel.rei.api.plugins.REIPluginV0;
import me.shedaniel.rei.api.subsets.SubsetsRegistry;
import me.shedaniel.rei.api.widgets.*;
import me.shedaniel.rei.gui.ContainerScreenOverlay;
import me.shedaniel.rei.gui.widget.EntryWidget;
import me.shedaniel.rei.gui.widget.QueuedTooltip;
import me.shedaniel.rei.gui.widget.Widget;
import me.shedaniel.rei.impl.*;
import me.shedaniel.rei.impl.subsets.SubsetsRegistryImpl;
import me.shedaniel.rei.impl.widgets.*;
import me.shedaniel.rei.tests.plugin.REITestPlugin;
import net.fabricmc.api.ClientModInitializer;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.fabricmc.fabric.api.network.ClientSidePacketRegistry;
import net.fabricmc.loader.api.FabricLoader;
import net.fabricmc.loader.api.ModContainer;
import net.minecraft.class_1074;
import net.minecraft.class_1269;
import net.minecraft.class_1714;
import net.minecraft.class_1735;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1856;
import net.minecraft.class_1863;
import net.minecraft.class_2561;
import net.minecraft.class_2585;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_342;
import net.minecraft.class_344;
import net.minecraft.class_3611;
import net.minecraft.class_364;
import net.minecraft.class_437;
import net.minecraft.class_465;
import net.minecraft.class_479;
import net.minecraft.class_481;
import net.minecraft.class_490;
import net.minecraft.class_505;
import net.minecraft.class_507;
import net.minecraft.class_518;
import net.minecraft.class_5348;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.BiFunction;
import java.util.function.Supplier;

import static me.shedaniel.rei.impl.Internals.attachInstance;

@ApiStatus.Internal
@Environment(EnvType.CLIENT)
public class RoughlyEnoughItemsCore implements ClientModInitializer {
    
    @ApiStatus.Internal public static final Logger LOGGER = LogManager.getFormatterLogger("REI");
    private static final Map<class_2960, REIPluginEntry> plugins = Maps.newHashMap();
    private static final ExecutorService SYNC_RECIPES = Executors.newSingleThreadScheduledExecutor(r -> new Thread(r, "REI-SyncRecipes"));
    @ApiStatus.Experimental
    public static boolean isLeftModePressed = false;
    
    static {
        attachInstance(new RecipeHelperImpl(), RecipeHelper.class);
        attachInstance(new EntryRegistryImpl(), EntryRegistry.class);
        attachInstance(new DisplayHelperImpl(), DisplayHelper.class);
        attachInstance(new FluidSupportProviderImpl(), FluidSupportProvider.class);
        attachInstance(new SubsetsRegistryImpl(), SubsetsRegistry.class);
        attachInstance(new Internals.EntryStackProvider() {
            @Override
            public EntryStack empty() {
                return EmptyEntryStack.EMPTY;
            }
            
            @Override
            public EntryStack fluid(class_3611 fluid) {
                return new FluidEntryStack(fluid);
            }
            
            @Override
            public EntryStack fluid(class_3611 fluid, Fraction amount) {
                return new FluidEntryStack(fluid, amount);
            }
            
            @Override
            public EntryStack item(class_1799 stack) {
                return new ItemEntryStack(stack);
            }
        }, Internals.EntryStackProvider.class);
        attachInstance(new Internals.WidgetsProvider() {
            @Override
            public boolean isRenderingPanel(Panel panel) {
                return PanelWidget.isRendering(panel);
            }
            
            @Override
            public Widget createDrawableWidget(DrawableConsumer drawable) {
                return new DrawableWidget(drawable);
            }
            
            @Override
            public me.shedaniel.rei.api.widgets.Slot createSlot(Point point) {
                return EntryWidget.create(point);
            }
            
            @Override
            public Button createButton(Rectangle bounds, class_2561 text) {
                return new ButtonWidget(bounds, text);
            }
            
            @Override
            public Panel createPanelWidget(Rectangle bounds) {
                return new PanelWidget(bounds);
            }
            
            @Override
            public Label createLabel(Point point, class_5348 text) {
                return new LabelWidget(point, text);
            }
            
            @Override
            public Arrow createArrow(Rectangle rectangle) {
                return new ArrowWidget(rectangle);
            }
            
            @Override
            public BurningFire createBurningFire(Rectangle rectangle) {
                return new BurningFireWidget(rectangle);
            }
            
            @Override
            public DrawableConsumer createTexturedConsumer(class_2960 texture, int x, int y, int width, int height, float u, float v, int uWidth, int vHeight, int textureWidth, int textureHeight) {
                return new TexturedDrawableConsumer(texture, x, y, width, height, u, v, uWidth, vHeight, textureWidth, textureHeight);
            }
            
            @Override
            public DrawableConsumer createFillRectangleConsumer(Rectangle rectangle, int color) {
                return new FillRectangleDrawableConsumer(rectangle, color);
            }
        }, Internals.WidgetsProvider.class);
        attachInstance((BiFunction<@Nullable Point, Collection<class_2561>, Tooltip>) QueuedTooltip::create, "tooltipProvider");
    }
    
    /**
     * Registers a REI plugin
     *
     * @param plugin the plugin instance
     * @return the plugin itself
     */
    @ApiStatus.Internal
    public static REIPluginEntry registerPlugin(REIPluginEntry plugin) {
        plugins.put(plugin.getPluginIdentifier(), plugin);
        RoughlyEnoughItemsCore.LOGGER.debug("Registered plugin %s from %s", plugin.getPluginIdentifier().toString(), plugin.getClass().getSimpleName());
        return plugin;
    }
    
    public static List<REIPluginEntry> getPlugins() {
        return new LinkedList<>(plugins.values());
    }
    
    public static Optional<class_2960> getPluginIdentifier(REIPluginEntry plugin) {
        for (class_2960 identifier : plugins.keySet())
            if (identifier != null && plugins.get(identifier).equals(plugin))
                return Optional.of(identifier);
        return Optional.empty();
    }
    
    public static boolean hasPermissionToUsePackets() {
        try {
            class_310.method_1551().method_1562().method_2875().method_9259(0);
            return hasOperatorPermission() && canUsePackets();
        } catch (NullPointerException e) {
            return true;
        }
    }
    
    public static boolean hasOperatorPermission() {
        try {
            return class_310.method_1551().method_1562().method_2875().method_9259(1);
        } catch (NullPointerException e) {
            return true;
        }
    }
    
    public static boolean canUsePackets() {
        return ClientSidePacketRegistry.INSTANCE.canServerReceive(RoughlyEnoughItemsNetwork.CREATE_ITEMS_PACKET) && ClientSidePacketRegistry.INSTANCE.canServerReceive(RoughlyEnoughItemsNetwork.CREATE_ITEMS_GRAB_PACKET) && ClientSidePacketRegistry.INSTANCE.canServerReceive(RoughlyEnoughItemsNetwork.DELETE_ITEMS_PACKET);
    }
    
    @ApiStatus.Internal
    public static void syncRecipes(long[] lastSync) {
        if (lastSync != null) {
            if (lastSync[0] > 0 && System.currentTimeMillis() - lastSync[0] <= 5000) {
                RoughlyEnoughItemsCore.LOGGER.warn("Suppressing Sync Recipes!");
                return;
            }
            lastSync[0] = System.currentTimeMillis();
        }
        class_1863 recipeManager = class_310.method_1551().method_1562().method_2877();
        if (ConfigObject.getInstance().doesRegisterRecipesInAnotherThread()) {
            CompletableFuture.runAsync(() -> ((RecipeHelperImpl) RecipeHelper.getInstance()).tryRecipesLoaded(recipeManager), SYNC_RECIPES);
        } else {
            ((RecipeHelperImpl) RecipeHelper.getInstance()).tryRecipesLoaded(recipeManager);
        }
    }
    
    @ApiStatus.Internal
    public static boolean isDebugModeEnabled() {
        return System.getProperty("rei.test", "false").equals("true");
    }
    
    public static boolean canDeleteItems() {
        return hasPermissionToUsePackets() || class_310.method_1551().field_1761.method_2914();
    }
    
    @SuppressWarnings("deprecation")
    @Override
    public void onInitializeClient() {
        attachInstance(new ConfigManagerImpl(), ConfigManager.class);
        
        detectFabricLoader();
        registerClothEvents();
        discoverPluginEntries();
        for (ModContainer modContainer : FabricLoader.getInstance().getAllMods()) {
            if (modContainer.getMetadata().containsCustomElement("roughlyenoughitems:plugins"))
                RoughlyEnoughItemsCore.LOGGER.error("REI plugin from " + modContainer.getMetadata().getId() + " is not loaded because it is too old!");
        }
        
        boolean networkingLoaded = FabricLoader.getInstance().isModLoaded("fabric-networking-v0");
        if (!networkingLoaded) {
            RoughlyEnoughItemsState.error("Fabric API is not installed!", "https://www.curseforge.com/minecraft/mc-mods/fabric-api/files/all");
            return;
        }
        Executor.run(() -> () -> {
            ClientSidePacketRegistry.INSTANCE.register(RoughlyEnoughItemsNetwork.CREATE_ITEMS_MESSAGE_PACKET, (packetContext, packetByteBuf) -> {
                class_1799 stack = packetByteBuf.method_10819();
                String player = packetByteBuf.method_10800(32767);
                packetContext.getPlayer().method_7353(new class_2585(class_1074.method_4662("text.rei.cheat_items").replaceAll("\\{item_name}", EntryStack.create(stack.method_7972()).asFormattedText().getString()).replaceAll("\\{item_count}", stack.method_7972().method_7947() + "").replaceAll("\\{player_name}", player)), false);
            });
            ClientSidePacketRegistry.INSTANCE.register(RoughlyEnoughItemsNetwork.NOT_ENOUGH_ITEMS_PACKET, (packetContext, packetByteBuf) -> {
                class_437 currentScreen = class_310.method_1551().field_1755;
                if (currentScreen instanceof class_479) {
                    class_507 recipeBookGui = ((class_518) currentScreen).method_2659();
                    class_505 ghostSlots = recipeBookGui.field_3092;
                    ghostSlots.method_2571();
                    
                    List<List<class_1799>> input = Lists.newArrayList();
                    int mapSize = packetByteBuf.readInt();
                    for (int i = 0; i < mapSize; i++) {
                        List<class_1799> list = Lists.newArrayList();
                        int count = packetByteBuf.readInt();
                        for (int j = 0; j < count; j++) {
                            list.add(packetByteBuf.method_10819());
                        }
                        input.add(list);
                    }
                    
                    ghostSlots.method_2569(class_1856.method_8091(class_1802.field_20391), 381203812, 12738291);
                    class_1714 container = ((class_479) currentScreen).method_17577();
                    for (int i = 0; i < input.size(); i++) {
                        List<class_1799> stacks = input.get(i);
                        if (!stacks.isEmpty()) {
                            class_1735 slot = container.method_7611(i + container.method_7655() + 1);
                            ghostSlots.method_2569(class_1856.method_8101(stacks.toArray(new class_1799[0])), slot.field_7873, slot.field_7872);
                        }
                    }
                }
            });
        });
    }
    
    private void detectFabricLoader() {
        Executor.run(() -> () -> {
            try {
                FabricLoader instance = FabricLoader.getInstance();
                for (Field field : instance.getClass().getDeclaredFields()) {
                    if (Logger.class.isAssignableFrom(field.getType())) {
                        field.setAccessible(true);
                        Logger logger = (Logger) field.get(instance);
                        if (logger.getName().toLowerCase(Locale.ROOT).contains("subsystem")) {
                            File reiConfigFolder = new File(instance.getConfigDirectory(), "roughlyenoughitems");
                            File ignoreFile = new File(reiConfigFolder, ".ignoresubsystem");
                            if (!ignoreFile.exists()) {
                                RoughlyEnoughItemsState.warn("Subsystem is detected (probably though Aristois), please contact support from them if anything happens.");
                                RoughlyEnoughItemsState.onContinue(() -> {
                                    try {
                                        reiConfigFolder.mkdirs();
                                        ignoreFile.createNewFile();
                                    } catch (IOException e) {
                                        e.printStackTrace();
                                    }
                                });
                            }
                        }
                    }
                }
            } catch (Throwable ignored) {
            }
        });
    }
    
    private void discoverPluginEntries() {
        for (REIPluginEntry reiPlugin : FabricLoader.getInstance().getEntrypoints("rei_plugins", REIPluginEntry.class)) {
            try {
                if (!REIPluginV0.class.isAssignableFrom(reiPlugin.getClass()))
                    throw new IllegalArgumentException("REI plugin is too old!");
                registerPlugin(reiPlugin);
            } catch (Exception e) {
                e.printStackTrace();
                RoughlyEnoughItemsCore.LOGGER.error("Can't load REI plugins from %s: %s", reiPlugin.getClass(), e.getLocalizedMessage());
            }
        }
        for (REIPluginV0 reiPlugin : FabricLoader.getInstance().getEntrypoints("rei_plugins_v0", REIPluginV0.class)) {
            try {
                registerPlugin(reiPlugin);
            } catch (Exception e) {
                e.printStackTrace();
                RoughlyEnoughItemsCore.LOGGER.error("Can't load REI plugins from %s: %s", reiPlugin.getClass(), e.getLocalizedMessage());
            }
        }
        
        // Test Only
        loadTestPlugins();
    }
    
    private void loadTestPlugins() {
        if (isDebugModeEnabled()) {
            registerPlugin(new REITestPlugin());
        }
    }
    
    private boolean shouldReturn(class_437 screen) {
        if (screen == null) return true;
        return shouldReturn(screen.getClass());
    }
    
    private boolean shouldReturn(Class<?> screen) {
        try {
            for (OverlayDecider decider : DisplayHelper.getInstance().getAllOverlayDeciders()) {
                if (!decider.isHandingScreen(screen))
                    continue;
                class_1269 result = decider.shouldScreenBeOverlayed(screen);
                if (result != class_1269.field_5811)
                    return result == class_1269.field_5814 || REIHelper.getInstance().getPreviousContainerScreen() == null;
            }
        } catch (ConcurrentModificationException ignored) {
        }
        return true;
    }
    
    private void registerClothEvents() {
        final class_2960 recipeButtonTex = new class_2960("textures/gui/recipe_button.png");
        long[] lastSync = {-1};
        ClothClientHooks.SYNC_RECIPES.register((minecraftClient, recipeManager, synchronizeRecipesS2CPacket) -> syncRecipes(lastSync));
        ClothClientHooks.SCREEN_ADD_BUTTON.register((minecraftClient, screen, abstractButtonWidget) -> {
            if (ConfigObject.getInstance().doesDisableRecipeBook() && screen instanceof class_465 && abstractButtonWidget instanceof class_344)
                if (((class_344) abstractButtonWidget).field_2127.equals(recipeButtonTex))
                    return class_1269.field_5814;
            return class_1269.field_5811;
        });
        ClothClientHooks.SCREEN_INIT_POST.register((minecraftClient, screen, screenHooks) -> {
            if (shouldReturn(screen))
                return;
            if (screen instanceof class_490 && minecraftClient.field_1761.method_2914())
                return;
            if (screen instanceof class_465)
                ScreenHelper.setPreviousContainerScreen((class_465<?>) screen);
            boolean alreadyAdded = false;
            for (class_364 element : Lists.newArrayList(screenHooks.cloth$getChildren()))
                if (ContainerScreenOverlay.class.isAssignableFrom(element.getClass()))
                    if (alreadyAdded)
                        screenHooks.cloth$getChildren().remove(element);
                    else
                        alreadyAdded = true;
            if (!alreadyAdded)
                screenHooks.cloth$getChildren().add(ScreenHelper.getLastOverlay(true, false));
        });
        ClothClientHooks.SCREEN_RENDER_POST.register((matrices, minecraftClient, screen, i, i1, v) -> {
            if (shouldReturn(screen))
                return;
            ScreenHelper.getLastOverlay().method_25394(matrices, i, i1, v);
        });
        ClothClientHooks.SCREEN_MOUSE_DRAGGED.register((minecraftClient, screen, v, v1, i, v2, v3) -> {
            if (shouldReturn(screen))
                return class_1269.field_5811;
            if (ScreenHelper.isOverlayVisible() && ScreenHelper.getLastOverlay().method_25403(v, v1, i, v2, v3))
                return class_1269.field_5812;
            return class_1269.field_5811;
        });
        ClothClientHooks.SCREEN_MOUSE_CLICKED.register((minecraftClient, screen, v, v1, i) -> {
            isLeftModePressed = true;
            if (ScreenHelper.getOptionalOverlay().isPresent())
                if (screen instanceof class_481)
                    if (ScreenHelper.isOverlayVisible() && ScreenHelper.getLastOverlay().method_25402(v, v1, i)) {
                        screen.method_25395(ScreenHelper.getLastOverlay());
                        if (i == 0)
                            screen.method_25398(true);
                        return class_1269.field_5812;
                    }
            return class_1269.field_5811;
        });
        ClothClientHooks.SCREEN_MOUSE_RELEASED.register((minecraftClient, screen, v, v1, i) -> {
            isLeftModePressed = false;
            if (shouldReturn(screen))
                return class_1269.field_5811;
            if (ScreenHelper.getOptionalOverlay().isPresent())
                if (ScreenHelper.isOverlayVisible() && ScreenHelper.getLastOverlay().method_25406(v, v1, i)) {
                    return class_1269.field_5812;
                }
            return class_1269.field_5811;
        });
        ClothClientHooks.SCREEN_MOUSE_SCROLLED.register((minecraftClient, screen, v, v1, v2) -> {
            if (shouldReturn(screen))
                return class_1269.field_5811;
            if (ScreenHelper.isOverlayVisible() && ScreenHelper.getLastOverlay().method_25401(v, v1, v2))
                return class_1269.field_5812;
            return class_1269.field_5811;
        });
        ClothClientHooks.SCREEN_CHAR_TYPED.register((minecraftClient, screen, character, keyCode) -> {
            if (shouldReturn(screen))
                return class_1269.field_5811;
            if (ScreenHelper.getLastOverlay().method_25400(character, keyCode))
                return class_1269.field_5812;
            return class_1269.field_5811;
        });
        ClothClientHooks.SCREEN_LATE_RENDER.register((matrices, minecraftClient, screen, i, i1, v) -> {
            if (shouldReturn(screen))
                return;
            if (!ScreenHelper.isOverlayVisible())
                return;
            ScreenHelper.getLastOverlay().lateRender(matrices, i, i1, v);
        });
        ClothClientHooks.SCREEN_KEY_PRESSED.register((minecraftClient, screen, i, i1, i2) -> {
            if (shouldReturn(screen))
                return class_1269.field_5811;
            if (screen instanceof class_465 && ConfigObject.getInstance().doesDisableRecipeBook() && ConfigObject.getInstance().doesFixTabCloseContainer())
                if (i == 258 && minecraftClient.field_1690.field_1822.method_1417(i, i1)) {
                    minecraftClient.field_1724.method_7346();
                    return class_1269.field_5812;
                }
            if (screen.method_25399() != null && screen.method_25399() instanceof class_342 || (screen.method_25399() instanceof class_507 && ((class_507) screen.method_25399()).field_3089 != null && ((class_507) screen.method_25399()).field_3089.method_25370()))
                return class_1269.field_5811;
            if (ScreenHelper.getLastOverlay().method_25404(i, i1, i2))
                return class_1269.field_5812;
            return class_1269.field_5811;
        });
    }
    
}
