/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl.filtering.rules;

import com.google.common.collect.Lists;
import me.shedaniel.rei.api.EntryStack;
import me.shedaniel.rei.gui.config.entry.FilteringEntry;
import me.shedaniel.rei.gui.config.entry.FilteringRuleOptionsScreen;
import me.shedaniel.rei.impl.SearchArgument;
import me.shedaniel.rei.impl.filtering.AbstractFilteringRule;
import me.shedaniel.rei.impl.filtering.FilteringContext;
import me.shedaniel.rei.impl.filtering.FilteringResult;
import me.shedaniel.rei.utils.CollectionUtils;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_124;
import net.minecraft.class_2487;
import net.minecraft.class_2561;
import net.minecraft.class_2588;
import net.minecraft.class_437;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.BiFunction;
import java.util.function.Consumer;

@Environment(EnvType.CLIENT)
public class SearchFilteringRule extends AbstractFilteringRule<SearchFilteringRule> {
    private String filter;
    private List<SearchArgument.SearchArguments> arguments;
    private boolean show;
    
    public SearchFilteringRule() {
    }
    
    public SearchFilteringRule(String filter, List<SearchArgument.SearchArguments> arguments, boolean show) {
        this.filter = filter;
        this.arguments = arguments;
        this.show = show;
    }
    
    @Override
    public class_2487 toTag(class_2487 tag) {
        tag.method_10582("filter", filter);
        tag.method_10556("show", show);
        return tag;
    }
    
    @Override
    public SearchFilteringRule createFromTag(class_2487 tag) {
        String filter = tag.method_10558("filter");
        boolean show = tag.method_10577("show");
        return new SearchFilteringRule(filter, SearchArgument.processSearchTerm(filter), show);
    }
    
    @NotNull
    @Override
    public FilteringResult processFilteredStacks(@NotNull FilteringContext context) {
        List<CompletableFuture<List<EntryStack>>> completableFutures = Lists.newArrayList();
        processList(context.getUnsetStacks(), completableFutures);
        if (show) processList(context.getHiddenStacks(), completableFutures);
        else processList(context.getShownStacks(), completableFutures);
        try {
            CompletableFuture.allOf(completableFutures.toArray(new CompletableFuture[0])).get(10, TimeUnit.SECONDS);
        } catch (InterruptedException | ExecutionException | TimeoutException e) {
            e.printStackTrace();
        }
        FilteringResult result = FilteringResult.create();
        for (CompletableFuture<List<EntryStack>> future : completableFutures) {
            List<EntryStack> now = future.getNow(null);
            if (now != null) {
                if (show) {
                    result.show(now);
                } else {
                    result.hide(now);
                }
            }
        }
        return result;
    }
    
    @Override
    public SearchFilteringRule createNew() {
        return new SearchFilteringRule("", Collections.singletonList(SearchArgument.SearchArguments.ALWAYS), true);
    }
    
    private void processList(Collection<EntryStack> stacks, List<CompletableFuture<List<EntryStack>>> completableFutures) {
        for (Iterable<EntryStack> partitionStacks : CollectionUtils.partition((List<EntryStack>) stacks, 100)) {
            completableFutures.add(CompletableFuture.supplyAsync(() -> {
                List<EntryStack> output = Lists.newArrayList();
                for (EntryStack stack : partitionStacks) {
                    boolean shown = SearchArgument.canSearchTermsBeAppliedTo(stack, arguments);
                    if (shown) {
                        output.add(stack);
                    }
                }
                return output;
            }));
        }
    }
    
    @Override
    public class_2561 getTitle() {
        return new class_2588("rule.roughlyenoughitems.filtering.search");
    }
    
    @Override
    public class_2561 getSubtitle() {
        return new class_2588("rule.roughlyenoughitems.filtering.search.subtitle");
    }
    
    @Override
    public Optional<BiFunction<FilteringEntry, class_437, class_437>> createEntryScreen() {
        return Optional.of((entry, screen) -> new FilteringRuleOptionsScreen<SearchFilteringRule>(entry, this, screen) {
            TextFieldRuleEntry entry = null;
            BooleanRuleEntry show = null;
            
            @Override
            public void addEntries(Consumer<RuleEntry> entryConsumer) {
                addEmpty(entryConsumer, 10);
                addText(entryConsumer, new class_2588("rule.roughlyenoughitems.filtering.search.filter").method_27692(class_124.field_1080));
                entryConsumer.accept(entry = new TextFieldRuleEntry(field_22789 - 36, rule, widget -> {
                    widget.method_1880(9999);
                    if (entry != null) widget.method_1852(entry.getWidget().method_1882());
                    else widget.method_1852(rule.filter);
                }));
                addEmpty(entryConsumer, 10);
                addText(entryConsumer, new class_2588("rule.roughlyenoughitems.filtering.search.show").method_27692(class_124.field_1080));
                entryConsumer.accept(show = new BooleanRuleEntry(field_22789 - 36, show == null ? rule.show : show.getBoolean(), rule, bool -> {
                    return new class_2588("rule.roughlyenoughitems.filtering.search.show." + bool);
                }));
            }
            
            @Override
            public void save() {
                rule.filter = entry.getWidget().method_1882();
                rule.arguments = SearchArgument.processSearchTerm(rule.filter);
                rule.show = show.getBoolean();
            }
        });
    }
}
