/*
 * Roughly Enough Items by Danielshe.
 * Licensed under the MIT License.
 */

package me.shedaniel.rei.client;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import me.shedaniel.rei.RoughlyEnoughItemsCore;
import me.shedaniel.rei.api.*;
import net.minecraft.class_1799;
import net.minecraft.class_1860;
import net.minecraft.class_1863;
import net.minecraft.class_2960;
import java.awt.*;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

public class RecipeHelperImpl implements RecipeHelper {
    
    private static final Comparator<DisplayVisibilityHandler> VISIBILITY_HANDLER_COMPARATOR;
    private static final Comparator<class_1860> RECIPE_COMPARATOR = (o1, o2) -> {
        int int_1 = o1.method_8114().method_12836().compareTo(o2.method_8114().method_12836());
        if (int_1 == 0)
            int_1 = o1.method_8114().method_12832().compareTo(o2.method_8114().method_12832());
        return int_1;
    };
    
    static {
        Comparator<DisplayVisibilityHandler> comparator = Comparator.comparingDouble(DisplayVisibilityHandler::getPriority);
        VISIBILITY_HANDLER_COMPARATOR = comparator.reversed();
    }
    
    public final List<RecipeFunction> recipeFunctions = Lists.newArrayList();
    private final AtomicInteger recipeCount = new AtomicInteger();
    private final Map<class_2960, List<RecipeDisplay>> recipeCategoryListMap = Maps.newHashMap();
    private final Map<class_2960, DisplaySettings> categoryDisplaySettingsMap = Maps.newHashMap();
    private final List<RecipeCategory> categories = Lists.newArrayList();
    private final Map<class_2960, ButtonAreaSupplier> speedCraftAreaSupplierMap = Maps.newHashMap();
    private final Map<class_2960, List<SpeedCraftFunctional>> speedCraftFunctionalMap = Maps.newHashMap();
    private final Map<class_2960, List<List<class_1799>>> categoryWorkingStations = Maps.newHashMap();
    private final List<DisplayVisibilityHandler> displayVisibilityHandlers = Lists.newArrayList();
    private final List<LiveRecipeGenerator> liveRecipeGenerators = Lists.newArrayList();
    private class_1863 recipeManager;
    
    @Override
    public List<class_1799> findCraftableByItems(List<class_1799> inventoryItems) {
        List<class_1799> craftables = new ArrayList<>();
        for(List<RecipeDisplay> value : recipeCategoryListMap.values())
            for(RecipeDisplay recipeDisplay : value) {
                int slotsCraftable = 0;
                List<List<class_1799>> requiredInput = (List<List<class_1799>>) recipeDisplay.getRequiredItems();
                for(List<class_1799> slot : requiredInput) {
                    if (slot.isEmpty()) {
                        slotsCraftable++;
                        continue;
                    }
                    boolean slotDone = false;
                    for(class_1799 possibleType : inventoryItems) {
                        for(class_1799 slotPossible : slot)
                            if (class_1799.method_7984(slotPossible, possibleType)) {
                                slotsCraftable++;
                                slotDone = true;
                                break;
                            }
                        if (slotDone)
                            break;
                    }
                }
                if (slotsCraftable == recipeDisplay.getRequiredItems().size())
                    craftables.addAll((List<class_1799>) recipeDisplay.getOutput());
            }
        return craftables.stream().distinct().collect(Collectors.toList());
    }
    
    @Override
    public void registerCategory(RecipeCategory category) {
        categories.add(category);
        categoryDisplaySettingsMap.put(category.getIdentifier(), category.getDisplaySettings());
        recipeCategoryListMap.put(category.getIdentifier(), Lists.newLinkedList());
        categoryWorkingStations.put(category.getIdentifier(), Lists.newLinkedList());
    }
    
    @Override
    public void registerWorkingStations(class_2960 category, List<class_1799>... workingStations) {
        categoryWorkingStations.get(category).addAll(Arrays.asList(workingStations));
    }
    
    @Override
    public void registerWorkingStations(class_2960 category, class_1799... workingStations) {
        categoryWorkingStations.get(category).addAll(Arrays.asList(workingStations).stream().map(Collections::singletonList).collect(Collectors.toList()));
    }
    
    @Override
    public List<List<class_1799>> getWorkingStations(class_2960 category) {
        return categoryWorkingStations.get(category);
    }
    
    @Override
    public void registerDisplay(class_2960 categoryIdentifier, RecipeDisplay display) {
        if (!recipeCategoryListMap.containsKey(categoryIdentifier))
            return;
        recipeCount.incrementAndGet();
        recipeCategoryListMap.get(categoryIdentifier).add(display);
    }
    
    private void registerDisplay(class_2960 categoryIdentifier, RecipeDisplay display, int index) {
        if (!recipeCategoryListMap.containsKey(categoryIdentifier))
            return;
        recipeCount.incrementAndGet();
        recipeCategoryListMap.get(categoryIdentifier).add(index, display);
    }
    
    @Override
    public Map<RecipeCategory, List<RecipeDisplay>> getRecipesFor(class_1799 stack) {
        Map<class_2960, List<RecipeDisplay>> categoriesMap = new HashMap<>();
        categories.forEach(f -> categoriesMap.put(f.getIdentifier(), Lists.newArrayList()));
        for(Map.Entry<class_2960, List<RecipeDisplay>> entry : recipeCategoryListMap.entrySet()) {
            RecipeCategory category = getCategory(entry.getKey());
            for(RecipeDisplay recipeDisplay : entry.getValue())
                for(class_1799 outputStack : (List<class_1799>) recipeDisplay.getOutput())
                    if (category.checkTags() ? class_1799.method_7973(stack, outputStack) : class_1799.method_7984(stack, outputStack))
                        categoriesMap.get(recipeDisplay.getRecipeCategory()).add(recipeDisplay);
        }
        for(LiveRecipeGenerator liveRecipeGenerator : liveRecipeGenerators)
            ((Optional<List>) liveRecipeGenerator.getRecipeFor(stack)).ifPresent(o -> categoriesMap.get(liveRecipeGenerator.getCategoryIdentifier()).addAll(o));
        Map<RecipeCategory, List<RecipeDisplay>> recipeCategoryListMap = Maps.newLinkedHashMap();
        categories.forEach(category -> {
            if (categoriesMap.containsKey(category.getIdentifier()) && !categoriesMap.get(category.getIdentifier()).isEmpty())
                recipeCategoryListMap.put(category, categoriesMap.get(category.getIdentifier()).stream().filter(display -> isDisplayVisible(display)).collect(Collectors.toList()));
        });
        for(RecipeCategory category : Lists.newArrayList(recipeCategoryListMap.keySet()))
            if (recipeCategoryListMap.get(category).isEmpty())
                recipeCategoryListMap.remove(category);
        return recipeCategoryListMap;
    }
    
    private RecipeCategory getCategory(class_2960 identifier) {
        return categories.stream().filter(category -> category.getIdentifier().equals(identifier)).findFirst().orElse(null);
    }
    
    @Override
    public class_1863 getRecipeManager() {
        return recipeManager;
    }
    
    @Override
    public Map<RecipeCategory, List<RecipeDisplay>> getUsagesFor(class_1799 stack) {
        Map<class_2960, List<RecipeDisplay>> categoriesMap = new HashMap<>();
        categories.forEach(f -> categoriesMap.put(f.getIdentifier(), Lists.newArrayList()));
        for(Map.Entry<class_2960, List<RecipeDisplay>> entry : recipeCategoryListMap.entrySet()) {
            RecipeCategory category = getCategory(entry.getKey());
            for(RecipeDisplay recipeDisplay : entry.getValue()) {
                boolean found = false;
                for(List<class_1799> input : (List<List<class_1799>>) recipeDisplay.getInput()) {
                    for(class_1799 itemStack : input) {
                        if (category.checkTags() ? class_1799.method_7973(itemStack, stack) : class_1799.method_7984(itemStack, stack)) {
                            categoriesMap.get(recipeDisplay.getRecipeCategory()).add(recipeDisplay);
                            found = true;
                            break;
                        }
                    }
                    if (found)
                        break;
                }
            }
        }
        for(LiveRecipeGenerator liveRecipeGenerator : liveRecipeGenerators)
            ((Optional<List>) liveRecipeGenerator.getUsageFor(stack)).ifPresent(o -> categoriesMap.get(liveRecipeGenerator.getCategoryIdentifier()).addAll(o));
        Map<RecipeCategory, List<RecipeDisplay>> recipeCategoryListMap = Maps.newLinkedHashMap();
        categories.forEach(category -> {
            if (categoriesMap.containsKey(category.getIdentifier()) && !categoriesMap.get(category.getIdentifier()).isEmpty())
                recipeCategoryListMap.put(category, categoriesMap.get(category.getIdentifier()).stream().filter(display -> isDisplayVisible(display)).collect(Collectors.toList()));
        });
        for(RecipeCategory category : Lists.newArrayList(recipeCategoryListMap.keySet()))
            if (recipeCategoryListMap.get(category).isEmpty())
                recipeCategoryListMap.remove(category);
        return recipeCategoryListMap;
    }
    
    @Override
    public List<RecipeCategory> getAllCategories() {
        return new LinkedList<>(categories);
    }
    
    @Override
    public Optional<ButtonAreaSupplier> getSpeedCraftButtonArea(RecipeCategory category) {
        if (!speedCraftAreaSupplierMap.containsKey(category.getIdentifier()))
            return Optional.ofNullable(bounds -> new Rectangle((int) bounds.getMaxX() - 16, (int) bounds.getMaxY() - 16, 10, 10));
        return Optional.ofNullable(speedCraftAreaSupplierMap.get(category.getIdentifier()));
    }
    
    @Override
    public void registerSpeedCraftButtonArea(class_2960 category, ButtonAreaSupplier rectangle) {
        if (rectangle == null) {
            if (speedCraftAreaSupplierMap.containsKey(category))
                speedCraftAreaSupplierMap.remove(category);
        } else
            speedCraftAreaSupplierMap.put(category, rectangle);
    }
    
    @SuppressWarnings("deprecation")
    @Override
    public void registerDefaultSpeedCraftButtonArea(class_2960 category) {
        registerSpeedCraftButtonArea(category, bounds -> new Rectangle((int) bounds.getMaxX() - 16, (int) bounds.getMaxY() - 16, 10, 10));
    }
    
    @Override
    public List<SpeedCraftFunctional> getSpeedCraftFunctional(RecipeCategory category) {
        if (speedCraftFunctionalMap.get(category.getIdentifier()) == null)
            return Lists.newArrayList();
        return speedCraftFunctionalMap.get(category.getIdentifier());
    }
    
    @Override
    public void registerSpeedCraftFunctional(class_2960 category, SpeedCraftFunctional functional) {
        List<SpeedCraftFunctional> list = speedCraftFunctionalMap.containsKey(category) ? new LinkedList<>(speedCraftFunctionalMap.get(category)) : Lists.newLinkedList();
        list.add(functional);
        speedCraftFunctionalMap.put(category, list);
    }
    
    @SuppressWarnings("deprecation")
    public void recipesLoaded(class_1863 recipeManager) {
        this.recipeCount.set(0);
        this.recipeManager = recipeManager;
        this.recipeCategoryListMap.clear();
        this.categories.clear();
        this.speedCraftAreaSupplierMap.clear();
        this.categoryWorkingStations.clear();
        this.speedCraftFunctionalMap.clear();
        this.categoryDisplaySettingsMap.clear();
        this.recipeFunctions.clear();
        this.displayVisibilityHandlers.clear();
        this.liveRecipeGenerators.clear();
        ((DisplayHelperImpl) RoughlyEnoughItemsCore.getDisplayHelper()).resetCache();
        BaseBoundsHandler baseBoundsHandler = new BaseBoundsHandlerImpl();
        RoughlyEnoughItemsCore.getDisplayHelper().registerBoundsHandler(baseBoundsHandler);
        ((DisplayHelperImpl) RoughlyEnoughItemsCore.getDisplayHelper()).setBaseBoundsHandler(baseBoundsHandler);
        long startTime = System.currentTimeMillis();
        List<REIPluginEntry> plugins = new LinkedList<>(RoughlyEnoughItemsCore.getPlugins());
        plugins.sort((first, second) -> {
            return second.getPriority() - first.getPriority();
        });
        RoughlyEnoughItemsCore.LOGGER.info("[REI] Loading %d plugins: %s", plugins.size(), plugins.stream().map(REIPluginEntry::getPluginIdentifier).map(class_2960::toString).collect(Collectors.joining(", ")));
        Collections.reverse(plugins);
        RoughlyEnoughItemsCore.getItemRegisterer().getModifiableItemList().clear();
        PluginDisabler pluginDisabler = RoughlyEnoughItemsCore.getPluginDisabler();
        plugins.forEach(plugin -> {
            class_2960 identifier = plugin.getPluginIdentifier();
            try {
                if (pluginDisabler.isFunctionEnabled(identifier, PluginFunction.REGISTER_ITEMS))
                    plugin.registerItems(RoughlyEnoughItemsCore.getItemRegisterer());
                if (pluginDisabler.isFunctionEnabled(identifier, PluginFunction.REGISTER_CATEGORIES))
                    plugin.registerPluginCategories(this);
                if (pluginDisabler.isFunctionEnabled(identifier, PluginFunction.REGISTER_RECIPE_DISPLAYS))
                    plugin.registerRecipeDisplays(this);
                if (pluginDisabler.isFunctionEnabled(identifier, PluginFunction.REGISTER_BOUNDS))
                    plugin.registerBounds(RoughlyEnoughItemsCore.getDisplayHelper());
                if (pluginDisabler.isFunctionEnabled(identifier, PluginFunction.REGISTER_OTHERS))
                    plugin.registerOthers(this);
            } catch (Exception e) {
                RoughlyEnoughItemsCore.LOGGER.error("[REI] " + identifier.toString() + " plugin failed to load!", e);
            }
        });
        if (!recipeFunctions.isEmpty()) {
            List<class_1860> allSortedRecipes = getAllSortedRecipes();
            Collections.reverse(allSortedRecipes);
            recipeFunctions.forEach(recipeFunction -> {
                try {
                    allSortedRecipes.stream().filter(recipe -> recipeFunction.recipeFilter.test(recipe)).forEach(t -> registerDisplay(recipeFunction.category, (RecipeDisplay) recipeFunction.mappingFunction.apply(t), 0));
                } catch (Exception e) {
                    RoughlyEnoughItemsCore.LOGGER.error("[REI] Failed to add recipes!", e);
                }
            });
        }
        if (getDisplayVisibilityHandlers().isEmpty())
            registerRecipeVisibilityHandler(new DisplayVisibilityHandler() {
                @Override
                public DisplayVisibility handleDisplay(RecipeCategory category, RecipeDisplay display) {
                    return DisplayVisibility.ALWAYS_VISIBLE;
                }
                
                @Override
                public float getPriority() {
                    return -1f;
                }
            });
        // Clear Cache
        this.categoryDisplaySettingsMap.clear();
        categories.forEach(category -> categoryDisplaySettingsMap.put(category.getIdentifier(), category.getDisplaySettings()));
        ((DisplayHelperImpl) RoughlyEnoughItemsCore.getDisplayHelper()).resetCache();
        ScreenHelper.getOptionalOverlay().ifPresent(overlay -> overlay.shouldReInit = true);
        
        long usedTime = System.currentTimeMillis() - startTime;
        RoughlyEnoughItemsCore.LOGGER.info("[REI] Registered %d stack entries, %d recipes displays, %d bounds handler, %d visibility handlers and %d categories (%s) in %d ms.", RoughlyEnoughItemsCore.getItemRegisterer().getItemList().size(), recipeCount.get(), RoughlyEnoughItemsCore.getDisplayHelper().getAllBoundsHandlers().size(), getDisplayVisibilityHandlers().size(), categories.size(), String.join(", ", categories.stream().map(RecipeCategory::getCategoryName).collect(Collectors.toList())), usedTime);
    }
    
    @Override
    public int getRecipeCount() {
        return recipeCount.get();
    }
    
    @Override
    public List<class_1860> getAllSortedRecipes() {
        return getRecipeManager().method_8126().stream().sorted(RECIPE_COMPARATOR).collect(Collectors.toList());
    }
    
    @Override
    public Map<RecipeCategory, List<RecipeDisplay>> getAllRecipes() {
        Map<RecipeCategory, List<RecipeDisplay>> map = Maps.newLinkedHashMap();
        categories.forEach(recipeCategory -> {
            if (recipeCategoryListMap.containsKey(recipeCategory.getIdentifier())) {
                List<RecipeDisplay> list = recipeCategoryListMap.get(recipeCategory.getIdentifier()).stream().filter(display -> isDisplayVisible(display)).collect(Collectors.toList());
                if (!list.isEmpty())
                    map.put(recipeCategory, list);
            }
        });
        return map;
    }
    
    @Override
    public void registerRecipeVisibilityHandler(DisplayVisibilityHandler visibilityHandler) {
        displayVisibilityHandlers.add(visibilityHandler);
    }
    
    @Override
    public void unregisterRecipeVisibilityHandler(DisplayVisibilityHandler visibilityHandler) {
        displayVisibilityHandlers.remove(visibilityHandler);
    }
    
    @Override
    public List<DisplayVisibilityHandler> getDisplayVisibilityHandlers() {
        return Collections.unmodifiableList(displayVisibilityHandlers);
    }
    
    @SuppressWarnings("deprecation")
    @Override
    public boolean isDisplayVisible(RecipeDisplay display, boolean respectConfig) {
        return isDisplayVisible(display);
    }
    
    @SuppressWarnings("deprecation")
    @Override
    public boolean isDisplayVisible(RecipeDisplay display) {
        RecipeCategory category = getCategory(display.getRecipeCategory());
        List<DisplayVisibilityHandler> list = getDisplayVisibilityHandlers().stream().sorted(VISIBILITY_HANDLER_COMPARATOR).collect(Collectors.toList());
        for(DisplayVisibilityHandler displayVisibilityHandler : list) {
            try {
                DisplayVisibility visibility = displayVisibilityHandler.handleDisplay(category, display);
                if (visibility != DisplayVisibility.PASS)
                    return visibility == DisplayVisibility.ALWAYS_VISIBLE || visibility == DisplayVisibility.CONFIG_OPTIONAL;
            } catch (Throwable throwable) {
                RoughlyEnoughItemsCore.LOGGER.error("[REI] Failed to check if the recipe is visible!", throwable);
            }
        }
        return true;
    }
    
    @Override
    public <T extends class_1860<?>> void registerRecipes(class_2960 category, Class<T> recipeClass, Function<T, RecipeDisplay> mappingFunction) {
        recipeFunctions.add(new RecipeFunction(category, recipe -> recipeClass.isAssignableFrom(recipe.getClass()), mappingFunction));
    }
    
    @Override
    public <T extends class_1860<?>> void registerRecipes(class_2960 category, Function<class_1860, Boolean> recipeFilter, Function<T, RecipeDisplay> mappingFunction) {
        recipeFunctions.add(new RecipeFunction(category, recipeFilter::apply, mappingFunction));
    }
    
    @Override
    public <T extends class_1860<?>> void registerRecipes(class_2960 category, Predicate<class_1860> recipeFilter, Function<T, RecipeDisplay> mappingFunction) {
        recipeFunctions.add(new RecipeFunction(category, recipeFilter, mappingFunction));
    }
    
    @Override
    public Optional<DisplaySettings> getCachedCategorySettings(class_2960 category) {
        return categoryDisplaySettingsMap.entrySet().stream().filter(entry -> entry.getKey().equals(category)).map(Map.Entry::getValue).findAny();
    }
    
    @Override
    public void registerLiveRecipeGenerator(LiveRecipeGenerator liveRecipeGenerator) {
        liveRecipeGenerators.add(liveRecipeGenerator);
    }
    
    private class RecipeFunction {
        class_2960 category;
        Predicate<class_1860> recipeFilter;
        Function mappingFunction;
        
        public RecipeFunction(class_2960 category, Predicate<class_1860> recipeFilter, Function<?, RecipeDisplay> mappingFunction) {
            this.category = category;
            this.recipeFilter = recipeFilter;
            this.mappingFunction = mappingFunction;
        }
    }
    
}
