/*
 * Roughly Enough Items by Danielshe.
 * Licensed under the MIT License.
 */

package me.shedaniel.rei.api;

import me.shedaniel.rei.gui.renderers.EmptyRenderer;
import me.shedaniel.rei.gui.renderers.FluidRenderer;
import me.shedaniel.rei.gui.renderers.ItemStackRenderer;
import me.shedaniel.rei.gui.renderers.SimpleRecipeRenderer;
import me.shedaniel.rei.gui.widget.QueuedTooltip;
import net.minecraft.class_1799;
import net.minecraft.class_332;
import net.minecraft.class_3532;
import net.minecraft.class_3611;
import javax.annotation.Nullable;
import java.util.Collections;
import java.util.List;
import java.util.function.Function;
import java.util.function.Supplier;

public abstract class Renderer extends class_332 {
    /**
     * Gets an item stack renderer by an item stack supplier
     *
     * @param supplier the supplier for getting the item stack
     * @return the item stack renderer
     */
    public static ItemStackRenderer fromItemStackSupplier(Supplier<class_1799> supplier) {
        return fromItemStacks(() -> Collections.singletonList(supplier.get()), true, null);
    }
    
    /**
     * Gets an item stack renderer by an item stack supplier
     *
     * @param supplier the supplier for getting the item stack
     * @return the item stack renderer
     */
    public static ItemStackRenderer fromItemStackSupplierNoCounts(Supplier<class_1799> supplier) {
        return fromItemStacks(() -> Collections.singletonList(supplier.get()), false, null);
    }
    
    /**
     * Gets an item stack renderer by an item stack
     *
     * @param stack the item stack to be displayed
     * @return the item stack renderer
     */
    public static ItemStackRenderer fromItemStack(class_1799 stack) {
        return fromItemStacks(() -> Collections.singletonList(stack), true, null);
    }
    
    public static FluidRenderer fromFluid(class_3611 fluid) {
        return fromFluid(() -> fluid, null);
    }
    
    public static FluidRenderer fromFluid(Supplier<class_3611> fluidSupplier, @Nullable Function<class_3611, List<String>> extraTooltipSupplier) {
        return new FluidRenderer() {
            @Override
            public class_3611 getFluid() {
                return fluidSupplier.get();
            }
            
            @Override
            protected List<String> getExtraToolTips(class_3611 fluid) {
                if (extraTooltipSupplier == null)
                    return super.getExtraToolTips(fluid);
                List<String> apply = extraTooltipSupplier.apply(fluid);
                if (apply == null)
                    return super.getExtraToolTips(fluid);
                return apply;
            }
        };
    }
    
    /**
     * Gets an item stack renderer by an item stack
     *
     * @param stack the item stack to be displayed
     * @return the item stack renderer
     */
    public static ItemStackRenderer fromItemStackNoCounts(class_1799 stack) {
        return fromItemStacks(() -> Collections.singletonList(stack), false, null);
    }
    
    /**
     * Gets an empty renderer
     *
     * @return an empty renderer
     */
    public static EmptyRenderer empty() {
        return EmptyRenderer.INSTANCE;
    }
    
    /**
     * Gets a simple recipe renderer from inputs and outputs
     *
     * @param input  the list of input items
     * @param output the list of output items
     * @return the recipe renderer
     */
    public static SimpleRecipeRenderer fromRecipe(Supplier<List<List<class_1799>>> input, Supplier<List<class_1799>> output) {
        return new SimpleRecipeRenderer(input, output);
    }
    
    public static ItemStackRenderer fromItemStacks(List<class_1799> stacks) {
        return fromItemStacks(() -> stacks, true, null);
    }
    
    public static ItemStackRenderer fromItemStacks(Supplier<List<class_1799>> stacksSupplier, boolean renderCounts, @Nullable Function<class_1799, List<String>> extraTooltipSupplier) {
        return fromItemStacks(stacksSupplier, stack -> renderCounts ? null : "", extraTooltipSupplier);
    }
    
    public static ItemStackRenderer fromItemStacks(Supplier<List<class_1799>> stacksSupplier, Function<class_1799, String> countsFunction, @Nullable Function<class_1799, List<String>> extraTooltipSupplier) {
        return new ItemStackRenderer() {
            @Override
            public class_1799 getItemStack() {
                if (stacksSupplier.get().isEmpty())
                    return class_1799.field_8037;
                return stacksSupplier.get().get(class_3532.method_15357((System.currentTimeMillis() / 500 % (double) stacksSupplier.get().size()) / 1f));
            }
            
            @Override
            protected String getCounts() {
                return countsFunction.apply(getItemStack());
            }
            
            @Override
            protected List<String> getExtraToolTips(class_1799 stack) {
                if (extraTooltipSupplier == null)
                    return super.getExtraToolTips(stack);
                List<String> apply = extraTooltipSupplier.apply(stack);
                if (apply == null)
                    return super.getExtraToolTips(stack);
                return apply;
            }
        };
    }
    
    public static ItemStackRenderer fromItemStacksNoCounts(List<class_1799> stacks) {
        return fromItemStacks(() -> stacks, false, null);
    }
    
    /**
     * Gets the current blit offset
     *
     * @return the blit offset
     */
    public int getBlitOffset() {
        return this.blitOffset;
    }
    
    /**
     * Sets the current blit offset
     *
     * @param offset the new blit offset
     */
    public void setBlitOffset(int offset) {
        this.blitOffset = offset;
    }
    
    /**
     * Renders of the renderable
     *
     * @param x      the x coordinate of the renderable
     * @param y      the y coordinate of the renderable
     * @param mouseX the x coordinate of the mouse
     * @param mouseY the y coordinate of the mouse
     * @param delta  the delta
     */
    public abstract void render(int x, int y, double mouseX, double mouseY, float delta);
    
    @Nullable
    public QueuedTooltip getQueuedTooltip(float delta) {
        return null;
    }
    
}
