/*
 * Roughly Enough Items by Danielshe.
 * Licensed under the MIT License.
 */

package me.shedaniel.rei.api;

import me.shedaniel.math.api.Rectangle;
import me.shedaniel.rei.RoughlyEnoughItemsCore;
import net.minecraft.class_1860;
import net.minecraft.class_1863;
import net.minecraft.class_2960;
import net.minecraft.class_465;
import org.jetbrains.annotations.ApiStatus;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Predicate;

public interface RecipeHelper {
    
    @SuppressWarnings("deprecation")
    static RecipeHelper getInstance() {
        return RoughlyEnoughItemsCore.getRecipeHelper();
    }
    
    AutoTransferHandler registerAutoCraftingHandler(AutoTransferHandler handler);
    
    List<AutoTransferHandler> getSortedAutoCraftingHandler();
    
    /**
     * Gets the total recipe count registered
     *
     * @return the recipe count
     */
    int getRecipeCount();
    
    /**
     * @return a list of sorted recipes
     */
    List<class_1860> getAllSortedRecipes();
    
    /**
     * Gets all craftable items from materials.
     *
     * @param inventoryItems the materials
     * @return the list of craftable entries
     */
    List<EntryStack> findCraftableEntriesByItems(List<EntryStack> inventoryItems);
    
    /**
     * Registers a category
     *
     * @param category the category to register
     */
    void registerCategory(RecipeCategory<?> category);
    
    /**
     * Registers the working stations of a category
     *
     * @param category        the category
     * @param workingStations the working stations
     */
    void registerWorkingStations(class_2960 category, List<EntryStack>... workingStations);
    
    /**
     * Registers the working stations of a category
     *
     * @param category        the category
     * @param workingStations the working stations
     */
    void registerWorkingStations(class_2960 category, EntryStack... workingStations);
    
    List<List<EntryStack>> getWorkingStations(class_2960 category);
    
    /**
     * Registers a recipe display
     *
     * @param categoryIdentifier the category to display in
     * @param display            the recipe display
     */
    void registerDisplay(class_2960 categoryIdentifier, RecipeDisplay display);
    
    /**
     * Gets a map of recipes for an entry
     *
     * @param stack the stack to be crafted
     * @return the map of recipes
     */
    Map<RecipeCategory<?>, List<RecipeDisplay>> getRecipesFor(EntryStack stack);
    
    RecipeCategory<?> getCategory(class_2960 identifier);
    
    /**
     * Gets the vanilla recipe manager
     *
     * @return the recipe manager
     */
    class_1863 getRecipeManager();
    
    /**
     * Gets all registered categories
     *
     * @return the list of categories
     */
    List<RecipeCategory<?>> getAllCategories();
    
    /**
     * Gets a map of usages for an entry
     *
     * @param stack the stack to be used
     * @return the map of recipes
     */
    Map<RecipeCategory<?>, List<RecipeDisplay>> getUsagesFor(EntryStack stack);
    
    /**
     * Gets the optional of the auto crafting button area from a category
     *
     * @param category the category of the display
     * @return the optional of auto crafting button area
     */
    Optional<ButtonAreaSupplier> getAutoCraftButtonArea(RecipeCategory<?> category);
    
    /**
     * Registers a auto crafting button area
     *
     * @param category  the category of the button area
     * @param rectangle the button area
     */
    void registerAutoCraftButtonArea(class_2960 category, ButtonAreaSupplier rectangle);
    
    /**
     * Removes the auto crafting button
     *
     * @param category the category of the button
     */
    default void removeAutoCraftButton(class_2960 category) {
        registerAutoCraftButtonArea(category, bounds -> null);
    }
    
    /**
     * Gets the map of all recipes visible to the player
     *
     * @return the map of recipes
     */
    Map<RecipeCategory<?>, List<RecipeDisplay>> getAllRecipes();
    
    List<RecipeDisplay> getAllRecipesFromCategory(RecipeCategory<?> category);
    
    /**
     * Registers a recipe visibility handler
     *
     * @param visibilityHandler the handler to be registered
     */
    void registerRecipeVisibilityHandler(DisplayVisibilityHandler visibilityHandler);
    
    /**
     * Unregisters a recipe visibility handler
     *
     * @param visibilityHandler the handler to be unregistered
     */
    void unregisterRecipeVisibilityHandler(DisplayVisibilityHandler visibilityHandler);
    
    /**
     * Gets an unmodifiable list of recipe visibility handlers
     *
     * @return the unmodifiable list of handlers
     */
    List<DisplayVisibilityHandler> getDisplayVisibilityHandlers();
    
    /**
     * Checks if the display is visible by asking recipe visibility handlers
     *
     * @param display       the display to be checked
     * @param respectConfig whether it should respect the user's config
     * @return whether the display should be visible
     * @deprecated {@link RecipeHelper#isDisplayVisible(RecipeDisplay)} )}
     */
    @Deprecated
    default boolean isDisplayVisible(RecipeDisplay display, boolean respectConfig) {
        return isDisplayVisible(display);
    }
    
    boolean isDisplayNotVisible(RecipeDisplay display);
    
    /**
     * Checks if the display is visible by asking recipe visibility handlers
     *
     * @param display the display to be checked
     * @return whether the display should be visible
     */
    boolean isDisplayVisible(RecipeDisplay display);
    
    <T extends class_1860<?>> void registerRecipes(class_2960 category, Predicate<class_1860> recipeFilter, Function<T, RecipeDisplay> mappingFunction);
    
    /**
     * Registers a live recipe generator.
     *
     * @param liveRecipeGenerator the generator to register
     * @apiNote Still work in progress
     */
    void registerLiveRecipeGenerator(LiveRecipeGenerator<?> liveRecipeGenerator);
    
    void registerScreenClickArea(Rectangle rectangle, Class<? extends class_465<?>> screenClass, class_2960... categories);
    
    <T extends class_1860<?>> void registerRecipes(class_2960 category, Class<T> recipeClass, Function<T, RecipeDisplay> mappingFunction);
    
    <T extends class_1860<?>> void registerRecipes(class_2960 category, Function<class_1860, Boolean> recipeFilter, Function<T, RecipeDisplay> mappingFunction);
    
    List<RecipeHelper.ScreenClickArea> getScreenClickAreas();
    
    @ApiStatus.Internal
    boolean arePluginsLoading();
    
    interface ScreenClickArea {
        Class<? extends class_465> getScreenClass();
        
        Rectangle getRectangle();
        
        class_2960[] getCategories();
    }
    
}

