/*
 * Roughly Enough Items by Danielshe.
 * Licensed under the MIT License.
 */

package me.shedaniel.rei.utils;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import me.shedaniel.rei.api.EntryStack;
import org.jetbrains.annotations.ApiStatus;

import java.util.*;
import java.util.function.Function;
import java.util.function.Predicate;

@ApiStatus.Internal
public class CollectionUtils {
    public static <A, B> List<B> getOrPutEmptyList(Map<A, List<B>> map, A key) {
        List<B> b = map.get(key);
        if (b != null)
            return b;
        map.put(key, Lists.newArrayList());
        return map.get(key);
    }
    
    public static <T> T findFirstOrNullEquals(List<T> list, T obj) {
        for (T t : list) {
            if (t.equals(obj))
                return t;
        }
        return null;
    }
    
    public static <T> T findFirstOrNull(List<T> list, Predicate<T> predicate) {
        for (T t : list) {
            if (predicate.test(t))
                return t;
        }
        return null;
    }
    
    public static <T> boolean anyMatch(List<T> list, Predicate<T> predicate) {
        for (T t : list) {
            if (predicate.test(t))
                return true;
        }
        return false;
    }
    
    public static boolean anyMatchEqualsAll(List<EntryStack> list, EntryStack stack) {
        for (EntryStack t : list) {
            if (t.equalsAll(stack))
                return true;
        }
        return false;
    }
    
    public static EntryStack firstOrNullEqualsAll(List<EntryStack> list, EntryStack stack) {
        for (EntryStack t : list) {
            if (t.equalsAll(stack))
                return t;
        }
        return null;
    }
    
    public static <T> List<T> filter(List<T> list, Predicate<T> predicate) {
        List<T> l = Lists.newArrayList();
        for (T t : list) {
            if (predicate.test(t)) {
                l.add(t);
            }
        }
        return l;
    }
    
    public static <T> Set<T> filter(Set<T> list, Predicate<T> predicate) {
        Set<T> l = Sets.newLinkedHashSet();
        for (T t : list) {
            if (predicate.test(t)) {
                l.add(t);
            }
        }
        return l;
    }
    
    public static <T> List<T> filterSetToList(Set<T> list, Predicate<T> predicate) {
        List<T> l = Lists.newArrayList();
        for (T t : list) {
            if (predicate.test(t)) {
                l.add(t);
            }
        }
        return l;
    }
    
    public static <T, R> List<R> map(List<T> list, Function<T, R> function) {
        List<R> l = Lists.newArrayList();
        for (T t : list) {
            l.add(function.apply(t));
        }
        return l;
    }
    
    public static <T, R> List<R> map(T[] list, Function<T, R> function) {
        List<R> l = Lists.newArrayList();
        for (T t : list) {
            l.add(function.apply(t));
        }
        return l;
    }
    
    public static <T, R> Optional<R> mapAndMax(List<T> list, Function<T, R> function, Comparator<R> comparator) {
        if (list.isEmpty())
            return Optional.empty();
        List<R> copyOf = CollectionUtils.map(list, function);
        copyOf.sort(comparator);
        return Optional.ofNullable(copyOf.get(copyOf.size() - 1));
    }
    
    public static <T, R> Optional<R> mapAndMax(T[] list, Function<T, R> function, Comparator<R> comparator) {
        if (list.length <= 0)
            return Optional.empty();
        List<R> copyOf = CollectionUtils.map(list, function);
        copyOf.sort(comparator);
        return Optional.ofNullable(copyOf.get(copyOf.size() - 1));
    }
    
    public static <T> Optional<T> max(List<T> list, Comparator<T> comparator) {
        if (list.isEmpty())
            return Optional.empty();
        ArrayList<T> ts = new ArrayList<>(list);
        ts.sort(comparator);
        return Optional.ofNullable(ts.get(ts.size() - 1));
    }
    
    public static <T> Optional<T> max(T[] list, Comparator<T> comparator) {
        if (list.length <= 0)
            return Optional.empty();
        T[] copyOf = list.clone();
        Arrays.sort(copyOf, comparator);
        return Optional.ofNullable(copyOf[copyOf.length - 1]);
    }
    
    public static String joinToString(List<String> list, String separator) {
        StringJoiner joiner = new StringJoiner(separator);
        for (String t : list) {
            joiner.add(t);
        }
        return joiner.toString();
    }
    
    public static String joinToString(String[] list, String separator) {
        StringJoiner joiner = new StringJoiner(separator);
        for (String t : list) {
            joiner.add(t);
        }
        return joiner.toString();
    }
    
    public static <T> String mapAndJoinToString(List<T> list, Function<T, String> function, String separator) {
        StringJoiner joiner = new StringJoiner(separator);
        for (T t : list) {
            joiner.add(function.apply(t));
        }
        return joiner.toString();
    }
    
    public static <T> String mapAndJoinToString(T[] list, Function<T, String> function, String separator) {
        StringJoiner joiner = new StringJoiner(separator);
        for (T t : list) {
            joiner.add(function.apply(t));
        }
        return joiner.toString();
    }
    
    public static <T, R> List<R> filterAndMap(List<T> list, Predicate<T> predicate, Function<T, R> function) {
        List<R> l = null;
        for (T t : list) {
            if (predicate.test(t)) {
                if (l == null)
                    l = Lists.newArrayList();
                l.add(function.apply(t));
            }
        }
        return l == null ? Collections.emptyList() : l;
    }
    
    public static <T> int sumInt(List<T> list, Function<T, Integer> function) {
        int sum = 0;
        for (T t : list) {
            sum += function.apply(t);
        }
        return sum;
    }
    
    public static <T> int sumInt(List<Integer> list) {
        int sum = 0;
        for (int t : list) {
            sum += t;
        }
        return sum;
    }
    
    public static <T> double sumDouble(List<T> list, Function<T, Double> function) {
        double sum = 0;
        for (T t : list) {
            sum += function.apply(t);
        }
        return sum;
    }
    
    public static <T> double sumDouble(List<Double> list) {
        double sum = 0;
        for (double t : list) {
            sum += t;
        }
        return sum;
    }
}
