/*
 * Copyright (c) 2018, 2019, 2020 shedaniel
 * Licensed under the MIT License (the "License").
 */

package me.shedaniel.rei.impl;

import com.google.common.collect.Lists;
import me.shedaniel.rei.api.ClientHelper;
import me.shedaniel.rei.api.EntryStack;
import me.shedaniel.rei.gui.widget.QueuedTooltip;
import me.shedaniel.rei.utils.CollectionUtils;
import net.minecraft.class_1074;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1836;
import net.minecraft.class_2378;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3611;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.ApiStatus;

import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.function.Function;

@ApiStatus.Internal
public class SearchArgument {
    
    private static final String SPACE = " ", EMPTY = "";
    private static final SearchArgument ALWAYS = new SearchArgument(ArgumentType.ALWAYS, "", true);
    private static List<class_1792> searchBlacklisted = Lists.newArrayList();
    private ArgumentType argumentType;
    private String text;
    private final Function<String, Boolean> INCLUDE = s -> s.contains(text);
    private final Function<String, Boolean> NOT_INCLUDE = s -> !s.contains(text);
    private boolean include;
    
    public SearchArgument(ArgumentType argumentType, String text, boolean include) {
        this(argumentType, text, include, true);
    }
    
    public SearchArgument(ArgumentType argumentType, String text, boolean include, boolean autoLowerCase) {
        this.argumentType = argumentType;
        this.text = autoLowerCase ? text.toLowerCase(Locale.ROOT) : text;
        this.include = include;
    }
    
    @ApiStatus.Internal
    public static List<SearchArgument.SearchArguments> processSearchTerm(String searchTerm) {
        List<SearchArgument.SearchArguments> searchArguments = Lists.newArrayList();
        for (String split : StringUtils.splitByWholeSeparatorPreserveAllTokens(searchTerm.toLowerCase(Locale.ROOT), "|")) {
            String[] terms = StringUtils.split(split);
            if (terms.length == 0)
                searchArguments.add(SearchArgument.SearchArguments.ALWAYS);
            else {
                SearchArgument[] arguments = new SearchArgument[terms.length];
                for (int i = 0; i < terms.length; i++) {
                    String term = terms[i];
                    if (term.startsWith("-@") || term.startsWith("@-")) {
                        arguments[i] = new SearchArgument(SearchArgument.ArgumentType.MOD, term.substring(2), false);
                    } else if (term.startsWith("@")) {
                        arguments[i] = new SearchArgument(SearchArgument.ArgumentType.MOD, term.substring(1), true);
                    } else if (term.startsWith("-$") || term.startsWith("$-")) {
                        arguments[i] = new SearchArgument(SearchArgument.ArgumentType.TAG, term.substring(2), false);
                    } else if (term.startsWith("$")) {
                        arguments[i] = new SearchArgument(SearchArgument.ArgumentType.TAG, term.substring(1), true);
                    } else if (term.startsWith("-#") || term.startsWith("#-")) {
                        arguments[i] = new SearchArgument(SearchArgument.ArgumentType.TOOLTIP, term.substring(2), false);
                    } else if (term.startsWith("#")) {
                        arguments[i] = new SearchArgument(SearchArgument.ArgumentType.TOOLTIP, term.substring(1), true);
                    } else if (term.startsWith("-")) {
                        arguments[i] = new SearchArgument(SearchArgument.ArgumentType.TEXT, term.substring(1), false);
                    } else {
                        arguments[i] = new SearchArgument(SearchArgument.ArgumentType.TEXT, term, true);
                    }
                }
                searchArguments.add(new SearchArgument.SearchArguments(arguments));
            }
        }
        return searchArguments;
    }
    
    @ApiStatus.Internal
    public static boolean canSearchTermsBeAppliedTo(EntryStack stack, List<SearchArgument.SearchArguments> searchArguments) {
        if (searchArguments.isEmpty())
            return true;
        class_310 minecraft = class_310.method_1551();
        String mod = null;
        String modName = null;
        String name = null;
        String tooltip = null;
        String[] tags = null;
        for (SearchArgument.SearchArguments arguments : searchArguments) {
            boolean applicable = true;
            for (SearchArgument argument : arguments.getArguments()) {
                if (argument.getArgumentType() == SearchArgument.ArgumentType.ALWAYS)
                    return true;
                else if (argument.getArgumentType() == SearchArgument.ArgumentType.MOD) {
                    if (mod == null)
                        mod = stack.getIdentifier().map(class_2960::method_12836).orElse("").replace(SPACE, EMPTY).toLowerCase(Locale.ROOT);
                    if (mod != null && !mod.isEmpty()) {
                        if (argument.getFunction(!argument.isInclude()).apply(mod)) {
                            if (modName == null)
                                modName = ClientHelper.getInstance().getModFromModId(mod).replace(SPACE, EMPTY).toLowerCase(Locale.ROOT);
                            if (modName == null || modName.isEmpty() || argument.getFunction(!argument.isInclude()).apply(modName)) {
                                applicable = false;
                                break;
                            }
                            break;
                        }
                    }
                } else if (argument.getArgumentType() == SearchArgument.ArgumentType.TEXT) {
                    if (name == null)
                        name = SearchArgument.tryGetEntryStackName(stack).replace(SPACE, EMPTY).toLowerCase(Locale.ROOT);
                    if (name != null && !name.isEmpty() && argument.getFunction(!argument.isInclude()).apply(name)) {
                        applicable = false;
                        break;
                    }
                } else if (argument.getArgumentType() == SearchArgument.ArgumentType.TOOLTIP) {
                    if (tooltip == null)
                        tooltip = SearchArgument.tryGetEntryStackTooltip(stack).replace(SPACE, EMPTY).toLowerCase(Locale.ROOT);
                    if (tooltip != null && !tooltip.isEmpty() && argument.getFunction(!argument.isInclude()).apply(tooltip)) {
                        applicable = false;
                        break;
                    }
                } else if (argument.getArgumentType() == SearchArgument.ArgumentType.TAG) {
                    if (tags == null) {
                        if (stack.getType() == EntryStack.Type.ITEM) {
                            class_2960[] tagsFor = minecraft.method_1562().method_2867().method_15201().method_15191(stack.getItem()).toArray(new class_2960[0]);
                            tags = new String[tagsFor.length];
                            for (int i = 0; i < tagsFor.length; i++)
                                tags[i] = tagsFor[i].toString();
                        } else if (stack.getType() == EntryStack.Type.FLUID) {
                            class_2960[] tagsFor = minecraft.method_1562().method_2867().method_15205().method_15191(stack.getFluid()).toArray(new class_2960[0]);
                            tags = new String[tagsFor.length];
                            for (int i = 0; i < tagsFor.length; i++)
                                tags[i] = tagsFor[i].toString();
                        } else
                            tags = new String[0];
                    }
                    if (tags != null && tags.length > 0) {
                        boolean a = false;
                        for (String tag : tags)
                            if (argument.getFunction(argument.isInclude()).apply(tag))
                                a = true;
                        if (!a) {
                            applicable = false;
                            break;
                        }
                    } else {
                        applicable = false;
                        break;
                    }
                }
            }
            if (applicable)
                return true;
        }
        return false;
    }
    
    public static String tryGetEntryStackName(EntryStack stack) {
        if (stack.getType() == EntryStack.Type.ITEM)
            return tryGetItemStackName(stack.getItemStack());
        else if (stack.getType() == EntryStack.Type.FLUID)
            return tryGetFluidName(stack.getFluid());
        return "";
    }
    
    public static String tryGetEntryStackNameNoFormatting(EntryStack stack) {
        if (stack.getType() == EntryStack.Type.ITEM)
            return tryGetItemStackNameNoFormatting(stack.getItemStack());
        else if (stack.getType() == EntryStack.Type.FLUID)
            return tryGetFluidName(stack.getFluid());
        return "";
    }
    
    public static String tryGetEntryStackTooltip(EntryStack stack) {
        QueuedTooltip tooltip = stack.getTooltip(0, 0);
        if (tooltip != null)
            return CollectionUtils.joinToString(tooltip.getText(), "\n");
        return "";
    }
    
    public static String tryGetFluidName(class_3611 fluid) {
        class_2960 id = class_2378.field_11154.method_10221(fluid);
        if (class_1074.method_4663("block." + id.toString().replaceFirst(":", ".")))
            return class_1074.method_4662("block." + id.toString().replaceFirst(":", "."));
        return CollectionUtils.mapAndJoinToString(id.method_12832().split("_"), StringUtils::capitalize, " ");
    }
    
    public static List<String> tryGetItemStackToolTip(class_1799 itemStack, boolean careAboutAdvanced) {
        if (!searchBlacklisted.contains(itemStack.method_7909()))
            try {
                return CollectionUtils.map(itemStack.method_7950(class_310.method_1551().field_1724, class_310.method_1551().field_1690.field_1827 && careAboutAdvanced ? class_1836.class_1837.field_8935 : class_1836.class_1837.field_8934), class_2561::method_10863);
            } catch (Throwable e) {
                e.printStackTrace();
                searchBlacklisted.add(itemStack.method_7909());
            }
        return Collections.singletonList(tryGetItemStackName(itemStack));
    }
    
    public static String tryGetItemStackName(class_1799 stack) {
        if (!searchBlacklisted.contains(stack.method_7909()))
            try {
                return stack.method_7964().method_10863();
            } catch (Throwable e) {
                e.printStackTrace();
                searchBlacklisted.add(stack.method_7909());
            }
        try {
            return class_1074.method_4662("item." + class_2378.field_11142.method_10221(stack.method_7909()).toString().replace(":", "."));
        } catch (Throwable e) {
            e.printStackTrace();
        }
        return "ERROR";
    }
    
    public static String tryGetItemStackNameNoFormatting(class_1799 stack) {
        if (!searchBlacklisted.contains(stack.method_7909()))
            try {
                return stack.method_7964().method_10851();
            } catch (Throwable e) {
                e.printStackTrace();
                searchBlacklisted.add(stack.method_7909());
            }
        try {
            return class_1074.method_4662("item." + class_2378.field_11142.method_10221(stack.method_7909()).toString().replace(":", "."));
        } catch (Throwable e) {
            e.printStackTrace();
        }
        return "ERROR";
    }
    
    public Function<String, Boolean> getFunction(boolean include) {
        return include ? INCLUDE : NOT_INCLUDE;
    }
    
    public ArgumentType getArgumentType() {
        return argumentType;
    }
    
    public String getText() {
        return text;
    }
    
    public boolean isInclude() {
        return include;
    }
    
    @Override
    public String toString() {
        return String.format("Argument[%s]: name = %s, include = %b", argumentType.name(), text, include);
    }
    
    public enum ArgumentType {
        TEXT,
        MOD,
        TOOLTIP,
        TAG,
        ALWAYS
    }
    
    public static class SearchArguments {
        public static final SearchArguments ALWAYS = new SearchArguments(new SearchArgument[]{SearchArgument.ALWAYS});
        private SearchArgument[] arguments;
        
        public SearchArguments(SearchArgument[] arguments) {
            this.arguments = arguments;
        }
        
        public SearchArgument[] getArguments() {
            return arguments;
        }
    }
    
}
