/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl;

import com.google.common.collect.Lists;
import me.shedaniel.math.api.Rectangle;
import me.shedaniel.rei.api.ClientHelper;
import me.shedaniel.rei.api.ConfigObject;
import me.shedaniel.rei.api.EntryStack;
import me.shedaniel.rei.gui.widget.QueuedTooltip;
import net.fabricmc.fabric.api.client.render.fluid.v1.FluidRenderHandler;
import net.fabricmc.fabric.api.client.render.fluid.v1.FluidRenderHandlerRegistry;
import net.minecraft.class_1058;
import net.minecraft.class_1059;
import net.minecraft.class_2338;
import net.minecraft.class_2378;
import net.minecraft.class_287;
import net.minecraft.class_289;
import net.minecraft.class_290;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3545;
import net.minecraft.class_3611;
import net.minecraft.class_3612;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.*;

@ApiStatus.Internal
public class FluidEntryStack extends AbstractEntryStack {
    private static final Map<class_3611, class_3545<class_1058, Integer>> FLUID_SPRITE_CACHE = new HashMap<>();
    private static final int EMPTY_AMOUNT = -1319182373;
    private class_3611 fluid;
    private int amount;
    
    public FluidEntryStack(class_3611 fluid) {
        this(fluid, EMPTY_AMOUNT);
    }
    
    public FluidEntryStack(class_3611 fluid, int amount) {
        this.fluid = fluid;
        this.amount = amount;
    }
    
    protected static class_3545<class_1058, Integer> getOrLoadSprite(class_3611 fluid) {
        class_3545<class_1058, Integer> possibleCached = FLUID_SPRITE_CACHE.get(fluid);
        if (possibleCached != null)
            return possibleCached;
        
        FluidRenderHandler fluidRenderHandler = FluidRenderHandlerRegistry.INSTANCE.get(fluid);
        if (fluidRenderHandler == null)
            return null;
        class_1058[] sprites = fluidRenderHandler.getFluidSprites(class_310.method_1551().field_1687, class_310.method_1551().field_1687 == null ? null : class_2338.field_10980, fluid.method_15785());
        int color = -1;
        if (class_310.method_1551().field_1687 != null)
            color = fluidRenderHandler.getFluidColor(class_310.method_1551().field_1687, class_2338.field_10980, fluid.method_15785());
        class_3545<class_1058, Integer> pair = new class_3545<>(sprites[0], color);
        FLUID_SPRITE_CACHE.put(fluid, pair);
        return pair;
    }
    
    @Override
    public Optional<class_2960> getIdentifier() {
        return Optional.ofNullable(class_2378.field_11154.method_10221(getFluid()));
    }
    
    @Override
    public Type getType() {
        return Type.FLUID;
    }
    
    @Override
    public int getAmount() {
        return amount;
    }
    
    @Override
    public void setAmount(int amount) {
        this.amount = amount == EMPTY_AMOUNT ? EMPTY_AMOUNT : Math.max(amount, 0);
        if (isEmpty()) {
            fluid = class_3612.field_15906;
        }
    }
    
    @Override
    public boolean isEmpty() {
        return (amount != EMPTY_AMOUNT && amount <= 0) || fluid == class_3612.field_15906;
    }
    
    @Override
    public EntryStack copy() {
        EntryStack stack = EntryStack.create(fluid, amount);
        for (Map.Entry<Settings<?>, Object> entry : getSettings().entrySet()) {
            stack.setting((Settings<? super Object>) entry.getKey(), entry.getValue());
        }
        return stack;
    }
    
    @Override
    public Object getObject() {
        return fluid;
    }
    
    @Override
    public boolean equalsIgnoreTagsAndAmount(EntryStack stack) {
        if (stack.getType() != Type.FLUID)
            return false;
        return fluid == stack.getFluid();
    }
    
    @Override
    public boolean equalsIgnoreTags(EntryStack stack) {
        if (stack.getType() != Type.FLUID)
            return false;
        return fluid == stack.getFluid() && amount == stack.getAmount();
    }
    
    @Override
    public boolean equalsIgnoreAmount(EntryStack stack) {
        if (stack.getType() != Type.FLUID)
            return false;
        return fluid == stack.getFluid();
    }
    
    @Override
    public boolean equalsAll(EntryStack stack) {
        if (stack.getType() != Type.FLUID)
            return false;
        return fluid == stack.getFluid() && amount == stack.getAmount();
    }
    
    @Override
    public int hashOfAll() {
        int result = hashIgnoreAmountAndTags();
        result = 31 * result + amount;
        return result;
    }
    
    @Override
    public int hashIgnoreAmountAndTags() {
        int result = 1;
        result = 31 * result + getType().ordinal();
        result = 31 * result + fluid.hashCode();
        return result;
    }
    
    @Override
    public int hashIgnoreTags() {
        return hashOfAll();
    }
    
    @Override
    public int hashIgnoreAmount() {
        return hashIgnoreAmountAndTags();
    }
    
    @Nullable
    @Override
    public QueuedTooltip getTooltip(int mouseX, int mouseY) {
        if (!get(Settings.TOOLTIP_ENABLED).get() || isEmpty())
            return null;
        List<String> toolTip = Lists.newArrayList(SearchArgument.tryGetEntryStackName(this));
        if (amount >= 0) {
            String amountTooltip = get(Settings.Fluid.AMOUNT_TOOLTIP).apply(this);
            if (amountTooltip != null)
                toolTip.addAll(Arrays.asList(amountTooltip.split("\n")));
        }
        toolTip.addAll(get(Settings.TOOLTIP_APPEND_EXTRA).apply(this));
        if (get(Settings.TOOLTIP_APPEND_MOD).get() && ConfigObject.getInstance().shouldAppendModNames()) {
            final String modString = ClientHelper.getInstance().getFormattedModFromIdentifier(class_2378.field_11154.method_10221(fluid));
            boolean alreadyHasMod = false;
            for (String s : toolTip)
                if (s.equalsIgnoreCase(modString)) {
                    alreadyHasMod = true;
                    break;
                }
            if (!alreadyHasMod)
                toolTip.add(modString);
        }
        return QueuedTooltip.create(toolTip);
    }
    
    @SuppressWarnings("deprecation")
    @Override
    public void render(Rectangle bounds, int mouseX, int mouseY, float delta) {
        if (get(Settings.RENDER).get()) {
            class_3545<class_1058, Integer> pair = getOrLoadSprite(getFluid());
            if (pair != null) {
                class_1058 sprite = pair.method_15442();
                int color = pair.method_15441();
                int a = 255;
                int r = (color >> 16 & 255);
                int g = (color >> 8 & 255);
                int b = (color & 255);
                class_310.method_1551().method_1531().method_22813(class_1059.field_5275);
                class_289 tess = class_289.method_1348();
                class_287 bb = tess.method_1349();
                bb.method_1328(7, class_290.field_1575);
                bb.method_22912(bounds.getMaxX(), bounds.y, getZ()).method_22913(sprite.method_4577(), sprite.method_4593()).method_1336(r, g, b, a).method_1344();
                bb.method_22912(bounds.x, bounds.y, getZ()).method_22913(sprite.method_4594(), sprite.method_4593()).method_1336(r, g, b, a).method_1344();
                bb.method_22912(bounds.x, bounds.getMaxY(), getZ()).method_22913(sprite.method_4594(), sprite.method_4575()).method_1336(r, g, b, a).method_1344();
                bb.method_22912(bounds.getMaxX(), bounds.getMaxY(), getZ()).method_22913(sprite.method_4577(), sprite.method_4575()).method_1336(r, g, b, a).method_1344();
                tess.method_1350();
            }
        }
    }
}
