/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.plugin.beacon;

import com.google.common.collect.Lists;
import com.mojang.blaze3d.systems.RenderSystem;
import me.shedaniel.clothconfig2.ClothConfigInitializer;
import me.shedaniel.clothconfig2.api.ScissorsHandler;
import me.shedaniel.clothconfig2.gui.widget.DynamicEntryListWidget;
import me.shedaniel.math.api.Rectangle;
import me.shedaniel.math.impl.PointHelper;
import me.shedaniel.rei.api.EntryStack;
import me.shedaniel.rei.api.RecipeCategory;
import me.shedaniel.rei.gui.entries.RecipeEntry;
import me.shedaniel.rei.gui.widget.*;
import me.shedaniel.rei.plugin.DefaultPlugin;
import me.shedaniel.rei.utils.CollectionUtils;
import net.minecraft.class_1074;
import net.minecraft.class_2246;
import net.minecraft.class_287;
import net.minecraft.class_289;
import net.minecraft.class_290;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3532;
import net.minecraft.class_364;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.function.Supplier;

public class DefaultBeaconBaseCategory implements RecipeCategory<DefaultBeaconBaseDisplay> {
    @Override
    public class_2960 getIdentifier() {
        return DefaultPlugin.BEACON;
    }
    
    @Override
    public String getCategoryName() {
        return class_1074.method_4662("category.rei.beacon_base");
    }
    
    @Override
    public EntryStack getLogo() {
        return EntryStack.create(class_2246.field_10327);
    }
    
    @Override
    public RecipeEntry getSimpleRenderer(DefaultBeaconBaseDisplay recipe) {
        String name = getCategoryName();
        return new RecipeEntry() {
            @Override
            public int getHeight() {
                return 10 + class_310.method_1551().field_1772.field_2000;
            }
            
            @Nullable
            @Override
            public QueuedTooltip getTooltip(int mouseX, int mouseY) {
                return null;
            }
            
            @Override
            public void render(Rectangle rectangle, int mouseX, int mouseY, float delta) {
                class_310.method_1551().field_1772.method_1729(name, rectangle.x + 5, rectangle.y + 6, -1);
            }
        };
    }
    
    @Override
    public List<Widget> setupDisplay(Supplier<DefaultBeaconBaseDisplay> recipeDisplaySupplier, Rectangle bounds) {
        DefaultBeaconBaseDisplay display = recipeDisplaySupplier.get();
        List<Widget> widgets = Lists.newArrayList();
        widgets.add(EntryWidget.create(bounds.getCenterX() - 8, bounds.y + 3).entry(getLogo()));
        Rectangle rectangle = new Rectangle(bounds.getCenterX() - (bounds.width / 2) - 1, bounds.y + 23, bounds.width + 2, bounds.height - 28);
        widgets.add(new SlotBaseWidget(rectangle));
        widgets.add(new ScrollableSlotsWidget(rectangle, CollectionUtils.map(display.getEntries(), t -> EntryWidget.create(0, 0).noBackground().entry(t))));
        return widgets;
    }
    
    @Override
    public int getDisplayHeight() {
        return 140;
    }
    
    @Override
    public int getFixedRecipesPerPage() {
        return 1;
    }
    
    private static class ScrollableSlotsWidget extends WidgetWithBounds {
        private Rectangle bounds;
        private List<EntryWidget> widgets;
        private double target;
        private double scroll;
        private long start;
        private long duration;
        
        public ScrollableSlotsWidget(Rectangle bounds, List<EntryWidget> widgets) {
            this.bounds = bounds;
            this.widgets = Lists.newArrayList(widgets);
        }
        
        @Override
        public boolean mouseScrolled(double double_1, double double_2, double double_3) {
            if (containsMouse(double_1, double_2)) {
                offset(ClothConfigInitializer.getScrollStep() * -double_3, true);
                return true;
            }
            return false;
        }
        
        public void offset(double value, boolean animated) {
            scrollTo(target + value, animated);
        }
        
        public void scrollTo(double value, boolean animated) {
            scrollTo(value, animated, ClothConfigInitializer.getScrollDuration());
        }
        
        public void scrollTo(double value, boolean animated, long duration) {
            target = clamp(value);
            
            if (animated) {
                start = System.currentTimeMillis();
                this.duration = duration;
            } else
                scroll = target;
        }
        
        public final double clamp(double v) {
            return clamp(v, DynamicEntryListWidget.SmoothScrollingSettings.CLAMP_EXTENSION);
        }
        
        public final double clamp(double v, double clampExtension) {
            return class_3532.method_15350(v, -clampExtension, getMaxScroll() + clampExtension);
        }
        
        protected int getMaxScroll() {
            return Math.max(0, this.getMaxScrollPosition() - this.getBounds().height + 1);
        }
        
        protected int getMaxScrollPosition() {
            return class_3532.method_15386(widgets.size() / 8f) * 18;
        }
        
        @Override
        public Rectangle getBounds() {
            return bounds;
        }
        
        @Override
        public void render(int mouseX, int mouseY, float delta) {
            updatePosition(delta);
            Rectangle innerBounds = new Rectangle(bounds.x + 1, bounds.y + 1, bounds.width - 7, bounds.height - 2);
            ScissorsHandler.INSTANCE.scissor(innerBounds);
            for (int y = 0; y < class_3532.method_15386(widgets.size() / 8f); y++) {
                for (int x = 0; x < 8; x++) {
                    int index = y * 8 + x;
                    if (widgets.size() <= index)
                        break;
                    EntryWidget widget = widgets.get(index);
                    widget.getBounds().setLocation(bounds.x + 1 + x * 18, (int) (bounds.y + 1 + y * 18 - scroll));
                    widget.render(mouseX, mouseY, delta);
                }
            }
            ScissorsHandler.INSTANCE.removeLastScissor();
            ScissorsHandler.INSTANCE.scissor(bounds);
            RenderSystem.enableBlend();
            RenderSystem.blendFuncSeparate(770, 771, 0, 1);
            RenderSystem.disableAlphaTest();
            RenderSystem.shadeModel(7425);
            RenderSystem.disableTexture();
            renderScrollBar();
            RenderSystem.enableTexture();
            RenderSystem.shadeModel(7424);
            RenderSystem.enableAlphaTest();
            RenderSystem.disableBlend();
            ScissorsHandler.INSTANCE.removeLastScissor();
        }
        
        @SuppressWarnings("deprecation")
        private void renderScrollBar() {
            int maxScroll = getMaxScroll();
            int scrollbarPositionMinX = getBounds().getMaxX() - 7;
            int scrollbarPositionMaxX = scrollbarPositionMinX + 6;
            class_289 tessellator = class_289.method_1348();
            class_287 buffer = tessellator.method_1349();
            if (maxScroll > 0) {
                int height = (int) (((this.getBounds().height - 2f) * (this.getBounds().height - 2f)) / this.getMaxScrollPosition());
                height = class_3532.method_15340(height, 32, this.getBounds().height - 2);
                height -= Math.min((scroll < 0 ? (int) -scroll : scroll > maxScroll ? (int) scroll - maxScroll : 0), height * .95);
                height = Math.max(10, height);
                int minY = Math.min(Math.max((int) scroll * (this.getBounds().height - 2 - height) / maxScroll + getBounds().y + 1, getBounds().y + 1), getBounds().getMaxY() - 1 - height);
                
                boolean hovered = new Rectangle(scrollbarPositionMinX, minY, scrollbarPositionMaxX - scrollbarPositionMinX, height).contains(PointHelper.fromMouse());
                int bottomC = hovered ? 168 : 128;
                int topC = hovered ? 222 : 172;
                
                // Black Bar
                buffer.method_1328(7, class_290.field_1575);
                buffer.method_22912(scrollbarPositionMinX, this.getBounds().y + 1, 0.0D).method_22913(0, 1).method_1336(0, 0, 0, 255).method_1344();
                buffer.method_22912(scrollbarPositionMaxX, this.getBounds().y + 1, 0.0D).method_22913(1, 1).method_1336(0, 0, 0, 255).method_1344();
                buffer.method_22912(scrollbarPositionMaxX, getBounds().getMaxY() - 1, 0.0D).method_22913(1, 0).method_1336(0, 0, 0, 255).method_1344();
                buffer.method_22912(scrollbarPositionMinX, getBounds().getMaxY() - 1, 0.0D).method_22913(0, 0).method_1336(0, 0, 0, 255).method_1344();
                tessellator.method_1350();
                
                // Bottom
                buffer.method_1328(7, class_290.field_1575);
                buffer.method_22912(scrollbarPositionMinX, minY + height, 0.0D).method_22913(0, 1).method_1336(bottomC, bottomC, bottomC, 255).method_1344();
                buffer.method_22912(scrollbarPositionMaxX, minY + height, 0.0D).method_22913(1, 1).method_1336(bottomC, bottomC, bottomC, 255).method_1344();
                buffer.method_22912(scrollbarPositionMaxX, minY, 0.0D).method_22913(1, 0).method_1336(bottomC, bottomC, bottomC, 255).method_1344();
                buffer.method_22912(scrollbarPositionMinX, minY, 0.0D).method_22913(0, 0).method_1336(bottomC, bottomC, bottomC, 255).method_1344();
                tessellator.method_1350();
                
                // Top
                buffer.method_1328(7, class_290.field_1575);
                buffer.method_22912(scrollbarPositionMinX, (minY + height - 1), 0.0D).method_22913(0, 1).method_1336(topC, topC, topC, 255).method_1344();
                buffer.method_22912((scrollbarPositionMaxX - 1), (minY + height - 1), 0.0D).method_22913(1, 1).method_1336(topC, topC, topC, 255).method_1344();
                buffer.method_22912((scrollbarPositionMaxX - 1), minY, 0.0D).method_22913(1, 0).method_1336(topC, topC, topC, 255).method_1344();
                buffer.method_22912(scrollbarPositionMinX, minY, 0.0D).method_22913(0, 0).method_1336(topC, topC, topC, 255).method_1344();
                tessellator.method_1350();
            }
        }
        
        private void updatePosition(float delta) {
            double[] target = new double[]{this.target};
            this.scroll = ClothConfigInitializer.handleScrollingPosition(target, this.scroll, this.getMaxScroll(), delta, this.start, this.duration);
            this.target = target[0];
        }
        
        @Override
        public List<? extends class_364> children() {
            return widgets;
        }
    }
}
