/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl;

import com.google.common.collect.Lists;
import me.shedaniel.math.Point;
import me.shedaniel.math.Rectangle;
import me.shedaniel.rei.api.ClientHelper;
import me.shedaniel.rei.api.ConfigObject;
import me.shedaniel.rei.api.EntryStack;
import me.shedaniel.rei.api.widgets.Tooltip;
import me.shedaniel.rei.utils.CollectionUtils;
import net.fabricmc.fabric.api.client.render.fluid.v1.FluidRenderHandler;
import net.fabricmc.fabric.api.client.render.fluid.v1.FluidRenderHandlerRegistry;
import net.minecraft.class_1058;
import net.minecraft.class_1059;
import net.minecraft.class_1074;
import net.minecraft.class_1159;
import net.minecraft.class_2378;
import net.minecraft.class_2561;
import net.minecraft.class_2585;
import net.minecraft.class_2588;
import net.minecraft.class_287;
import net.minecraft.class_289;
import net.minecraft.class_290;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3611;
import net.minecraft.class_3612;
import net.minecraft.class_4587;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@ApiStatus.Internal
public class FluidEntryStack extends AbstractEntryStack {
    private static final double IGNORE_AMOUNT = -1319182373;
    private class_3611 fluid;
    private double amount;
    
    public FluidEntryStack(class_3611 fluid) {
        this(fluid, IGNORE_AMOUNT);
    }
    
    public FluidEntryStack(class_3611 fluid, int amount) {
        this(fluid, (double) amount);
    }
    
    public FluidEntryStack(class_3611 fluid, double amount) {
        this.fluid = fluid;
        this.amount = amount;
    }
    
    @Override
    public Optional<class_2960> getIdentifier() {
        return Optional.ofNullable(class_2378.field_11154.method_10221(getFluid()));
    }
    
    @Override
    public Type getType() {
        return Type.FLUID;
    }
    
    @Override
    public double getFloatingAmount() {
        return amount;
    }
    
    @Override
    public void setFloatingAmount(double amount) {
        this.amount = amount == IGNORE_AMOUNT ? IGNORE_AMOUNT : Math.max(amount, 0);
        if (isEmpty()) {
            fluid = class_3612.field_15906;
        }
    }
    
    @Override
    public boolean isEmpty() {
        return (amount != IGNORE_AMOUNT && amount <= 0) || fluid == class_3612.field_15906;
    }
    
    @Override
    public EntryStack copy() {
        EntryStack stack = EntryStack.create(fluid, amount);
        for (Map.Entry<Settings<?>, Object> entry : getSettings().entrySet()) {
            stack.setting((Settings<? super Object>) entry.getKey(), entry.getValue());
        }
        return stack;
    }
    
    @Override
    public Object getObject() {
        return fluid;
    }
    
    @Override
    public boolean equalsIgnoreTagsAndAmount(EntryStack stack) {
        if (stack.getType() == Type.ITEM)
            return equalsIgnoreTagsAndAmount(EntryStack.copyItemToFluid(stack));
        if (stack.getType() != Type.FLUID)
            return false;
        return fluid == stack.getFluid();
    }
    
    @Override
    public boolean equalsIgnoreTags(EntryStack stack) {
        if (stack.getType() == Type.ITEM)
            return equalsIgnoreTags(EntryStack.copyItemToFluid(stack));
        if (stack.getType() != Type.FLUID)
            return false;
        return fluid == stack.getFluid() && (amount == IGNORE_AMOUNT || stack.getAmount() == IGNORE_AMOUNT || amount == stack.getAmount());
    }
    
    @Override
    public boolean equalsIgnoreAmount(EntryStack stack) {
        if (stack.getType() == Type.ITEM)
            return equalsIgnoreAmount(EntryStack.copyItemToFluid(stack));
        if (stack.getType() != Type.FLUID)
            return false;
        return fluid == stack.getFluid();
    }
    
    @Override
    public boolean equalsAll(EntryStack stack) {
        if (stack.getType() != Type.FLUID)
            return false;
        return fluid == stack.getFluid() && (amount == IGNORE_AMOUNT || stack.getAmount() == IGNORE_AMOUNT || amount == stack.getAmount());
    }
    
    @Override
    public int hashOfAll() {
        int result = hashIgnoreAmountAndTags();
        result = 31 * result + Double.hashCode(amount);
        return result;
    }
    
    @Override
    public int hashIgnoreAmountAndTags() {
        int result = 1;
        result = 31 * result + getType().ordinal();
        result = 31 * result + fluid.hashCode();
        return result;
    }
    
    @Override
    public int hashIgnoreTags() {
        return hashOfAll();
    }
    
    @Override
    public int hashIgnoreAmount() {
        return hashIgnoreAmountAndTags();
    }
    
    @Nullable
    @Override
    public Tooltip getTooltip(Point point) {
        if (!get(Settings.TOOLTIP_ENABLED).get() || isEmpty())
            return null;
        List<class_2561> toolTip = Lists.newArrayList(asFormattedText());
        if (amount >= 0) {
            String amountTooltip = get(Settings.Fluid.AMOUNT_TOOLTIP).apply(this);
            if (amountTooltip != null)
                toolTip.addAll(Stream.of(amountTooltip.split("\n")).map(class_2585::new).collect(Collectors.toList()));
        }
        toolTip.addAll(get(Settings.TOOLTIP_APPEND_EXTRA).apply(this));
        if (get(Settings.TOOLTIP_APPEND_MOD).get() && ConfigObject.getInstance().shouldAppendModNames()) {
            class_2960 id = class_2378.field_11154.method_10221(fluid);
            final class_2561 modString = ClientHelper.getInstance().getFormattedModFromIdentifier(id);
            final String modId = ClientHelper.getInstance().getModFromIdentifier(id);
            boolean alreadyHasMod = false;
            for (class_2561 s : toolTip)
                if (s.method_10851().equalsIgnoreCase(modId)) {
                    alreadyHasMod = true;
                    break;
                }
            if (!alreadyHasMod)
                toolTip.add(modString);
        }
        return Tooltip.create(toolTip);
    }
    
    @SuppressWarnings("deprecation")
    @Override
    public void render(class_4587 matrices, Rectangle bounds, int mouseX, int mouseY, float delta) {
        if (get(Settings.RENDER).get()) {
            SimpleFluidRenderer.FluidRenderingData renderingData = SimpleFluidRenderer.fromFluid(fluid);
            if (renderingData != null) {
                class_1058 sprite = renderingData.getSprite();
                int color = renderingData.getColor();
                int a = 255;
                int r = (color >> 16 & 255);
                int g = (color >> 8 & 255);
                int b = (color & 255);
                class_310.method_1551().method_1531().method_22813(class_1059.field_5275);
                class_289 tess = class_289.method_1348();
                class_287 bb = tess.method_1349();
                class_1159 matrix = matrices.method_23760().method_23761();
                bb.method_1328(7, class_290.field_1575);
                bb.method_22918(matrix, bounds.getMaxX(), bounds.y, getZ()).method_22913(sprite.method_4577(), sprite.method_4593()).method_1336(r, g, b, a).method_1344();
                bb.method_22918(matrix, bounds.x, bounds.y, getZ()).method_22913(sprite.method_4594(), sprite.method_4593()).method_1336(r, g, b, a).method_1344();
                bb.method_22918(matrix, bounds.x, bounds.getMaxY(), getZ()).method_22913(sprite.method_4594(), sprite.method_4575()).method_1336(r, g, b, a).method_1344();
                bb.method_22918(matrix, bounds.getMaxX(), bounds.getMaxY(), getZ()).method_22913(sprite.method_4577(), sprite.method_4575()).method_1336(r, g, b, a).method_1344();
                tess.method_1350();
            }
        }
    }
    
    @NotNull
    @Override
    public class_2561 asFormattedText() {
        class_2960 id = class_2378.field_11154.method_10221(fluid);
        if (class_1074.method_4663("block." + id.toString().replaceFirst(":", ".")))
            return new class_2588("block." + id.toString().replaceFirst(":", "."));
        return new class_2585(CollectionUtils.mapAndJoinToString(id.method_12832().split("_"), StringUtils::capitalize, " "));
    }
}
