/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl;

import com.google.common.collect.Lists;
import me.shedaniel.math.Point;
import me.shedaniel.math.Rectangle;
import me.shedaniel.rei.api.*;
import me.shedaniel.rei.api.widgets.Tooltip;
import net.minecraft.class_1059;
import net.minecraft.class_1087;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1836;
import net.minecraft.class_2378;
import net.minecraft.class_2487;
import net.minecraft.class_2499;
import net.minecraft.class_2520;
import net.minecraft.class_2561;
import net.minecraft.class_2585;
import net.minecraft.class_2588;
import net.minecraft.class_2960;
import net.minecraft.class_308;
import net.minecraft.class_310;
import net.minecraft.class_3532;
import net.minecraft.class_4493;
import net.minecraft.class_4587;
import net.minecraft.class_4597;
import net.minecraft.class_4608;
import net.minecraft.class_809;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;

@ApiStatus.Internal
public class ItemEntryStack extends AbstractEntryStack implements OptimalEntryStack {
    private class_1799 itemStack;
    
    public ItemEntryStack(class_1799 itemStack) {
        this.itemStack = itemStack;
    }
    
    @Override
    public Optional<class_2960> getIdentifier() {
        return Optional.ofNullable(class_2378.field_11142.method_10221(getItem()));
    }
    
    @Override
    public Type getType() {
        return Type.ITEM;
    }
    
    @Override
    public int getAmount() {
        return itemStack.method_7947();
    }
    
    @Override
    public double getFloatingAmount() {
        return itemStack.method_7947();
    }
    
    @Override
    public void setAmount(int amount) {
        itemStack.method_7939(amount);
    }
    
    @Override
    public void setFloatingAmount(double amount) {
        itemStack.method_7939(class_3532.method_15357(amount));
    }
    
    @Override
    public boolean isEmpty() {
        return itemStack.method_7960();
    }
    
    @Override
    public EntryStack copy() {
        EntryStack stack = EntryStack.create(getItemStack().method_7972());
        for (Map.Entry<Settings<?>, Object> entry : getSettings().entrySet()) {
            stack.setting((Settings<? super Object>) entry.getKey(), entry.getValue());
        }
        return stack;
    }
    
    @Override
    public Object getObject() {
        return itemStack;
    }
    
    /**
     * type:
     * 0: ignore tags and amount
     * 1: ignore tags
     * 2: ignore amount
     * 3: all
     */
    private Boolean compareIfFluid(EntryStack stack, int type) {
        EntryStack fluid = EntryStack.copyItemToFluid(this);
        if (fluid.isEmpty()) return null;
        if (stack.getType() == Type.ITEM)
            stack = EntryStack.copyItemToFluid(stack);
        if (stack.isEmpty()) return null;
        switch (type) {
            case 0:
                return fluid.equalsIgnoreTagsAndAmount(stack);
            case 1:
                return fluid.equalsIgnoreTags(stack);
            case 2:
                return fluid.equalsIgnoreAmount(stack);
            case 3:
                return fluid.equalsAll(stack);
        }
        return null;
    }
    
    @Override
    public boolean equalsIgnoreTagsAndAmount(EntryStack stack) {
        Boolean ifFluid = compareIfFluid(stack, 0);
        if (ifFluid != null) return ifFluid;
        if (stack.getType() != Type.ITEM)
            return false;
        return itemStack.method_7909() == stack.getItem();
    }
    
    @Override
    public boolean equalsAll(EntryStack stack) {
        if (stack.getType() != Type.ITEM)
            return false;
        return itemStack.method_7909() == stack.getItem() && getAmount() != stack.getAmount() && class_1799.method_7975(itemStack, stack.getItemStack());
    }
    
    @Override
    public boolean equalsIgnoreAmount(EntryStack stack) {
        Boolean ifFluid = compareIfFluid(stack, 2);
        if (ifFluid != null) return ifFluid;
        if (stack.getType() != Type.ITEM)
            return false;
        if (itemStack.method_7909() != stack.getItem())
            return false;
        class_1799 otherStack = stack.getItemStack();
        class_2487 o1 = itemStack.method_7969();
        class_2487 o2 = otherStack.method_7969();
        return o1 == o2 || ((o1 != null && o2 != null) && equalsTagWithoutCount(o1, o2));
    }
    
    private boolean equalsTagWithoutCount(class_2487 o1, class_2487 o2) {
        int o1Size = 0;
        int o2Size = 0;
        for (String key : o1.method_10541()) {
            if (key.equals("Count"))
                continue;
            o1Size++;
        }
        for (String key : o2.method_10541()) {
            if (key.equals("Count"))
                continue;
            o2Size++;
            if (o2Size > o1Size)
                return false;
        }
        if (o1Size != o2Size)
            return false;
        
        try {
            for (String key : o1.method_10541()) {
                if (key.equals("Count"))
                    continue;
                class_2520 value = o1.method_10580(key);
                class_2520 otherValue = o2.method_10580(key);
                if (!equalsTag(value, otherValue))
                    return false;
            }
        } catch (ClassCastException | NullPointerException unused) {
            return false;
        }
        
        return true;
    }
    
    private boolean equalsTag(class_2520 tag, class_2520 otherTag) {
        if (tag == null || otherTag == null) {
            return tag == otherTag;
        }
        if (tag instanceof class_2499 && otherTag instanceof class_2499)
            return equalsList((class_2499) tag, (class_2499) otherTag);
        return tag.equals(otherTag);
    }
    
    private boolean equalsList(class_2499 listTag, class_2499 otherListTag) {
        if (listTag.size() != otherListTag.size())
            return false;
        for (int i = 0; i < listTag.size(); i++) {
            class_2520 value = listTag.method_10534(i);
            class_2520 otherValue = otherListTag.method_10534(i);
            if (!equalsTag(value, otherValue))
                return false;
        }
        return true;
    }
    
    @Override
    public boolean equalsIgnoreTags(EntryStack stack) {
        Boolean ifFluid = compareIfFluid(stack, 1);
        if (ifFluid != null) return ifFluid;
        if (stack.getType() != Type.ITEM)
            return false;
        if (itemStack.method_7909() != stack.getItem())
            return false;
        return getAmount() == stack.getAmount();
    }
    
    @Override
    public int hashOfAll() {
        int result = hashIgnoreAmount();
        result = 31 * result + itemStack.method_7947();
        return result;
    }
    
    @Override
    public int hashIgnoreTags() {
        int result = hashIgnoreAmountAndTags();
        result = 31 * result + itemStack.method_7947();
        return result;
    }
    
    @Override
    public int hashIgnoreAmount() {
        int result = 1;
        result = 31 * result + getType().hashCode();
        result = 31 * result + itemStack.method_7909().hashCode();
        if (itemStack.method_7985()) {
            result = 31 * result + itemStack.method_7969().method_10714().hashCode();
        } else {
            result = 31 * result;
        }
        return result;
    }
    
    @Override
    public int hashIgnoreAmountAndTags() {
        int result = 1;
        result = 31 * result + getType().hashCode();
        result = 31 * result + itemStack.method_7909().hashCode();
        return result;
    }
    
    @Nullable
    @Override
    public Tooltip getTooltip(Point point) {
        if (isEmpty() || !get(Settings.TOOLTIP_ENABLED).get())
            return null;
        List<class_2561> toolTip = Lists.newArrayList(tryGetItemStackToolTip(true));
        toolTip.addAll(get(Settings.TOOLTIP_APPEND_EXTRA).apply(this));
        if (get(Settings.TOOLTIP_APPEND_MOD).get() && ConfigObject.getInstance().shouldAppendModNames()) {
            final class_2561 modString = ClientHelper.getInstance().getFormattedModFromItem(getItem());
            final String modId = ClientHelper.getInstance().getModFromItem(getItem());
            boolean alreadyHasMod = false;
            for (class_2561 s : toolTip)
                if (s.method_10851().equalsIgnoreCase(modId)) {
                    alreadyHasMod = true;
                    break;
                }
            if (!alreadyHasMod)
                toolTip.add(modString);
        }
        return Tooltip.create(toolTip);
    }
    
    @Override
    public void render(class_4587 matrices, Rectangle bounds, int mouseX, int mouseY, float delta) {
        optimisedRenderStart(matrices, delta);
        optimisedRenderBase(matrices, bounds, mouseX, mouseY, delta);
        optimisedRenderOverlay(matrices, bounds, mouseX, mouseY, delta);
        optimisedRenderEnd(matrices, delta);
    }
    
    @SuppressWarnings("deprecation")
    @Override
    public void optimisedRenderStart(class_4587 matrices, float delta) {
        class_310.method_1551().method_1531().method_22813(class_1059.field_5275);
        class_4493.method_21920();
    }
    
    @SuppressWarnings("deprecation")
    @Override
    public void optimisedRenderEnd(class_4587 matrices, float delta) {
        class_4493.method_21922();
    }
    
    private class_1087 getModelFromStack(class_1799 stack) {
        return class_310.method_1551().method_1480().method_4019(stack, null, null);
    }
    
    @Override
    public void optimisedRenderBase(class_4587 matrices, Rectangle bounds, int mouseX, int mouseY, float delta) {
        if (!isEmpty() && get(Settings.RENDER).get()) {
            class_1799 stack = getItemStack();
            ((ItemStackHook) (Object) stack).rei_setRenderEnchantmentGlint(get(Settings.Item.RENDER_ENCHANTMENT_GLINT).get());
            matrices.method_22903();
            matrices.method_22904(bounds.getCenterX(), bounds.getCenterY(), 100.0F + getZ());
            matrices.method_22905(bounds.getWidth(), (bounds.getWidth() + bounds.getHeight()) / -2f, bounds.getHeight());
            class_4597.class_4598 immediate = class_310.method_1551().method_22940().method_23000();
            class_1087 model = getModelFromStack(stack);
            boolean sideLit = !model.method_24304();
            if (sideLit)
                class_308.method_24210();
            class_310.method_1551().method_1480().method_23179(stack, class_809.class_811.field_4317, false, matrices, immediate, 15728880, class_4608.field_21444, model);
            immediate.method_22993();
            if (sideLit)
                class_308.method_24211();
            matrices.method_22909();
            ((ItemStackHook) (Object) stack).rei_setRenderEnchantmentGlint(false);
        }
    }
    
    @Override
    public void optimisedRenderOverlay(class_4587 matrices, Rectangle bounds, int mouseX, int mouseY, float delta) {
        if (!isEmpty() && get(Settings.RENDER).get()) {
            class_310.method_1551().method_1480().field_4730 = getZ();
            class_310.method_1551().method_1480().method_4022(class_310.method_1551().field_1772, getItemStack(), bounds.x, bounds.y, get(Settings.RENDER_COUNTS).get() ? get(Settings.COUNTS).apply(this) : "");
            class_310.method_1551().method_1480().field_4730 = 0.0F;
        }
    }
    
    private static final List<class_1792> SEARCH_BLACKLISTED = Lists.newArrayList();
    
    @Override
    public @NotNull class_2561 asFormattedText() {
        if (!SEARCH_BLACKLISTED.contains(getItem()))
            try {
                return getItemStack().method_7964();
            } catch (Throwable e) {
                e.printStackTrace();
                SEARCH_BLACKLISTED.add(getItem());
            }
        try {
            return new class_2588("item." + class_2378.field_11142.method_10221(getItem()).toString().replace(":", "."));
        } catch (Throwable e) {
            e.printStackTrace();
        }
        return new class_2585("ERROR");
    }
    
    private List<class_2561> tryGetItemStackToolTip(boolean careAboutAdvanced) {
        if (!SEARCH_BLACKLISTED.contains(getItem()))
            try {
                return itemStack.method_7950(class_310.method_1551().field_1724, class_310.method_1551().field_1690.field_1827 && careAboutAdvanced ? class_1836.class_1837.field_8935 : class_1836.class_1837.field_8934);
            } catch (Throwable e) {
                e.printStackTrace();
                SEARCH_BLACKLISTED.add(getItem());
            }
        return Collections.singletonList(asFormattedText());
    }
}
