/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import me.shedaniel.rei.RoughlyEnoughItemsCore;
import me.shedaniel.rei.api.ConfigObject;
import me.shedaniel.rei.api.EntryRegistry;
import me.shedaniel.rei.api.EntryStack;
import me.shedaniel.rei.impl.filtering.FilteringContextImpl;
import me.shedaniel.rei.impl.filtering.FilteringRule;
import me.shedaniel.rei.utils.CollectionUtils;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_2371;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.stream.Stream;

@ApiStatus.Internal
@Environment(EnvType.CLIENT)
public class EntryRegistryImpl implements EntryRegistry {
    
    private final List<EntryStack> preFilteredList = Lists.newCopyOnWriteArrayList();
    private final List<EntryStack> entries = Lists.newCopyOnWriteArrayList();
    private final List<AmountIgnoredEntryStackWrapper> reloadingRegistry = Lists.newArrayList();
    private boolean reloading;
    
    private static EntryStack findFirstOrNullEqualsEntryIgnoreAmount(Collection<EntryStack> list, EntryStack obj) {
        for (EntryStack t : list) {
            if (t.equalsIgnoreAmount(obj))
                return t;
        }
        return null;
    }
    
    public void finishReload() {
        reloading = false;
        preFilteredList.clear();
        reloadingRegistry.removeIf(AmountIgnoredEntryStackWrapper::isEmpty);
        entries.clear();
        entries.addAll(CollectionUtils.map(reloadingRegistry, AmountIgnoredEntryStackWrapper::unwrap));
        reloadingRegistry.clear();
    }
    
    @Override
    @NotNull
    public Stream<EntryStack> getEntryStacks() {
        return entries.stream();
    }
    
    @Override
    @NotNull
    public List<EntryStack> getPreFilteredList() {
        return preFilteredList;
    }
    
    public void refilter() {
        long started = System.currentTimeMillis();
        
        FilteringContextImpl context = new FilteringContextImpl(entries);
        List<FilteringRule<?>> rules = ConfigObject.getInstance().getFilteringRules();
        for (int i = rules.size() - 1; i >= 0; i--) {
            context.handleResult(rules.get(i).processFilteredStacks(context));
        }
        
        Set<AmountIgnoredEntryStackWrapper> set = Sets.newLinkedHashSet();
        set.addAll(CollectionUtils.map(entries, AmountIgnoredEntryStackWrapper::new));
        Collection<EntryStack> hiddenStacks = context.getHiddenStacks();
        set.removeAll(CollectionUtils.map(hiddenStacks, AmountIgnoredEntryStackWrapper::new));
        preFilteredList.clear();
        preFilteredList.addAll(CollectionUtils.map(set, AmountIgnoredEntryStackWrapper::unwrap));
        
        long time = System.currentTimeMillis() - started;
        RoughlyEnoughItemsCore.LOGGER.info("Refiltered %d entries with %d rules in %dms.", entries.size() - preFilteredList.size(), rules.size(), time);
    }
    
    public void reset() {
        entries.clear();
        reloadingRegistry.clear();
        preFilteredList.clear();
        reloading = true;
    }
    
    @NotNull
    @Override
    public List<class_1799> appendStacksForItem(@NotNull class_1792 item) {
        class_2371<class_1799> list = class_2371.method_10211();
        item.method_7850(item.method_7859(), list);
        if (list.isEmpty())
            return Collections.singletonList(item.method_7854());
        return list;
    }
    
    @NotNull
    @Override
    public class_1799[] getAllStacksFromItem(@NotNull class_1792 item) {
        List<class_1799> list = appendStacksForItem(item);
        class_1799[] array = list.toArray(new class_1799[0]);
        Arrays.sort(array, (a, b) -> class_1799.method_7973(a, b) ? 0 : 1);
        return array;
    }
    
    @Override
    public void registerEntriesAfter(@Nullable EntryStack afterStack, @NotNull Collection<@NotNull ? extends EntryStack> stacks) {
        if (reloading) {
            int index = afterStack != null ? reloadingRegistry.lastIndexOf(new AmountIgnoredEntryStackWrapper(afterStack)) : -1;
            if (index >= 0) {
                reloadingRegistry.addAll(index, CollectionUtils.map(stacks, AmountIgnoredEntryStackWrapper::new));
            } else reloadingRegistry.addAll(CollectionUtils.map(stacks, AmountIgnoredEntryStackWrapper::new));
        } else {
            if (afterStack != null) {
                int index = entries.lastIndexOf(afterStack);
                entries.addAll(index, stacks);
            } else entries.addAll(stacks);
        }
    }
}
