/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.biome;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.mojang.datafixers.util.Pair;
import com.mojang.logging.LogUtils;
import org.slf4j.Logger;
import net.fabricmc.fabric.impl.biome.modification.BuiltInRegistryKeys;
import net.minecraft.class_1959;
import net.minecraft.class_1966;
import net.minecraft.class_4766;
import net.minecraft.class_5321;
import net.minecraft.class_6544;
import net.minecraft.class_6880;
import net.minecraft.class_7871;

/**
 * Internal data for modding Vanilla's {@link class_4766.class_5305#field_24723}.
 */
public final class NetherBiomeData {
	// Cached sets of the biomes that would generate from Vanilla's default biome source without consideration
	// for data packs (as those would be distinct biome sources).
	private static final Set<class_5321<class_1959>> NETHER_BIOMES = new HashSet<>();

	private static final Map<class_5321<class_1959>, class_6544.class_4762> NETHER_BIOME_NOISE_POINTS = new HashMap<>();

	private static final Logger LOGGER = LogUtils.getLogger();

	private NetherBiomeData() {
	}

	public static void addNetherBiome(class_5321<class_1959> biome, class_6544.class_4762 spawnNoisePoint) {
		Preconditions.checkArgument(biome != null, "Biome is null");
		Preconditions.checkArgument(spawnNoisePoint != null, "MultiNoiseUtil.NoiseValuePoint is null");
		NETHER_BIOME_NOISE_POINTS.put(biome, spawnNoisePoint);
		clearBiomeSourceCache();
	}

	public static Map<class_5321<class_1959>, class_6544.class_4762> getNetherBiomeNoisePoints() {
		return NETHER_BIOME_NOISE_POINTS;
	}

	public static boolean canGenerateInNether(class_5321<class_1959> biome) {
		if (NETHER_BIOMES.isEmpty()) {
			class_4766 source = class_4766.class_5305.field_24723.method_28469(BuiltInRegistryKeys.biomeRegistryWrapper());

			for (class_6880<class_1959> entry : source.method_28443()) {
				entry.method_40230().ifPresent(NETHER_BIOMES::add);
			}
		}

		return NETHER_BIOMES.contains(biome) || NETHER_BIOME_NOISE_POINTS.containsKey(biome);
	}

	private static void clearBiomeSourceCache() {
		NETHER_BIOMES.clear(); // Clear cached biome source data
	}

	private static class_6544.class_6547<class_6880<class_1959>> withModdedBiomeEntries(class_6544.class_6547<class_6880<class_1959>> entries, class_7871<class_1959> biomes) {
		if (NETHER_BIOME_NOISE_POINTS.isEmpty()) {
			return entries;
		}

		ArrayList<Pair<class_6544.class_4762, class_6880<class_1959>>> entryList = new ArrayList<>(entries.method_38128());

		for (Map.Entry<class_5321<class_1959>, class_6544.class_4762> entry : NETHER_BIOME_NOISE_POINTS.entrySet()) {
			class_6880.class_6883<class_1959> biomeEntry = biomes.method_46746(entry.getKey()).orElse(null);

			if (biomeEntry != null) {
				entryList.add(Pair.of(entry.getValue(), biomeEntry));
			} else {
				LOGGER.warn("Nether biome {} not loaded", entry.getKey().method_29177());
			}
		}

		return new class_6544.class_6547<>(entryList);
	}

	public static void modifyBiomeSource(class_7871<class_1959> biomeRegistry, class_1966 biomeSource) {
		if (biomeSource instanceof class_4766 multiNoiseBiomeSource) {
			if (((BiomeSourceAccess) multiNoiseBiomeSource).fabric_shouldModifyBiomeEntries() && multiNoiseBiomeSource.method_38168(class_4766.class_5305.field_24723)) {
				multiNoiseBiomeSource.field_34498 = NetherBiomeData.withModdedBiomeEntries(
						class_4766.class_5305.field_24723.field_24726.apply(biomeRegistry),
						biomeRegistry);
				multiNoiseBiomeSource.field_20643 = multiNoiseBiomeSource.field_34498.getEntries().stream().map(Pair::getSecond).collect(Collectors.toSet());
				((BiomeSourceAccess) multiNoiseBiomeSource).fabric_setModifyBiomeEntries(false);
			}
		}
	}
}
