/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.biome.v1;

import java.util.Optional;
import java.util.OptionalInt;
import java.util.function.BiPredicate;
import net.minecraft.class_1299;
import net.minecraft.class_1311;
import net.minecraft.class_1959;
import net.minecraft.class_2893;
import net.minecraft.class_2922;
import net.minecraft.class_3414;
import net.minecraft.class_4761;
import net.minecraft.class_4763;
import net.minecraft.class_4967;
import net.minecraft.class_4968;
import net.minecraft.class_5195;
import net.minecraft.class_5321;
import net.minecraft.class_5483;
import net.minecraft.class_6796;
import net.minecraft.class_6880;
import org.jetbrains.annotations.NotNull;

/**
 * Allows {@link class_1959} properties to be modified.
 */
public interface BiomeModificationContext {
	/**
	 * Returns the modification context for the biomes weather properties.
	 */
	WeatherContext getWeather();

	/**
	 * Returns the modification context for the biomes effects.
	 */
	EffectsContext getEffects();

	/**
	 * Returns the modification context for the biomes generation settings.
	 */
	GenerationSettingsContext getGenerationSettings();

	/**
	 * Returns the modification context for the biomes spawn settings.
	 */
	SpawnSettingsContext getSpawnSettings();

	interface WeatherContext {
		/**
		 * @see class_1959#method_48163()
		 * @see class_1959.class_1960#method_48164(boolean)
		 */
		void setPrecipitation(boolean hasPrecipitation);

		/**
		 * @see class_1959#method_8712()
		 * @see class_1959.class_1960#method_8747(float)
		 */
		void setTemperature(float temperature);

		/**
		 * @see class_1959.class_1960#method_30777(class_1959.class_5484)
		 */
		void setTemperatureModifier(class_1959.class_5484 temperatureModifier);

		/**
		 * @see class_1959.class_5482#downfall()
		 * @see class_1959.class_1960#method_8727(float)
		 */
		void setDownfall(float downfall);
	}

	interface EffectsContext {
		/**
		 * @see class_4763#method_24387()
		 * @see class_4763.class_4764#method_24392(int)
		 */
		void setFogColor(int color);

		/**
		 * @see class_4763#method_24388()
		 * @see class_4763.class_4764#method_24395(int)
		 */
		void setWaterColor(int color);

		/**
		 * @see class_4763#method_24389()
		 * @see class_4763.class_4764#method_24397(int)
		 */
		void setWaterFogColor(int color);

		/**
		 * @see class_4763#method_30810()
		 * @see class_4763.class_4764#method_30820(int)
		 */
		void setSkyColor(int color);

		/**
		 * @see class_4763#method_30811()
		 * @see class_4763.class_4764#method_30821(int)
		 */
		void setFoliageColor(Optional<Integer> color);

		/**
		 * @see class_4763#method_30811()
		 * @see class_4763.class_4764#method_30821(int)
		 */
		default void setFoliageColor(int color) {
			setFoliageColor(Optional.of(color));
		}

		/**
		 * @see class_4763#method_30811()
		 * @see class_4763.class_4764#method_30821(int)
		 */
		default void setFoliageColor(OptionalInt color) {
			color.ifPresentOrElse(this::setFoliageColor, this::clearFoliageColor);
		}

		/**
		 * @see class_4763#method_30811()
		 * @see class_4763.class_4764#method_30821(int)
		 */
		default void clearFoliageColor() {
			setFoliageColor(Optional.empty());
		}

		/**
		 * @see class_4763#method_30812()
		 * @see class_4763.class_4764#method_30822(int)
		 */
		void setGrassColor(Optional<Integer> color);

		/**
		 * @see class_4763#method_30812()
		 * @see class_4763.class_4764#method_30822(int)
		 */
		default void setGrassColor(int color) {
			setGrassColor(Optional.of(color));
		}

		/**
		 * @see class_4763#method_30812()
		 * @see class_4763.class_4764#method_30822(int)
		 */
		default void setGrassColor(OptionalInt color) {
			color.ifPresentOrElse(this::setGrassColor, this::clearGrassColor);
		}

		/**
		 * @see class_4763#method_30812()
		 * @see class_4763.class_4764#method_30822(int)
		 */
		default void clearGrassColor() {
			setGrassColor(Optional.empty());
		}

		/**
		 * @see class_4763#method_30814()
		 * @see class_4763.class_4764#method_30818(class_4763.class_5486)
		 */
		void setGrassColorModifier(@NotNull class_4763.class_5486 colorModifier);

		/**
		 * @see class_4763#method_24390()
		 * @see class_4763.class_4764#method_24393(class_4761)
		 */
		void setParticleConfig(Optional<class_4761> particleConfig);

		/**
		 * @see class_4763#method_24390()
		 * @see class_4763.class_4764#method_24393(class_4761)
		 */
		default void setParticleConfig(@NotNull class_4761 particleConfig) {
			setParticleConfig(Optional.of(particleConfig));
		}

		/**
		 * @see class_4763#method_24390()
		 * @see class_4763.class_4764#method_24393(class_4761)
		 */
		default void clearParticleConfig() {
			setParticleConfig(Optional.empty());
		}

		/**
		 * @see class_4763#method_24939()
		 * @see class_4763.class_4764#method_24942(class_6880)
		 */
		void setAmbientSound(Optional<class_6880<class_3414>> sound);

		/**
		 * @see class_4763#method_24939()
		 * @see class_4763.class_4764#method_24942(class_6880)
		 */
		default void setAmbientSound(@NotNull class_6880<class_3414> sound) {
			setAmbientSound(Optional.of(sound));
		}

		/**
		 * @see class_4763#method_24939()
		 * @see class_4763.class_4764#method_24942(class_6880)
		 */
		default void clearAmbientSound() {
			setAmbientSound(Optional.empty());
		}

		/**
		 * @see class_4763#method_24940()
		 * @see class_4763.class_4764#method_24943(class_4968)
		 */
		void setMoodSound(Optional<class_4968> sound);

		/**
		 * @see class_4763#method_24940()
		 * @see class_4763.class_4764#method_24943(class_4968)
		 */
		default void setMoodSound(@NotNull class_4968 sound) {
			setMoodSound(Optional.of(sound));
		}

		/**
		 * @see class_4763#method_24940()
		 * @see class_4763.class_4764#method_24943(class_4968)
		 */
		default void clearMoodSound() {
			setMoodSound(Optional.empty());
		}

		/**
		 * @see class_4763#method_24941()
		 * @see class_4763.class_4764#method_24944(class_4967)
		 */
		void setAdditionsSound(Optional<class_4967> sound);

		/**
		 * @see class_4763#method_24941()
		 * @see class_4763.class_4764#method_24944(class_4967)
		 */
		default void setAdditionsSound(@NotNull class_4967 sound) {
			setAdditionsSound(Optional.of(sound));
		}

		/**
		 * @see class_4763#method_24941()
		 * @see class_4763.class_4764#method_24944(class_4967)
		 */
		default void clearAdditionsSound() {
			setAdditionsSound(Optional.empty());
		}

		/**
		 * @see class_4763#method_27345()
		 * @see class_4763.class_4764#method_27346(class_5195)
		 */
		void setMusic(Optional<class_5195> sound);

		/**
		 * @see class_4763#method_27345()
		 * @see class_4763.class_4764#method_27346(class_5195)
		 */
		default void setMusic(@NotNull class_5195 sound) {
			setMusic(Optional.of(sound));
		}

		/**
		 * @see class_4763#method_27345()
		 * @see class_4763.class_4764#method_27346(class_5195)
		 */
		default void clearMusic() {
			setMusic(Optional.empty());
		}
	}

	interface GenerationSettingsContext {
		/**
		 * Removes a feature from one of this biomes generation steps, and returns if any features were removed.
		 */
		boolean removeFeature(class_2893.class_2895 step, class_5321<class_6796> placedFeatureKey);

		/**
		 * Removes a feature from all of this biomes generation steps, and returns if any features were removed.
		 */
		default boolean removeFeature(class_5321<class_6796> placedFeatureKey) {
			boolean anyFound = false;

			for (class_2893.class_2895 step : class_2893.class_2895.values()) {
				if (removeFeature(step, placedFeatureKey)) {
					anyFound = true;
				}
			}

			return anyFound;
		}

		/**
		 * Adds a feature to one of this biomes generation steps, identified by the placed feature's registry key.
		 */
		void addFeature(class_2893.class_2895 step, class_5321<class_6796> placedFeatureKey);

		/**
		 * Adds a configured carver to one of this biomes generation steps.
		 */
		void addCarver(class_2893.class_2894 step, class_5321<class_2922<?>> carverKey);

		/**
		 * Removes all carvers with the given key from one of this biomes generation steps.
		 *
		 * @return True if any carvers were removed.
		 */
		boolean removeCarver(class_2893.class_2894 step, class_5321<class_2922<?>> configuredCarverKey);

		/**
		 * Removes all carvers with the given key from all of this biomes generation steps.
		 *
		 * @return True if any carvers were removed.
		 */
		default boolean removeCarver(class_5321<class_2922<?>> configuredCarverKey) {
			boolean anyFound = false;

			for (class_2893.class_2894 step : class_2893.class_2894.values()) {
				if (removeCarver(step, configuredCarverKey)) {
					anyFound = true;
				}
			}

			return anyFound;
		}
	}

	interface SpawnSettingsContext {
		/**
		 * Associated JSON property: <code>creature_spawn_probability</code>.
		 *
		 * @see class_5483#method_31002()
		 * @see class_5483.class_5496#method_31008(float)
		 */
		void setCreatureSpawnProbability(float probability);

		/**
		 * Associated JSON property: <code>spawners</code>.
		 *
		 * @see class_5483#method_31004(class_1311)
		 * @see class_5483.class_5496#method_31011(class_1311, class_5483.class_1964)
		 */
		void addSpawn(class_1311 spawnGroup, class_5483.class_1964 spawnEntry);

		/**
		 * Removes any spawns matching the given predicate from this biome, and returns true if any matched.
		 *
		 * <p>Associated JSON property: <code>spawners</code>.
		 */
		boolean removeSpawns(BiPredicate<class_1311, class_5483.class_1964> predicate);

		/**
		 * Removes all spawns of the given entity type.
		 *
		 * <p>Associated JSON property: <code>spawners</code>.
		 *
		 * @return True if any spawns were removed.
		 */
		default boolean removeSpawnsOfEntityType(class_1299<?> entityType) {
			return removeSpawns((spawnGroup, spawnEntry) -> spawnEntry.field_9389 == entityType);
		}

		/**
		 * Removes all spawns of the given spawn group.
		 *
		 * <p>Associated JSON property: <code>spawners</code>.
		 */
		default void clearSpawns(class_1311 group) {
			removeSpawns((spawnGroup, spawnEntry) -> spawnGroup == group);
		}

		/**
		 * Removes all spawns.
		 *
		 * <p>Associated JSON property: <code>spawners</code>.
		 */
		default void clearSpawns() {
			removeSpawns((spawnGroup, spawnEntry) -> true);
		}

		/**
		 * Associated JSON property: <code>spawn_costs</code>.
		 *
		 * @see class_5483#method_31003(class_1299)
		 * @see class_5483.class_5496#method_31009(class_1299, double, double)
		 */
		void setSpawnCost(class_1299<?> entityType, double mass, double gravityLimit);

		/**
		 * Removes a spawn cost entry for a given entity type.
		 *
		 * <p>Associated JSON property: <code>spawn_costs</code>.
		 */
		void clearSpawnCost(class_1299<?> entityType);
	}
}
