/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.biome.modification;

import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiPredicate;
import java.util.stream.Collectors;

import com.google.common.base.Suppliers;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.api.biome.v1.BiomeModificationContext;
import net.minecraft.class_1299;
import net.minecraft.class_1311;
import net.minecraft.class_1959;
import net.minecraft.class_2378;
import net.minecraft.class_2893;
import net.minecraft.class_2922;
import net.minecraft.class_3031;
import net.minecraft.class_3414;
import net.minecraft.class_4761;
import net.minecraft.class_4763;
import net.minecraft.class_4967;
import net.minecraft.class_4968;
import net.minecraft.class_5195;
import net.minecraft.class_5321;
import net.minecraft.class_5455;
import net.minecraft.class_5482;
import net.minecraft.class_5483;
import net.minecraft.class_5485;
import net.minecraft.class_6010;
import net.minecraft.class_6012;
import net.minecraft.class_6796;
import net.minecraft.class_6880;
import net.minecraft.class_6885;
import net.minecraft.class_7924;

public class BiomeModificationContextImpl implements BiomeModificationContext {
	private final class_5455 registries;
	private final class_1959 biome;
	private final WeatherContext weather;
	private final EffectsContext effects;
	private final GenerationSettingsContextImpl generationSettings;
	private final SpawnSettingsContextImpl spawnSettings;

	public BiomeModificationContextImpl(class_5455 registries, class_1959 biome) {
		this.registries = registries;
		this.biome = biome;
		this.weather = new WeatherContextImpl();
		this.effects = new EffectsContextImpl();
		this.generationSettings = new GenerationSettingsContextImpl();
		this.spawnSettings = new SpawnSettingsContextImpl();
	}

	@Override
	public WeatherContext getWeather() {
		return weather;
	}

	@Override
	public EffectsContext getEffects() {
		return effects;
	}

	@Override
	public GenerationSettingsContext getGenerationSettings() {
		return generationSettings;
	}

	@Override
	public SpawnSettingsContext getSpawnSettings() {
		return spawnSettings;
	}

	/**
	 * Re-freeze any immutable lists and perform general post-modification cleanup.
	 */
	void freeze() {
		generationSettings.freeze();
		spawnSettings.freeze();
	}

	boolean shouldRebuildFeatures() {
		return generationSettings.rebuildFeatures;
	}

	private class WeatherContextImpl implements WeatherContext {
		@Override
		public void setPrecipitation(boolean hasPrecipitation) {
			biome.field_26393 = new class_5482(hasPrecipitation, biome.field_26393.comp_844(), biome.field_26393.comp_845(), biome.field_26393.comp_846());
		}

		@Override
		public void setTemperature(float temperature) {
			biome.field_26393 = new class_5482(biome.field_26393.comp_1187(), temperature, biome.field_26393.comp_845(), biome.field_26393.comp_846());
		}

		@Override
		public void setTemperatureModifier(class_1959.class_5484 temperatureModifier) {
			biome.field_26393 = new class_5482(biome.field_26393.comp_1187(), biome.field_26393.comp_844(), Objects.requireNonNull(temperatureModifier), biome.field_26393.comp_846());
		}

		@Override
		public void setDownfall(float downfall) {
			biome.field_26393 = new class_5482(biome.field_26393.comp_1187(), biome.field_26393.comp_844(), biome.field_26393.comp_845(), downfall);
		}
	}

	private class EffectsContextImpl implements EffectsContext {
		private final class_4763 effects = biome.method_24377();

		@Override
		public void setFogColor(int color) {
			effects.field_22067 = color;
		}

		@Override
		public void setWaterColor(int color) {
			effects.field_22068 = color;
		}

		@Override
		public void setWaterFogColor(int color) {
			effects.field_22069 = color;
		}

		@Override
		public void setSkyColor(int color) {
			effects.field_26418 = color;
		}

		@Override
		public void setFoliageColor(Optional<Integer> color) {
			effects.field_26419 = Objects.requireNonNull(color);
		}

		@Override
		public void setGrassColor(Optional<Integer> color) {
			effects.field_26420 = Objects.requireNonNull(color);
		}

		@Override
		public void setGrassColorModifier(@NotNull class_4763.class_5486 colorModifier) {
			effects.field_26421 = Objects.requireNonNull(colorModifier);
		}

		@Override
		public void setParticleConfig(Optional<class_4761> particleConfig) {
			effects.field_22070 = Objects.requireNonNull(particleConfig);
		}

		@Override
		public void setAmbientSound(Optional<class_6880<class_3414>> sound) {
			effects.field_22491 = Objects.requireNonNull(sound);
		}

		@Override
		public void setMoodSound(Optional<class_4968> sound) {
			effects.field_22492 = Objects.requireNonNull(sound);
		}

		@Override
		public void setAdditionsSound(Optional<class_4967> sound) {
			effects.field_22493 = Objects.requireNonNull(sound);
		}

		@Override
		public void setMusic(Optional<class_6012<class_5195>> sound) {
			effects.field_24113 = Objects.requireNonNull(sound);
		}

		@Override
		public void setMusicVolume(float volume) {
			effects.field_55050 = volume;
		}
	}

	private class GenerationSettingsContextImpl implements GenerationSettingsContext {
		private final class_2378<class_2922<?>> carvers = registries.method_30530(class_7924.field_41238);
		private final class_2378<class_6796> features = registries.method_30530(class_7924.field_41245);
		private final class_5485 generationSettings = biome.method_30970();

		boolean rebuildFeatures;

		/**
		 * Unfreeze the immutable lists found in the generation settings, and make sure they're filled up to every
		 * possible step if they're dense lists.
		 */
		GenerationSettingsContextImpl() {
			unfreezeFeatures();

			rebuildFeatures = false;
		}

		private void unfreezeFeatures() {
			generationSettings.field_26416 = new ArrayList<>(generationSettings.field_26416);
		}

		/**
		 * Re-freeze the lists in the generation settings to immutable variants, also fixes the flower features.
		 */
		public void freeze() {
			freezeFeatures();

			if (rebuildFeatures) {
				rebuildFlowerFeatures();
			}
		}

		private void freezeFeatures() {
			generationSettings.field_26416 = ImmutableList.copyOf(generationSettings.field_26416);
			// Replace the supplier to force a rebuild next time its called.
			generationSettings.field_34465 = Suppliers.memoize(() -> {
				return generationSettings.field_26416.stream().flatMap(class_6885::method_40239).map(class_6880::comp_349).collect(Collectors.toSet());
			});
		}

		private void rebuildFlowerFeatures() {
			// Replace the supplier to force a rebuild next time its called.
			generationSettings.field_26640 = Suppliers.memoize(() -> {
				return generationSettings.field_26416.stream().flatMap(class_6885::method_40239).map(class_6880::comp_349).flatMap(class_6796::method_39643).filter((configuredFeature) -> {
					return configuredFeature.comp_332() == class_3031.field_21219;
				}).collect(ImmutableList.toImmutableList());
			});
		}

		@Override
		public boolean removeFeature(class_2893.class_2895 step, class_5321<class_6796> placedFeatureKey) {
			class_6796 placedFeature = getEntry(features, placedFeatureKey).comp_349();

			int stepIndex = step.ordinal();
			List<class_6885<class_6796>> featureSteps = generationSettings.field_26416;

			if (stepIndex >= featureSteps.size()) {
				return false; // The step was not populated with any features yet
			}

			class_6885<class_6796> featuresInStep = featureSteps.get(stepIndex);
			List<class_6880<class_6796>> features = new ArrayList<>(featuresInStep.method_40239().toList());

			if (features.removeIf(feature -> feature.comp_349() == placedFeature)) {
				featureSteps.set(stepIndex, class_6885.method_40242(features));
				rebuildFeatures = true;

				return true;
			}

			return false;
		}

		@Override
		public void addFeature(class_2893.class_2895 step, class_5321<class_6796> entry) {
			List<class_6885<class_6796>> featureSteps = generationSettings.field_26416;
			int index = step.ordinal();

			// Add new empty lists for the generation steps that have no features yet
			while (index >= featureSteps.size()) {
				featureSteps.add(class_6885.method_40242(Collections.emptyList()));
			}

			class_6880.class_6883<class_6796> feature = getEntry(features, entry);

			// Don't add the feature if it's already present
			if (featureSteps.get(index).method_40241(feature)) {
				return;
			}

			featureSteps.set(index, plus(featureSteps.get(index), feature));

			// Ensure the list of flower features is up-to-date
			rebuildFeatures = true;
		}

		@Override
		public void addCarver(class_5321<class_2922<?>> entry) {
			// We do not need to delay evaluation of this since the registries are already fully built
			generationSettings.field_26415 = plus(generationSettings.field_26415, getEntry(carvers, entry));
		}

		@Override
		public boolean removeCarver(class_5321<class_2922<?>> configuredCarverKey) {
			class_2922<?> carver = getEntry(carvers, configuredCarverKey).comp_349();
			List<class_6880<class_2922<?>>> genCarvers = new ArrayList<>(generationSettings.field_26415.method_40239().toList());

			if (genCarvers.removeIf(entry -> entry.comp_349() == carver)) {
				generationSettings.field_26415 = class_6885.method_40242(genCarvers);
				return true;
			}

			return false;
		}

		private <T> class_6885<T> plus(@Nullable class_6885<T> values, class_6880<T> entry) {
			if (values == null) return class_6885.method_40246(entry);

			List<class_6880<T>> list = new ArrayList<>(values.method_40239().toList());
			list.add(entry);
			return class_6885.method_40242(list);
		}
	}

	/**
	 * Gets an entry from the given registry, assuming it's a registry loaded from data packs.
	 * Gives more helpful error messages if an entry is missing by checking if the modder
	 * forgot to data-gen the JSONs corresponding to their built-in objects.
	 */
	private static <T> class_6880.class_6883<T> getEntry(class_2378<T> registry, class_5321<T> key) {
		class_6880.class_6883<T> entry = registry.method_46746(key).orElse(null);

		if (entry == null) {
			// The key doesn't exist in the data packs
			throw new IllegalArgumentException("Couldn't find registry entry for " + key);
		}

		return entry;
	}

	private class SpawnSettingsContextImpl implements SpawnSettingsContext {
		private final class_5483 spawnSettings = biome.method_30966();
		private final EnumMap<class_1311, List<class_6010<class_5483.class_1964>>> fabricSpawners = new EnumMap<>(class_1311.class);

		SpawnSettingsContextImpl() {
			unfreezeSpawners();
			unfreezeSpawnCost();
		}

		private void unfreezeSpawners() {
			fabricSpawners.clear();

			for (class_1311 spawnGroup : class_1311.values()) {
				class_6012<class_5483.class_1964> entries = spawnSettings.field_26405.get(spawnGroup);

				if (entries != null) {
					fabricSpawners.put(spawnGroup, new ArrayList<>(entries.method_34994()));
				} else {
					fabricSpawners.put(spawnGroup, new ArrayList<>());
				}
			}
		}

		private void unfreezeSpawnCost() {
			spawnSettings.field_26406 = new HashMap<>(spawnSettings.field_26406);
		}

		public void freeze() {
			freezeSpawners();
			freezeSpawnCosts();
		}

		private void freezeSpawners() {
			Map<class_1311, class_6012<class_5483.class_1964>> spawners = new HashMap<>(spawnSettings.field_26405);

			for (Map.Entry<class_1311, List<class_6010<class_5483.class_1964>>> entry : fabricSpawners.entrySet()) {
				if (entry.getValue().isEmpty()) {
					spawners.put(entry.getKey(), class_6012.method_34990());
				} else {
					spawners.put(entry.getKey(), class_6012.method_34988(entry.getValue()));
				}
			}

			spawnSettings.field_26405 = ImmutableMap.copyOf(spawners);
		}

		private void freezeSpawnCosts() {
			spawnSettings.field_26406 = ImmutableMap.copyOf(spawnSettings.field_26406);
		}

		@Override
		public void setCreatureSpawnProbability(float probability) {
			spawnSettings.field_26404 = probability;
		}

		@Override
		public void addSpawn(class_1311 spawnGroup, class_5483.class_1964 spawnEntry, int weight) {
			Objects.requireNonNull(spawnGroup);
			Objects.requireNonNull(spawnEntry);

			fabricSpawners.get(spawnGroup).add(new class_6010<>(spawnEntry, weight));
		}

		@Override
		public boolean removeSpawns(BiPredicate<class_1311, class_5483.class_1964> predicate) {
			boolean anyRemoved = false;

			for (class_1311 group : class_1311.values()) {
				if (fabricSpawners.get(group).removeIf(entry -> predicate.test(group, entry.comp_2542()))) {
					anyRemoved = true;
				}
			}

			return anyRemoved;
		}

		@Override
		public void setSpawnCost(class_1299<?> entityType, double mass, double gravityLimit) {
			Objects.requireNonNull(entityType);
			spawnSettings.field_26406.put(entityType, new class_5483.class_5265(gravityLimit, mass));
		}

		@Override
		public void clearSpawnCost(class_1299<?> entityType) {
			spawnSettings.field_26406.remove(entityType);
		}
	}
}
