/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.biomes;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import net.fabricmc.fabric.api.biomes.v1.OverworldClimate;
import net.minecraft.class_1959;
import net.minecraft.class_1972;
import net.minecraft.class_2088;
import net.minecraft.class_2378;
import net.minecraft.class_3645;

/**
 * Lists and maps for internal use only! Stores data that is used by the various mixins into the world generation
 */
public final class InternalBiomeData {
	private InternalBiomeData() { }

	private static final EnumMap<OverworldClimate, WeightedBiomePicker> OVERWORLD_MODDED_CONTINENTAL_BIOME_PICKERS = new EnumMap<>(OverworldClimate.class);
	private static final Map<class_1959, WeightedBiomePicker> OVERWORLD_HILLS_MAP = new HashMap<>();
	private static final Map<class_1959, WeightedBiomePicker> OVERWORLD_SHORE_MAP = new HashMap<>();
	private static final Map<class_1959, WeightedBiomePicker> OVERWORLD_EDGE_MAP = new HashMap<>();
	private static final Map<class_1959, VariantTransformer> OVERWORLD_VARIANT_TRANSFORMERS = new HashMap<>();
	private static final Map<class_1959, class_1959> OVERWORLD_RIVER_MAP = new HashMap<>();

	private static final Set<class_1959> SPAWN_BIOMES = new HashSet<>();

	private static Method injectBiomeMethod = null;

	public static void addOverworldContinentalBiome(OverworldClimate climate, class_1959 biome, double weight) {
		Preconditions.checkArgument(climate != null, "Climate is null");
		Preconditions.checkArgument(biome != null, "Biome is null");
		Preconditions.checkArgument(!Double.isNaN(weight), "Weight is NaN");
		Preconditions.checkArgument(weight > 0.0, "Weight is less than or equal to 0.0 (%s)", weight);
		OVERWORLD_MODDED_CONTINENTAL_BIOME_PICKERS.computeIfAbsent(climate, k -> new WeightedBiomePicker()).addBiome(biome, weight);
		injectOverworldBiome(biome);
	}

	public static void addOverworldHillsBiome(class_1959 primary, class_1959 hills, double weight) {
		Preconditions.checkArgument(primary != null, "Primary biome is null");
		Preconditions.checkArgument(hills != null, "Hills biome is null");
		Preconditions.checkArgument(!Double.isNaN(weight), "Weight is NaN");
		Preconditions.checkArgument(weight > 0.0, "Weight is less than or equal to 0.0 (%s)", weight);
		OVERWORLD_HILLS_MAP.computeIfAbsent(primary, biome -> DefaultHillsData.injectDefaultHills(primary, new WeightedBiomePicker())).addBiome(hills, weight);
		injectOverworldBiome(hills);
	}

	public static void addOverworldShoreBiome(class_1959 primary, class_1959 shore, double weight) {
		Preconditions.checkArgument(primary != null, "Primary biome is null");
		Preconditions.checkArgument(shore != null, "Shore biome is null");
		Preconditions.checkArgument(!Double.isNaN(weight), "Weight is NaN");
		Preconditions.checkArgument(weight > 0.0, "Weight is less than or equal to 0.0 (%s)", weight);
		OVERWORLD_SHORE_MAP.computeIfAbsent(primary, biome -> new WeightedBiomePicker()).addBiome(shore, weight);
		injectOverworldBiome(shore);
	}

	public static void addOverworldEdgeBiome(class_1959 primary, class_1959 edge, double weight) {
		Preconditions.checkArgument(primary != null, "Primary biome is null");
		Preconditions.checkArgument(edge != null, "Edge biome is null");
		Preconditions.checkArgument(!Double.isNaN(weight), "Weight is NaN");
		Preconditions.checkArgument(weight > 0.0, "Weight is less than or equal to 0.0 (%s)", weight);
		OVERWORLD_EDGE_MAP.computeIfAbsent(primary, biome -> new WeightedBiomePicker()).addBiome(edge, weight);
		injectOverworldBiome(edge);
	}

	public static void addOverworldBiomeReplacement(class_1959 replaced, class_1959 variant, double chance, OverworldClimate[] climates) {
		Preconditions.checkArgument(replaced != null, "Replaced biome is null");
		Preconditions.checkArgument(variant != null, "Variant biome is null");
		Preconditions.checkArgument(chance > 0 && chance <= 1, "Chance is not greater than 0 or less than or equal to 1");
		OVERWORLD_VARIANT_TRANSFORMERS.computeIfAbsent(replaced, biome -> new VariantTransformer()).addBiome(variant, chance, climates);
		injectOverworldBiome(variant);
	}

	public static void setOverworldRiverBiome(class_1959 primary, class_1959 river) {
		Preconditions.checkArgument(primary != null, "Primary biome is null");
		OVERWORLD_RIVER_MAP.put(primary, river);

		if (river != null) {
			injectOverworldBiome(river);
		}
	}

	public static void addSpawnBiome(class_1959 biome) {
		Preconditions.checkArgument(biome != null, "Biome is null");
		SPAWN_BIOMES.add(biome);
	}

	private static void injectOverworldBiome(class_1959 biome) {
		try {
			if (injectBiomeMethod == null) {
				injectBiomeMethod = class_2088.class.getDeclaredMethod("fabric_injectBiome", class_1959.class);
				injectBiomeMethod.setAccessible(true);
			}

			injectBiomeMethod.invoke(null, biome);
		} catch (NoSuchMethodException | SecurityException | IllegalAccessException | InvocationTargetException e) {
			throw new RuntimeException(e);
		}
	}

	public static Set<class_1959> getSpawnBiomes() {
		return SPAWN_BIOMES;
	}

	public static Map<class_1959, WeightedBiomePicker> getOverworldHills() {
		return OVERWORLD_HILLS_MAP;
	}

	public static Map<class_1959, WeightedBiomePicker> getOverworldShores() {
		return OVERWORLD_SHORE_MAP;
	}

	public static Map<class_1959, WeightedBiomePicker> getOverworldEdges() {
		return OVERWORLD_EDGE_MAP;
	}

	public static Map<class_1959, class_1959> getOverworldRivers() {
		return OVERWORLD_RIVER_MAP;
	}

	public static EnumMap<OverworldClimate, WeightedBiomePicker> getOverworldModdedContinentalBiomePickers() {
		return OVERWORLD_MODDED_CONTINENTAL_BIOME_PICKERS;
	}

	public static Map<class_1959, VariantTransformer> getOverworldVariantTransformers() {
		return OVERWORLD_VARIANT_TRANSFORMERS;
	}

	private static class DefaultHillsData {
		private static final ImmutableMap<class_1959, class_1959> DEFAULT_HILLS;

		static WeightedBiomePicker injectDefaultHills(class_1959 base, WeightedBiomePicker picker) {
			class_1959 defaultHill = DEFAULT_HILLS.get(base);

			if (defaultHill != null) {
				picker.addBiome(defaultHill, 1);
			} else if (class_3645.method_15844(class_2378.field_11153.method_10249(base), class_2378.field_11153.method_10249(class_1972.field_9410))) {
				picker.addBiome(class_1972.field_9415, 1);
			} else if (base == class_1972.field_9446 || base == class_1972.field_9439 || base == class_1972.field_9470) {
				picker.addBiome(class_1972.field_9451, 1);
				picker.addBiome(class_1972.field_9409, 1);
			} else if (base == class_1972.field_9418) {
				// Note: Vanilla Deep Frozen Oceans only have a 1/3 chance of having default hills.
				// This is a clever trick that ensures that when a mod adds hills with a weight of 1, the 1/3 chance is fulfilled.
				// 0.5 + 1.0 = 1.5, and 0.5 / 1.5 = 1/3.

				picker.addBiome(class_1972.field_9451, 0.25);
				picker.addBiome(class_1972.field_9409, 0.25);
			} else if (base == class_1972.field_9451) {
				picker.addBiome(class_1972.field_9459, 1);
				picker.addBiome(class_1972.field_9409, 2);
			}

			return picker;
		}

		static {
			ImmutableMap.Builder<class_1959, class_1959> builder = ImmutableMap.builder();
			builder.put(class_1972.field_9424, class_1972.field_9466);
			builder.put(class_1972.field_9409, class_1972.field_9459);
			builder.put(class_1972.field_9412, class_1972.field_9421);
			builder.put(class_1972.field_9475, class_1972.field_9451);
			builder.put(class_1972.field_9420, class_1972.field_9428);
			builder.put(class_1972.field_9477, class_1972.field_9429);
			builder.put(class_1972.field_9454, class_1972.field_9425);
			builder.put(class_1972.field_9452, class_1972.field_9444);
			builder.put(class_1972.field_9417, class_1972.field_9432);
			builder.put(class_1972.field_9440, class_1972.field_9468);
			builder.put(class_1972.field_9423, class_1972.field_9446);
			builder.put(class_1972.field_9441, class_1972.field_9439);
			builder.put(class_1972.field_9467, class_1972.field_9470);
			builder.put(class_1972.field_9435, class_1972.field_9418);
			builder.put(class_1972.field_9472, class_1972.field_9460);
			builder.put(class_1972.field_9449, class_1972.field_9430);
			DEFAULT_HILLS = builder.build();
		}
	}
}
