/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.mixin.blockview.client;

import java.util.ConcurrentModificationException;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.spongepowered.asm.mixin.Mixin;
import org.spongepowered.asm.mixin.Unique;
import org.spongepowered.asm.mixin.injection.At;
import org.spongepowered.asm.mixin.injection.Inject;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfoReturnable;
import org.spongepowered.asm.mixin.injection.callback.LocalCapture;
import net.fabricmc.fabric.impl.blockview.client.RenderDataMapConsumer;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2586;
import net.minecraft.class_2818;
import net.minecraft.class_6850;
import net.minecraft.class_853;

@Mixin(class_6850.class)
public abstract class ChunkRendererRegionBuilderMixin {
	private static final AtomicInteger ERROR_COUNTER = new AtomicInteger();
	private static final Logger LOGGER = LoggerFactory.getLogger(ChunkRendererRegionBuilderMixin.class);

	@Inject(method = "build", at = @At("RETURN"), locals = LocalCapture.CAPTURE_FAILHARD)
	private void createDataMap(class_1937 world, class_2338 startPos, class_2338 endPos, int offset, CallbackInfoReturnable<class_853> cir, int startX, int startZ, int endX, int endZ, class_6850.class_6851[][] chunksXZ) {
		class_853 rendererRegion = cir.getReturnValue();

		if (rendererRegion == null) {
			return;
		}

		// instantiated lazily - avoids allocation for chunks without any data objects - which is most of them!
		Long2ObjectOpenHashMap<Object> map = null;

		for (class_6850.class_6851[] chunksZ : chunksXZ) {
			for (class_6850.class_6851 chunk : chunksZ) {
				// Hash maps in chunks should generally not be modified outside of client thread
				// but does happen in practice, due to mods or inconsistent vanilla behaviors, causing
				// CMEs when we iterate the map. (Vanilla does not iterate these maps when it builds
				// the chunk cache and does not suffer from this problem.)
				//
				// We handle this simply by retrying until it works. Ugly but effective.
				while (true) {
					try {
						map = mapChunk(chunk.getChunk(), startPos, endPos, map);
						break;
					} catch (ConcurrentModificationException e) {
						final int count = ERROR_COUNTER.incrementAndGet();

						if (count <= 5) {
							LOGGER.warn("[Block Entity Render Data] Encountered CME during render region build. A mod is accessing or changing chunk data outside the main thread. Retrying.", e);

							if (count == 5) {
								LOGGER.info("[Block Entity Render Data] Subsequent exceptions will be suppressed.");
							}
						}
					}
				}
			}
		}

		if (map != null) {
			((RenderDataMapConsumer) rendererRegion).fabric_acceptRenderDataMap(map);
		}
	}

	@Unique
	private static Long2ObjectOpenHashMap<Object> mapChunk(class_2818 chunk, class_2338 posFrom, class_2338 posTo, Long2ObjectOpenHashMap<Object> map) {
		final int xMin = posFrom.method_10263();
		final int xMax = posTo.method_10263();
		final int yMin = posFrom.method_10264();
		final int yMax = posTo.method_10264();
		final int zMin = posFrom.method_10260();
		final int zMax = posTo.method_10260();

		for (Map.Entry<class_2338, class_2586> entry : chunk.method_12214().entrySet()) {
			final class_2338 pos = entry.getKey();

			if (pos.method_10263() >= xMin && pos.method_10263() <= xMax
					&& pos.method_10264() >= yMin && pos.method_10264() <= yMax
					&& pos.method_10260() >= zMin && pos.method_10260() <= zMax) {
				final Object data = entry.getValue().getRenderData();

				if (data != null) {
					if (map == null) {
						map = new Long2ObjectOpenHashMap<>();
					}

					map.put(pos.method_10063(), data);
				}
			}
		}

		return map;
	}
}
