/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.mixin.blockview.client;

import java.util.ConcurrentModificationException;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import com.llamalad7.mixinextras.sugar.Local;
import com.llamalad7.mixinextras.sugar.Share;
import com.llamalad7.mixinextras.sugar.ref.LocalRef;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.spongepowered.asm.mixin.Mixin;
import org.spongepowered.asm.mixin.Unique;
import org.spongepowered.asm.mixin.injection.At;
import org.spongepowered.asm.mixin.injection.Inject;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfoReturnable;
import net.fabricmc.fabric.impl.blockview.client.RenderDataMapConsumer;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2586;
import net.minecraft.class_2818;
import net.minecraft.class_4076;
import net.minecraft.class_6850;
import net.minecraft.class_853;

@Mixin(class_6850.class)
public abstract class ChunkRendererRegionBuilderMixin {
	private static final AtomicInteger ERROR_COUNTER = new AtomicInteger();
	private static final Logger LOGGER = LoggerFactory.getLogger(ChunkRendererRegionBuilderMixin.class);

	@Inject(method = "build", at = @At(value = "INVOKE", target = "Lnet/minecraft/client/render/chunk/ChunkRendererRegionBuilder$ClientChunk;getRenderedChunk()Lnet/minecraft/client/render/chunk/RenderedChunk;"))
	private void copyDataForChunk(class_1937 world, class_4076 chunkSectionPos, CallbackInfoReturnable<class_853> cir, @Local(ordinal = 1) class_6850.class_6851 clientChunk, @Share("dataMap") LocalRef<Long2ObjectOpenHashMap<Object>> mapRef) {
		// Hash maps in chunks should generally not be modified outside of client thread
		// but does happen in practice, due to mods or inconsistent vanilla behaviors, causing
		// CMEs when we iterate the map. (Vanilla does not iterate these maps when it builds
		// the chunk cache and does not suffer from this problem.)
		//
		// We handle this simply by retrying until it works. Ugly but effective.
		while (true) {
			try {
				mapRef.set(mapChunk(clientChunk.method_39971(), chunkSectionPos, mapRef.get()));
				break;
			} catch (ConcurrentModificationException e) {
				final int count = ERROR_COUNTER.incrementAndGet();

				if (count <= 5) {
					LOGGER.warn("[Block Entity Render Data] Encountered CME during render region build. A mod is accessing or changing chunk data outside the main thread. Retrying.", e);

					if (count == 5) {
						LOGGER.info("[Block Entity Render Data] Subsequent exceptions will be suppressed.");
					}
				}
			}
		}
	}

	@Inject(method = "build", at = @At(value = "RETURN", ordinal = 1))
	private void createDataMap(class_1937 world, class_4076 chunkSectionPos, CallbackInfoReturnable<class_853> cir, @Share("dataMap") LocalRef<Long2ObjectOpenHashMap<Object>> mapRef) {
		class_853 rendererRegion = cir.getReturnValue();
		Long2ObjectOpenHashMap<Object> map = mapRef.get();

		if (map != null) {
			((RenderDataMapConsumer) rendererRegion).fabric_acceptRenderDataMap(map);
		}
	}

	@Unique
	private static Long2ObjectOpenHashMap<Object> mapChunk(class_2818 chunk, class_4076 chunkSectionPos, Long2ObjectOpenHashMap<Object> map) {
		// Skip the math below if the chunk contains no block entities
		if (chunk.method_12214().isEmpty()) {
			return map;
		}

		final int xMin = class_4076.method_18688(chunkSectionPos.method_18674() - 1);
		final int yMin = class_4076.method_18688(chunkSectionPos.method_18683() - 1);
		final int zMin = class_4076.method_18688(chunkSectionPos.method_18687() - 1);
		final int xMax = class_4076.method_18688(chunkSectionPos.method_18674() + 1);
		final int yMax = class_4076.method_18688(chunkSectionPos.method_18683() + 1);
		final int zMax = class_4076.method_18688(chunkSectionPos.method_18687() + 1);

		for (Map.Entry<class_2338, class_2586> entry : chunk.method_12214().entrySet()) {
			final class_2338 pos = entry.getKey();

			if (pos.method_10263() >= xMin && pos.method_10263() <= xMax
					&& pos.method_10264() >= yMin && pos.method_10264() <= yMax
					&& pos.method_10260() >= zMin && pos.method_10260() <= zMax) {
				final Object data = entry.getValue().getRenderData();

				if (data != null) {
					if (map == null) {
						map = new Long2ObjectOpenHashMap<>();
					}

					map.put(pos.method_10063(), data);
				}
			}
		}

		return map;
	}
}
