/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.client.gametest;

import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.function.FailableConsumer;
import org.apache.commons.lang3.function.FailableFunction;
import org.apache.commons.lang3.mutable.MutableBoolean;
import org.apache.commons.lang3.mutable.MutableObject;
import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.api.client.gametest.v1.ClientGameTestContext;
import net.fabricmc.fabric.mixin.client.gametest.CyclingButtonWidgetAccessor;
import net.fabricmc.fabric.mixin.client.gametest.GameOptionsAccessor;
import net.fabricmc.fabric.mixin.client.gametest.ScreenAccessor;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_1157;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_315;
import net.minecraft.class_318;
import net.minecraft.class_339;
import net.minecraft.class_3419;
import net.minecraft.class_4063;
import net.minecraft.class_4068;
import net.minecraft.class_4185;
import net.minecraft.class_4264;
import net.minecraft.class_437;
import net.minecraft.class_5676;
import net.minecraft.class_7172;
import net.minecraft.class_8021;
import net.minecraft.class_8144;

public final class ClientGameTestContextImpl implements ClientGameTestContext {
	private final ClientGameTestInputImpl input = new ClientGameTestInputImpl(this);

	private static final Map<String, Object> DEFAULT_GAME_OPTIONS = new HashMap<>();

	public static void initGameOptions(class_315 options) {
		// Messes with the consistency of gametests
		options.field_1875 = class_1157.field_5653;
		options.method_42528().method_41748(class_4063.field_18162);

		// Messes with game tests starting
		options.field_41785 = false;

		// Just annoying
		options.method_45578(class_3419.field_15253).method_41748(0.0);

		((GameOptionsAccessor) options).invokeAccept(new class_315.class_5823() {
			@Override
			public int method_33680(String key, int current) {
				DEFAULT_GAME_OPTIONS.put(key, current);
				return current;
			}

			@Override
			public boolean method_33684(String key, boolean current) {
				DEFAULT_GAME_OPTIONS.put(key, current);
				return current;
			}

			@Override
			public String method_33683(String key, String current) {
				DEFAULT_GAME_OPTIONS.put(key, current);
				return current;
			}

			@Override
			public float method_33679(String key, float current) {
				DEFAULT_GAME_OPTIONS.put(key, current);
				return current;
			}

			@Override
			public <T> T method_33681(String key, T current, Function<String, T> decoder, Function<T, String> encoder) {
				DEFAULT_GAME_OPTIONS.put(key, current);
				return current;
			}

			@Override
			public <T> void method_42570(String key, class_7172<T> option) {
				DEFAULT_GAME_OPTIONS.put(key, option.method_41753());
			}
		});
	}

	@Override
	public void waitTick() {
		ThreadingImpl.checkOnGametestThread("waitTick");
		ThreadingImpl.runTick();
	}

	@Override
	public void waitTicks(int ticks) {
		ThreadingImpl.checkOnGametestThread("waitTicks");
		Preconditions.checkArgument(ticks >= 0, "ticks cannot be negative");

		for (int i = 0; i < ticks; i++) {
			ThreadingImpl.runTick();
		}
	}

	@Override
	public void waitFor(Predicate<class_310> predicate) {
		ThreadingImpl.checkOnGametestThread("waitFor");
		Preconditions.checkNotNull(predicate, "predicate");
		waitFor(predicate, DEFAULT_TIMEOUT);
	}

	@Override
	public void waitFor(Predicate<class_310> predicate, int timeout) {
		ThreadingImpl.checkOnGametestThread("waitFor");
		Preconditions.checkNotNull(predicate, "predicate");

		if (timeout == NO_TIMEOUT) {
			while (!computeOnClient(predicate::test)) {
				ThreadingImpl.runTick();
			}
		} else {
			Preconditions.checkArgument(timeout > 0, "timeout must be positive");

			for (int i = 0; i < timeout; i++) {
				if (computeOnClient(predicate::test)) {
					return;
				}

				ThreadingImpl.runTick();
			}

			if (!computeOnClient(predicate::test)) {
				throw new AssertionError("Timed out waiting for predicate");
			}
		}
	}

	@Override
	public void waitForScreen(@Nullable Class<? extends class_437> screenClass) {
		ThreadingImpl.checkOnGametestThread("waitForScreen");

		if (screenClass == null) {
			waitFor(client -> client.field_1755 == null);
		} else {
			waitFor(client -> screenClass.isInstance(client.field_1755));
		}
	}

	@Override
	public void setScreen(Supplier<@Nullable class_437> screen) {
		ThreadingImpl.checkOnGametestThread("setScreen");
		runOnClient(client -> client.method_1507(screen.get()));
	}

	@Override
	public void clickScreenButton(String translationKey) {
		ThreadingImpl.checkOnGametestThread("clickScreenButton");
		Preconditions.checkNotNull(translationKey, "translationKey");

		runOnClient(client -> {
			if (!tryClickScreenButtonImpl(client.field_1755, translationKey)) {
				throw new AssertionError("Could not find button '%s' in screen '%s'".formatted(
					translationKey,
					class_8144.method_49077(client.field_1755, screen -> screen.getClass().getName())
				));
			}
		});
	}

	@Override
	public boolean tryClickScreenButton(String translationKey) {
		ThreadingImpl.checkOnGametestThread("tryClickScreenButton");
		Preconditions.checkNotNull(translationKey, "translationKey");

		return computeOnClient(client -> tryClickScreenButtonImpl(client.field_1755, translationKey));
	}

	private static boolean tryClickScreenButtonImpl(@Nullable class_437 screen, String translationKey) {
		if (screen == null) {
			return false;
		}

		final String buttonText = class_2561.method_43471(translationKey).getString();
		final ScreenAccessor screenAccessor = (ScreenAccessor) screen;

		for (class_4068 drawable : screenAccessor.getDrawables()) {
			if (drawable instanceof class_4264 pressableWidget && pressMatchingButton(pressableWidget, buttonText)) {
				return true;
			}

			if (drawable instanceof class_8021 widget) {
				MutableBoolean found = new MutableBoolean(false);
				widget.method_48206(clickableWidget -> {
					if (!found.booleanValue()) {
						found.setValue(pressMatchingButton(clickableWidget, buttonText));
					}
				});

				if (found.booleanValue()) {
					return true;
				}
			}
		}

		// Was unable to find the button to press
		return false;
	}

	private static boolean pressMatchingButton(class_339 widget, String text) {
		if (widget instanceof class_4185 buttonWidget) {
			if (text.equals(buttonWidget.method_25369().getString())) {
				buttonWidget.method_25306();
				return true;
			}
		}

		if (widget instanceof class_5676<?> buttonWidget) {
			CyclingButtonWidgetAccessor accessor = (CyclingButtonWidgetAccessor) buttonWidget;

			if (text.equals(accessor.getOptionText().getString())) {
				buttonWidget.method_25306();
				return true;
			}
		}

		return false;
	}

	@Override
	public Path takeScreenshot(String name) {
		ThreadingImpl.checkOnGametestThread("takeScreenshot");
		Preconditions.checkNotNull(name, "name");
		return takeScreenshot(name, 1);
	}

	@Override
	public Path takeScreenshot(String name, int delay) {
		ThreadingImpl.checkOnGametestThread("takeScreenshot");
		Preconditions.checkNotNull(name, "name");
		Preconditions.checkArgument(delay >= 0, "delay cannot be negative");

		waitTicks(delay);
		runOnClient(client -> {
			class_318.method_22690(FabricLoader.getInstance().getGameDir().toFile(), name + ".png", client.method_1522(), (message) -> {
			});
		});

		return FabricLoader.getInstance().getGameDir().resolve("screenshots").resolve(name + ".png");
	}

	@Override
	public ClientGameTestInputImpl getInput() {
		return input;
	}

	@Override
	public void restoreDefaultGameOptions() {
		ThreadingImpl.checkOnGametestThread("restoreDefaultGameOptions");

		runOnClient(client -> {
			((GameOptionsAccessor) class_310.method_1551().field_1690).invokeAccept(new class_315.class_5823() {
				@Override
				public int method_33680(String key, int current) {
					return (Integer) DEFAULT_GAME_OPTIONS.get(key);
				}

				@Override
				public boolean method_33684(String key, boolean current) {
					return (Boolean) DEFAULT_GAME_OPTIONS.get(key);
				}

				@Override
				public String method_33683(String key, String current) {
					return (String) DEFAULT_GAME_OPTIONS.get(key);
				}

				@Override
				public float method_33679(String key, float current) {
					return (Float) DEFAULT_GAME_OPTIONS.get(key);
				}

				@SuppressWarnings("unchecked")
				@Override
				public <T> T method_33681(String key, T current, Function<String, T> decoder, Function<T, String> encoder) {
					return (T) DEFAULT_GAME_OPTIONS.get(key);
				}

				@SuppressWarnings("unchecked")
				@Override
				public <T> void method_42570(String key, class_7172<T> option) {
					option.method_41748((T) DEFAULT_GAME_OPTIONS.get(key));
				}
			});
		});
	}

	@Override
	public <E extends Throwable> void runOnClient(FailableConsumer<class_310, E> action) throws E {
		ThreadingImpl.checkOnGametestThread("runOnClient");
		Preconditions.checkNotNull(action, "action");

		ThreadingImpl.runOnClient(() -> action.accept(class_310.method_1551()));
	}

	@Override
	public <T, E extends Throwable> T computeOnClient(FailableFunction<class_310, T, E> function) throws E {
		ThreadingImpl.checkOnGametestThread("computeOnClient");
		Preconditions.checkNotNull(function, "function");

		MutableObject<T> result = new MutableObject<>();
		ThreadingImpl.runOnClient(() -> result.setValue(function.apply(class_310.method_1551())));
		return result.getValue();
	}
}
