/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.client.gametest;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;
import java.util.function.Function;

import com.google.common.base.Preconditions;
import org.lwjgl.glfw.GLFW;
import net.fabricmc.fabric.api.client.gametest.v1.ClientGameTestContext;
import net.fabricmc.fabric.api.client.gametest.v1.TestInput;
import net.fabricmc.fabric.mixin.client.gametest.KeyBindingAccessor;
import net.fabricmc.fabric.mixin.client.gametest.KeyboardAccessor;
import net.fabricmc.fabric.mixin.client.gametest.MouseAccessor;
import net.minecraft.class_304;
import net.minecraft.class_310;
import net.minecraft.class_315;
import net.minecraft.class_3675;

public final class TestInputImpl implements TestInput {
	private static final Set<class_3675.class_306> KEYS_DOWN = new HashSet<>();
	private final ClientGameTestContext context;

	public TestInputImpl(ClientGameTestContext context) {
		this.context = context;
	}

	public static boolean isKeyDown(int keyCode) {
		return KEYS_DOWN.contains(class_3675.class_307.field_1668.method_1447(keyCode));
	}

	public void clearKeysDown() {
		for (class_3675.class_306 key : new ArrayList<>(KEYS_DOWN)) {
			releaseKey(key);
		}
	}

	@Override
	public void holdKey(class_304 keyBinding) {
		ThreadingImpl.checkOnGametestThread("holdKey");
		Preconditions.checkNotNull(keyBinding, "keyBinding");

		holdKey(getBoundKey(keyBinding, "hold"));
	}

	@Override
	public void holdKey(Function<class_315, class_304> keyBindingGetter) {
		ThreadingImpl.checkOnGametestThread("holdKey");
		Preconditions.checkNotNull(keyBindingGetter, "keyBindingGetter");

		class_304 keyBinding = context.computeOnClient(client -> keyBindingGetter.apply(client.field_1690));
		holdKey(keyBinding);
	}

	@Override
	public void holdKey(class_3675.class_306 key) {
		ThreadingImpl.checkOnGametestThread("holdKey");
		Preconditions.checkNotNull(key, "key");

		if (KEYS_DOWN.add(key)) {
			context.runOnClient(client -> pressOrReleaseKey(client, key, GLFW.GLFW_PRESS));
		}
	}

	@Override
	public void holdKey(int keyCode) {
		ThreadingImpl.checkOnGametestThread("holdKey");

		holdKey(class_3675.class_307.field_1668.method_1447(keyCode));
	}

	@Override
	public void holdMouse(int button) {
		ThreadingImpl.checkOnGametestThread("holdMouse");

		holdKey(class_3675.class_307.field_1672.method_1447(button));
	}

	@Override
	public void holdControl() {
		ThreadingImpl.checkOnGametestThread("holdControl");

		holdKey(class_310.field_1703 ? class_3675.field_31952 : class_3675.field_31950);
	}

	@Override
	public void holdShift() {
		ThreadingImpl.checkOnGametestThread("holdShift");

		holdKey(class_3675.field_31951);
	}

	@Override
	public void holdAlt() {
		ThreadingImpl.checkOnGametestThread("holdAlt");

		holdKey(class_3675.field_31949);
	}

	@Override
	public void releaseKey(class_304 keyBinding) {
		ThreadingImpl.checkOnGametestThread("releaseKey");
		Preconditions.checkNotNull(keyBinding, "keyBinding");

		releaseKey(getBoundKey(keyBinding, "release"));
	}

	@Override
	public void releaseKey(Function<class_315, class_304> keyBindingGetter) {
		ThreadingImpl.checkOnGametestThread("releaseKey");
		Preconditions.checkNotNull(keyBindingGetter, "keyBindingGetter");

		class_304 keyBinding = context.computeOnClient(client -> keyBindingGetter.apply(client.field_1690));
		releaseKey(keyBinding);
	}

	@Override
	public void releaseKey(class_3675.class_306 key) {
		ThreadingImpl.checkOnGametestThread("releaseKey");
		Preconditions.checkNotNull(key, "key");

		if (KEYS_DOWN.remove(key)) {
			context.runOnClient(client -> pressOrReleaseKey(client, key, GLFW.GLFW_RELEASE));
		}
	}

	@Override
	public void releaseKey(int keyCode) {
		ThreadingImpl.checkOnGametestThread("releaseKey");

		releaseKey(class_3675.class_307.field_1668.method_1447(keyCode));
	}

	@Override
	public void releaseMouse(int button) {
		ThreadingImpl.checkOnGametestThread("releaseMouse");

		releaseKey(class_3675.class_307.field_1672.method_1447(button));
	}

	@Override
	public void releaseControl() {
		ThreadingImpl.checkOnGametestThread("releaseControl");

		releaseKey(class_310.field_1703 ? class_3675.field_31952 : class_3675.field_31950);
	}

	@Override
	public void releaseShift() {
		ThreadingImpl.checkOnGametestThread("releaseShift");

		releaseKey(class_3675.field_31951);
	}

	@Override
	public void releaseAlt() {
		ThreadingImpl.checkOnGametestThread("releaseAlt");

		releaseKey(class_3675.field_31949);
	}

	private static void pressOrReleaseKey(class_310 client, class_3675.class_306 key, int action) {
		switch (key.method_1442()) {
		case field_1668 -> client.field_1774.method_1466(client.method_22683().method_4490(), key.method_1444(), 0, action, 0);
		case field_1671 -> client.field_1774.method_1466(client.method_22683().method_4490(), GLFW.GLFW_KEY_UNKNOWN, key.method_1444(), action, 0);
		case field_1672 -> ((MouseAccessor) client.field_1729).invokeOnMouseButton(client.method_22683().method_4490(), key.method_1444(), action, 0);
		}
	}

	@Override
	public void pressKey(class_304 keyBinding) {
		ThreadingImpl.checkOnGametestThread("pressKey");
		Preconditions.checkNotNull(keyBinding, "keyBinding");

		pressKey(getBoundKey(keyBinding, "press"));
	}

	@Override
	public void pressKey(Function<class_315, class_304> keyBindingGetter) {
		ThreadingImpl.checkOnGametestThread("pressKey");
		Preconditions.checkNotNull(keyBindingGetter, "keyBindingGetter");

		class_304 keyBinding = context.computeOnClient(client -> keyBindingGetter.apply(client.field_1690));
		pressKey(keyBinding);
	}

	@Override
	public void pressKey(class_3675.class_306 key) {
		ThreadingImpl.checkOnGametestThread("pressKey");
		Preconditions.checkNotNull(key, "key");

		holdKey(key);
		releaseKey(key);
	}

	@Override
	public void pressKey(int keyCode) {
		ThreadingImpl.checkOnGametestThread("pressKey");

		pressKey(class_3675.class_307.field_1668.method_1447(keyCode));
	}

	@Override
	public void pressMouse(int button) {
		ThreadingImpl.checkOnGametestThread("pressMouse");

		pressKey(class_3675.class_307.field_1672.method_1447(button));
	}

	@Override
	public void holdKeyFor(class_304 keyBinding, int ticks) {
		ThreadingImpl.checkOnGametestThread("holdKeyFor");
		Preconditions.checkNotNull(keyBinding, "keyBinding");
		Preconditions.checkArgument(ticks > 0, "ticks must be positive");

		holdKeyFor(getBoundKey(keyBinding, "hold"), ticks);
	}

	@Override
	public void holdKeyFor(Function<class_315, class_304> keyBindingGetter, int ticks) {
		ThreadingImpl.checkOnGametestThread("holdKeyFor");
		Preconditions.checkNotNull(keyBindingGetter, "keyBindingGetter");
		Preconditions.checkArgument(ticks > 0, "ticks must be positive");

		class_304 keyBinding = context.computeOnClient(client -> keyBindingGetter.apply(client.field_1690));
		holdKeyFor(keyBinding, ticks);
	}

	@Override
	public void holdKeyFor(class_3675.class_306 key, int ticks) {
		ThreadingImpl.checkOnGametestThread("holdKeyFor");
		Preconditions.checkNotNull(key, "key");
		Preconditions.checkArgument(ticks > 0, "ticks must be positive");

		holdKey(key);
		context.waitTicks(ticks);
		releaseKey(key);
	}

	@Override
	public void holdKeyFor(int keyCode, int ticks) {
		ThreadingImpl.checkOnGametestThread("holdKeyFor");
		Preconditions.checkArgument(ticks > 0, "ticks must be positive");

		holdKeyFor(class_3675.class_307.field_1668.method_1447(keyCode), ticks);
	}

	@Override
	public void holdMouseFor(int button, int ticks) {
		ThreadingImpl.checkOnGametestThread("holdMouseFor");
		Preconditions.checkArgument(ticks > 0, "ticks must be positive");

		holdKeyFor(class_3675.class_307.field_1672.method_1447(button), ticks);
	}

	@Override
	public void typeChar(int codePoint) {
		ThreadingImpl.checkOnGametestThread("typeChar");

		context.runOnClient(client -> ((KeyboardAccessor) client.field_1774).invokeOnChar(client.method_22683().method_4490(), codePoint, 0));
	}

	@Override
	public void typeChars(String chars) {
		ThreadingImpl.checkOnGametestThread("typeChars");

		context.runOnClient(client -> {
			chars.chars().forEach(codePoint -> {
				((KeyboardAccessor) client.field_1774).invokeOnChar(client.method_22683().method_4490(), codePoint, 0);
			});
		});
	}

	@Override
	public void scroll(double amount) {
		ThreadingImpl.checkOnGametestThread("scroll");

		scroll(0, amount);
	}

	@Override
	public void scroll(double xAmount, double yAmount) {
		ThreadingImpl.checkOnGametestThread("scroll");

		context.runOnClient(client -> ((MouseAccessor) client.field_1729).invokeOnMouseScroll(client.method_22683().method_4490(), xAmount, yAmount));
	}

	@Override
	public void setCursorPos(double x, double y) {
		ThreadingImpl.checkOnGametestThread("setCursorPos");

		context.runOnClient(client -> ((MouseAccessor) client.field_1729).invokeOnCursorPos(client.method_22683().method_4490(), x, y));
	}

	@Override
	public void moveCursor(double deltaX, double deltaY) {
		ThreadingImpl.checkOnGametestThread("moveCursor");

		context.runOnClient(client -> {
			double newX = client.field_1729.method_1603() + deltaX;
			double newY = client.field_1729.method_1604() + deltaY;
			((MouseAccessor) client.field_1729).invokeOnCursorPos(client.method_22683().method_4490(), newX, newY);
		});
	}

	private static class_3675.class_306 getBoundKey(class_304 keyBinding, String action) {
		class_3675.class_306 boundKey = ((KeyBindingAccessor) keyBinding).getBoundKey();

		if (boundKey == class_3675.field_16237) {
			throw new AssertionError("Cannot %s binding '%s' because it isn't bound to a key".formatted(action, keyBinding.method_1431()));
		}

		return boundKey;
	}
}
