/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.mixin.client.gametest;

import com.google.common.base.Preconditions;
import com.llamalad7.mixinextras.injector.ModifyExpressionValue;
import com.llamalad7.mixinextras.injector.wrapmethod.WrapMethod;
import com.llamalad7.mixinextras.injector.wrapoperation.Operation;
import com.llamalad7.mixinextras.sugar.Share;
import com.llamalad7.mixinextras.sugar.ref.LocalIntRef;
import org.jetbrains.annotations.Nullable;
import org.spongepowered.asm.mixin.Final;
import org.spongepowered.asm.mixin.Mixin;
import org.spongepowered.asm.mixin.Shadow;
import org.spongepowered.asm.mixin.Unique;
import org.spongepowered.asm.mixin.injection.At;
import org.spongepowered.asm.mixin.injection.Inject;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfo;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfoReturnable;
import net.fabricmc.fabric.impl.client.gametest.FabricClientGameTestRunner;
import net.fabricmc.fabric.impl.client.gametest.NetworkSynchronizer;
import net.fabricmc.fabric.impl.client.gametest.ThreadingImpl;
import net.fabricmc.fabric.impl.client.gametest.WindowHooks;
import net.minecraft.class_1041;
import net.minecraft.class_1255;
import net.minecraft.class_310;
import net.minecraft.class_32;
import net.minecraft.class_3283;
import net.minecraft.class_4071;
import net.minecraft.class_437;
import net.minecraft.class_6904;

@Mixin(class_310.class)
public class MinecraftClientMixin {
	@Unique
	private boolean startedClientGametests = false;
	@Unique
	private boolean inMergedRunTasksLoop = false;
	@Unique
	private Runnable deferredTask = null;

	@Shadow
	@Nullable
	private class_4071 overlay;

	@Shadow
	@Final
	private class_1041 window;

	@WrapMethod(method = "run")
	private void onRun(Operation<Void> original) throws Throwable {
		if (ThreadingImpl.isClientRunning) {
			throw new IllegalStateException("Client is already running");
		}

		ThreadingImpl.isClientRunning = true;
		ThreadingImpl.PHASER.register();

		try {
			original.call();
		} finally {
			deregisterClient();

			if (ThreadingImpl.testFailureException != null) {
				throw ThreadingImpl.testFailureException;
			}
		}
	}

	@Inject(method = "cleanUpAfterCrash", at = @At("HEAD"))
	private void deregisterAfterCrash(CallbackInfo ci) {
		// Deregister a bit earlier than normal to allow for the integrated server to stop without waiting for the client
		ThreadingImpl.setGameCrashed();
		deregisterClient();
	}

	@Inject(method = "tick", at = @At("HEAD"))
	private void onTick(CallbackInfo ci) {
		if (!startedClientGametests && overlay == null) {
			startedClientGametests = true;
			FabricClientGameTestRunner.start();
		}
	}

	@ModifyExpressionValue(method = "render", at = @At(value = "INVOKE", target = "Lnet/minecraft/client/render/RenderTickCounter$Dynamic;beginRenderTick(JZ)I"))
	private int captureTicksPerFrame(int capturedTicksPerFrame, @Share("ticksPerFrame") LocalIntRef ticksPerFrame) {
		// limit the number of ticks in a single frame to 1 (disable the "catch-up" mechanism)
		if (capturedTicksPerFrame > 1) {
			capturedTicksPerFrame = 1;
		}

		ticksPerFrame.set(capturedTicksPerFrame);
		return capturedTicksPerFrame;
	}

	@Inject(method = "render", at = @At(value = "INVOKE", target = "Lnet/minecraft/client/MinecraftClient;runTasks()V"))
	private void preRunTasksHook(CallbackInfo ci) {
		// "merge" multiple possible iterations of runTasks into one block from the point of view of locking
		if (!inMergedRunTasksLoop) {
			inMergedRunTasksLoop = true;
			preRunTasks();
		}

		// we still allow runTasks() to go ahead even when ticksPerFrame is 0, as the results of these tasks won't be
		// observable until the next tick or gametest thread unlock anyway
	}

	@Inject(method = "render", at = @At(value = "INVOKE", target = "Lnet/minecraft/client/MinecraftClient;runTasks()V", shift = At.Shift.AFTER))
	private void postRunTasksHook(CallbackInfo ci, @Share("ticksPerFrame") LocalIntRef ticksPerFrame) {
		// end our "merged" runTasks block if there is going to be a tick this frame
		if (ticksPerFrame.get() > 0) {
			NetworkSynchronizer.CLIENTBOUND.waitForPacketHandlers((class_1255<?>) (Object) this);
			postRunTasks();
			inMergedRunTasksLoop = false;
		}
	}

	@Inject(method = "startIntegratedServer", at = @At("HEAD"), cancellable = true)
	private void deferStartIntegratedServer(class_32.class_5143 session, class_3283 dataPackManager, class_6904 saveLoader, boolean newWorld, CallbackInfo ci) {
		if (ThreadingImpl.taskToRun != null) {
			// don't start the integrated server (which busywaits) inside a task
			deferredTask = () -> class_310.method_1551().method_29610(session, dataPackManager, saveLoader, newWorld);
			ci.cancel();
		}
	}

	@Inject(method = "startIntegratedServer", at = @At(value = "INVOKE", target = "Ljava/lang/Thread;sleep(J)V", remap = false))
	private void onStartIntegratedServerBusyWait(CallbackInfo ci) {
		// give the server a chance to tick too
		preRunTasks();
		postRunTasks();
	}

	@Inject(method = "disconnect(Lnet/minecraft/client/gui/screen/Screen;Z)V", at = @At("HEAD"), cancellable = true)
	private void deferDisconnect(class_437 disconnectionScreen, boolean transferring, CallbackInfo ci) {
		if (class_310.method_1551().method_1576() != null && ThreadingImpl.taskToRun != null) {
			// don't disconnect (which busywaits) inside a task
			deferredTask = () -> class_310.method_1551().method_18096(disconnectionScreen, transferring);
			ci.cancel();
		}
	}

	@Inject(method = "disconnect(Lnet/minecraft/client/gui/screen/Screen;Z)V", at = @At(value = "INVOKE", target = "Lnet/minecraft/client/MinecraftClient;cancelTasks()V"))
	private void onDisconnectCancelTasks(CallbackInfo ci) {
		NetworkSynchronizer.CLIENTBOUND.reset();
	}

	@Inject(method = "disconnect(Lnet/minecraft/client/gui/screen/Screen;Z)V", at = @At(value = "INVOKE", target = "Lnet/minecraft/client/MinecraftClient;render(Z)V", shift = At.Shift.AFTER))
	private void onDisconnectBusyWait(CallbackInfo ci) {
		// give the server a chance to tick too
		preRunTasks();
		postRunTasks();
	}

	@Unique
	private void preRunTasks() {
		if (ThreadingImpl.getCurrentPhase() == ThreadingImpl.PHASE_CLIENT_TASKS) {
			postRunTasks();
		}

		if (!NetworkSynchronizer.DISABLED) {
			ThreadingImpl.enterPhase(ThreadingImpl.PHASE_SERVER_TASKS);
			// server tasks happen here
			ThreadingImpl.enterPhase(ThreadingImpl.PHASE_CLIENT_TASKS);
		}
	}

	@Unique
	private void postRunTasks() {
		ThreadingImpl.clientCanAcceptTasks = true;
		ThreadingImpl.enterPhase(ThreadingImpl.PHASE_TEST);

		if (ThreadingImpl.testThread != null) {
			while (true) {
				try {
					ThreadingImpl.CLIENT_SEMAPHORE.acquire();
				} catch (InterruptedException e) {
					throw new RuntimeException(e);
				}

				if (ThreadingImpl.taskToRun != null) {
					ThreadingImpl.taskToRun.run();
				} else {
					break;
				}
			}
		}

		ThreadingImpl.enterPhase(ThreadingImpl.PHASE_TICK);

		Runnable deferredTask = this.deferredTask;
		this.deferredTask = null;

		if (deferredTask != null) {
			deferredTask.run();
		}
	}

	@Inject(method = "getInstance", at = @At("HEAD"))
	private static void checkThreadOnGetInstance(CallbackInfoReturnable<class_310> cir) {
		Preconditions.checkState(
				Thread.currentThread() != ThreadingImpl.testThread,
				"MinecraftClient.getInstance() cannot be called from the gametest thread. Try using ClientGameTestContext.runOnClient or ClientGameTestContext.computeOnClient"
		);
	}

	@ModifyExpressionValue(method = "render", at = @At(value = "INVOKE", target = "Lnet/minecraft/client/util/Window;hasZeroWidthOrHeight()Z"))
	private boolean hasZeroRealWidthOrHeight(boolean original) {
		WindowHooks windowHooks = (WindowHooks) (Object) window;
		return windowHooks.fabric_getRealFramebufferWidth() == 0 || windowHooks.fabric_getRealFramebufferHeight() == 0;
	}

	@Unique
	private static void deregisterClient() {
		if (ThreadingImpl.isClientRunning) {
			ThreadingImpl.clientCanAcceptTasks = false;
			ThreadingImpl.PHASER.arriveAndDeregister();
			ThreadingImpl.isClientRunning = false;
		}
	}
}
