/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.client.gametest.screenshot;

import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;

import com.google.common.base.Preconditions;
import com.mojang.datafixers.util.Either;
import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.api.client.gametest.v1.screenshot.TestScreenshotComparisonAlgorithm;
import net.fabricmc.fabric.api.client.gametest.v1.screenshot.TestScreenshotComparisonOptions;
import net.fabricmc.fabric.impl.client.gametest.FabricClientGameTestRunner;
import net.minecraft.class_1011;
import net.minecraft.class_768;

public final class TestScreenshotComparisonOptionsImpl extends TestScreenshotCommonOptionsImpl<TestScreenshotComparisonOptions> implements TestScreenshotComparisonOptions {
	private final Either<String, class_1011> templateImage;
	@Nullable
	public String savedFileName;
	public TestScreenshotComparisonAlgorithm algorithm = TestScreenshotComparisonAlgorithm.defaultAlgorithm();
	public boolean grayscale = false;
	@Nullable
	public class_768 region;

	public TestScreenshotComparisonOptionsImpl(String templateImage) {
		this.templateImage = Either.left(templateImage);
	}

	public TestScreenshotComparisonOptionsImpl(class_1011 templateImage) {
		this.templateImage = Either.right(templateImage);
	}

	@Override
	public TestScreenshotComparisonOptions save() {
		return saveWithFileName(getTemplateImagePath());
	}

	@Override
	public TestScreenshotComparisonOptions saveWithFileName(String fileName) {
		Preconditions.checkNotNull(fileName, "fileName");

		this.savedFileName = fileName;
		return this;
	}

	@Override
	public TestScreenshotComparisonOptions withAlgorithm(TestScreenshotComparisonAlgorithm algorithm) {
		Preconditions.checkNotNull(algorithm, "algorithm");

		this.algorithm = algorithm;
		return this;
	}

	@Override
	public TestScreenshotComparisonOptions withGrayscale() {
		this.grayscale = true;

		return this;
	}

	@Override
	public TestScreenshotComparisonOptions withRegion(int x, int y, int width, int height) {
		Preconditions.checkArgument(x >= 0, "x cannot be negative");
		Preconditions.checkArgument(y >= 0, "y cannot be negative");
		Preconditions.checkArgument(width > 0, "width must be positive");
		Preconditions.checkArgument(height > 0, "height must be positive");

		this.region = new class_768(x, y, width, height);
		return this;
	}

	public String getTemplateImagePath() {
		return this.templateImage.left().orElseThrow();
	}

	@Nullable
	public TestScreenshotComparisonAlgorithm.RawImage<byte[]> getGrayscaleTemplateImage() {
		return this.templateImage.map(fileName -> {
			try (class_1011 image = loadNativeImage(fileName)) {
				if (image == null) {
					return null;
				}

				return new TestScreenshotComparisonAlgorithms.RawImageImpl<>(
						image.method_4307(),
						image.method_4323(),
						((NativeImageHooks) (Object) image).fabric_copyPixelsLuminance()
				);
			}
		}, image -> {
			assertNoTransparency(image);
			return TestScreenshotComparisonAlgorithms.RawImageImpl.fromGrayscaleNativeImage(image);
		});
	}

	@Nullable
	public TestScreenshotComparisonAlgorithm.RawImage<int[]> getColorTemplateImage() {
		return this.templateImage.map(fileName -> {
			try (class_1011 image = loadNativeImage(fileName)) {
				if (image == null) {
					return null;
				}

				return new TestScreenshotComparisonAlgorithms.RawImageImpl<>(
						image.method_4307(),
						image.method_4323(),
						((NativeImageHooks) (Object) image).fabric_copyPixelsRgb()
				);
			}
		}, image -> {
			assertNoTransparency(image);
			return TestScreenshotComparisonAlgorithms.RawImageImpl.fromColorNativeImage(image);
		});
	}

	@Nullable
	private static class_1011 loadNativeImage(String templateImagePath) {
		Path filePath = FabricClientGameTestRunner.currentlyRunningGameTest.getProvider()
				.findPath("templates/" + templateImagePath + ".png")
				.orElse(null);

		if (filePath == null) {
			return null;
		}

		try (InputStream stream = Files.newInputStream(filePath)) {
			class_1011 image = class_1011.method_4309(stream);
			assertNoTransparency(image);
			return image;
		} catch (IOException e) {
			throw new UncheckedIOException("Failed to load template image", e);
		}
	}

	private static void assertNoTransparency(class_1011 image) {
		if (!((NativeImageHooks) (Object) image).fabric_isFullyOpaque()) {
			throw new AssertionError("Template image is partially transparent which is not supported");
		}
	}
}
