/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.client.gametest.context;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.function.BiPredicate;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.function.FailableConsumer;
import org.apache.commons.lang3.function.FailableFunction;
import org.apache.commons.lang3.mutable.MutableBoolean;
import org.apache.commons.lang3.mutable.MutableObject;
import org.jetbrains.annotations.Nullable;
import org.joml.Vector2i;
import org.lwjgl.glfw.GLFW;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import net.minecraft.class_1011;
import net.minecraft.class_1157;
import net.minecraft.class_11910;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_315;
import net.minecraft.class_318;
import net.minecraft.class_339;
import net.minecraft.class_3419;
import net.minecraft.class_4063;
import net.minecraft.class_4068;
import net.minecraft.class_4185;
import net.minecraft.class_4264;
import net.minecraft.class_437;
import net.minecraft.class_5676;
import net.minecraft.class_7172;
import net.minecraft.class_768;
import net.minecraft.class_8021;
import net.minecraft.class_8144;
import net.fabricmc.fabric.api.client.gametest.v1.context.ClientGameTestContext;
import net.fabricmc.fabric.api.client.gametest.v1.screenshot.TestScreenshotComparisonAlgorithm;
import net.fabricmc.fabric.api.client.gametest.v1.screenshot.TestScreenshotComparisonOptions;
import net.fabricmc.fabric.api.client.gametest.v1.screenshot.TestScreenshotOptions;
import net.fabricmc.fabric.api.client.gametest.v1.world.TestWorldBuilder;
import net.fabricmc.fabric.impl.client.gametest.TestInputImpl;
import net.fabricmc.fabric.impl.client.gametest.TestSystemProperties;
import net.fabricmc.fabric.impl.client.gametest.screenshot.TestScreenshotCommonOptionsImpl;
import net.fabricmc.fabric.impl.client.gametest.screenshot.TestScreenshotComparisonAlgorithms;
import net.fabricmc.fabric.impl.client.gametest.screenshot.TestScreenshotComparisonOptionsImpl;
import net.fabricmc.fabric.impl.client.gametest.screenshot.TestScreenshotOptionsImpl;
import net.fabricmc.fabric.impl.client.gametest.threading.ThreadingImpl;
import net.fabricmc.fabric.impl.client.gametest.world.TestWorldBuilderImpl;
import net.fabricmc.fabric.mixin.client.gametest.CyclingButtonWidgetAccessor;
import net.fabricmc.fabric.mixin.client.gametest.ScreenAccessor;
import net.fabricmc.fabric.mixin.client.gametest.lifecycle.GameOptionsAccessor;
import net.fabricmc.fabric.mixin.client.gametest.screenshot.RenderTickCounterConstantAccessor;
import net.fabricmc.loader.api.FabricLoader;

public final class ClientGameTestContextImpl implements ClientGameTestContext {
	private static final Logger LOGGER = LoggerFactory.getLogger("fabric-client-gametest-api-v1");

	private final TestInputImpl input = new TestInputImpl(this);
	private static int screenshotCounter = 0;

	private static final Map<String, Object> DEFAULT_GAME_OPTIONS = new HashMap<>();

	public static void initGameOptions(class_315 options) {
		// Messes with the consistency of gametests
		options.field_1875 = class_1157.field_5653;
		options.method_42528().method_41748(class_4063.field_18162);

		// Messes with game tests starting
		options.field_41785 = false;

		// Makes chunk rendering finish sooner
		options.method_42503().method_41748(5);

		// Just annoying
		options.method_45578(class_3419.field_15253).method_41748(0.0);

		((GameOptionsAccessor) options).invokeAccept(new class_315.class_5823() {
			@Override
			public int method_33680(String key, int current) {
				DEFAULT_GAME_OPTIONS.put(key, current);
				return current;
			}

			@Override
			public boolean method_33684(String key, boolean current) {
				DEFAULT_GAME_OPTIONS.put(key, current);
				return current;
			}

			@Override
			public String method_33683(String key, String current) {
				DEFAULT_GAME_OPTIONS.put(key, current);
				return current;
			}

			@Override
			public float method_33679(String key, float current) {
				DEFAULT_GAME_OPTIONS.put(key, current);
				return current;
			}

			@Override
			public <T> T method_33681(String key, T current, Function<String, T> decoder, Function<T, String> encoder) {
				DEFAULT_GAME_OPTIONS.put(key, current);
				return current;
			}

			@Override
			public <T> void method_42570(String key, class_7172<T> option) {
				DEFAULT_GAME_OPTIONS.put(key, option.method_41753());
			}
		});
	}

	@Override
	public void waitTick() {
		ThreadingImpl.checkOnGametestThread("waitTick");
		ThreadingImpl.runTick();
	}

	@Override
	public void waitTicks(int ticks) {
		ThreadingImpl.checkOnGametestThread("waitTicks");
		Preconditions.checkArgument(ticks >= 0, "ticks cannot be negative");

		for (int i = 0; i < ticks; i++) {
			ThreadingImpl.runTick();
		}
	}

	@Override
	public int waitFor(Predicate<class_310> predicate) {
		ThreadingImpl.checkOnGametestThread("waitFor");
		Preconditions.checkNotNull(predicate, "predicate");
		return waitFor(predicate, DEFAULT_TIMEOUT);
	}

	@Override
	public int waitFor(Predicate<class_310> predicate, int timeout) {
		ThreadingImpl.checkOnGametestThread("waitFor");
		Preconditions.checkNotNull(predicate, "predicate");

		if (timeout == NO_TIMEOUT) {
			int ticksWaited = 0;

			while (!computeOnClient(predicate::test)) {
				ticksWaited++;
				ThreadingImpl.runTick();
			}

			return ticksWaited;
		} else {
			Preconditions.checkArgument(timeout > 0, "timeout must be positive");

			for (int i = 0; i < timeout; i++) {
				if (computeOnClient(predicate::test)) {
					return i;
				}

				ThreadingImpl.runTick();
			}

			if (!computeOnClient(predicate::test)) {
				throw new AssertionError("Timed out waiting for predicate");
			}

			return timeout;
		}
	}

	@Override
	public int waitForScreen(@Nullable Class<? extends class_437> screenClass) {
		ThreadingImpl.checkOnGametestThread("waitForScreen");

		if (screenClass == null) {
			return waitFor(client -> client.field_1755 == null);
		} else {
			return waitFor(client -> screenClass.isInstance(client.field_1755));
		}
	}

	@Override
	public void setScreen(Supplier<@Nullable class_437> screen) {
		ThreadingImpl.checkOnGametestThread("setScreen");
		runOnClient(client -> client.method_1507(screen.get()));
	}

	@Override
	public void clickScreenButton(String translationKey) {
		ThreadingImpl.checkOnGametestThread("clickScreenButton");
		Preconditions.checkNotNull(translationKey, "translationKey");

		runOnClient(client -> {
			if (!tryClickScreenButtonImpl(client.field_1755, translationKey)) {
				throw new AssertionError("Could not find button '%s' in screen '%s'".formatted(
					translationKey,
					class_8144.method_49077(client.field_1755, screen -> screen.getClass().getName())
				));
			}
		});
	}

	@Override
	public boolean tryClickScreenButton(String translationKey) {
		ThreadingImpl.checkOnGametestThread("tryClickScreenButton");
		Preconditions.checkNotNull(translationKey, "translationKey");

		return computeOnClient(client -> tryClickScreenButtonImpl(client.field_1755, translationKey));
	}

	private static boolean tryClickScreenButtonImpl(@Nullable class_437 screen, String translationKey) {
		if (screen == null) {
			return false;
		}

		final String buttonText = class_2561.method_43471(translationKey).getString();
		final ScreenAccessor screenAccessor = (ScreenAccessor) screen;

		for (class_4068 drawable : screenAccessor.getDrawables()) {
			if (drawable instanceof class_4264 pressableWidget && pressMatchingButton(pressableWidget, buttonText)) {
				return true;
			}

			if (drawable instanceof class_8021 widget) {
				MutableBoolean found = new MutableBoolean(false);
				widget.method_48206(clickableWidget -> {
					if (!found.booleanValue()) {
						found.setValue(pressMatchingButton(clickableWidget, buttonText));
					}
				});

				if (found.booleanValue()) {
					return true;
				}
			}
		}

		// Was unable to find the button to press
		return false;
	}

	private static boolean pressMatchingButton(class_339 widget, String text) {
		var clickEvent = new class_11910(GLFW.GLFW_KEY_UNKNOWN, 0);

		if (widget instanceof class_4185 buttonWidget) {
			if (text.equals(buttonWidget.method_25369().getString())) {
				buttonWidget.method_25306(clickEvent);
				return true;
			}
		}

		if (widget instanceof class_5676<?> buttonWidget) {
			CyclingButtonWidgetAccessor accessor = (CyclingButtonWidgetAccessor) buttonWidget;

			if (text.equals(accessor.getOptionText().getString())) {
				buttonWidget.method_25306(clickEvent);
				return true;
			}
		}

		return false;
	}

	@Override
	public Path takeScreenshot(TestScreenshotOptions options) {
		ThreadingImpl.checkOnGametestThread("takeScreenshot");
		Preconditions.checkNotNull(options, "options");

		TestScreenshotOptionsImpl optionsImpl = (TestScreenshotOptionsImpl) options;
		return doTakeScreenshot(optionsImpl, screenshot -> saveScreenshot(screenshot, optionsImpl.name, optionsImpl));
	}

	@Override
	public void assertScreenshotEquals(TestScreenshotComparisonOptions options) {
		ThreadingImpl.checkOnGametestThread("assertScreenshotEquals");
		Preconditions.checkNotNull(options, "options");
		doAssertScreenshotContains(options, (haystackImage, needleImage) -> haystackImage.width() == needleImage.width() && haystackImage.height() == needleImage.height());
	}

	@Override
	public Vector2i assertScreenshotContains(TestScreenshotComparisonOptions options) {
		ThreadingImpl.checkOnGametestThread("assertScreenshotContains");
		Preconditions.checkNotNull(options, "options");
		return doAssertScreenshotContains(options, (haystackImage, needleImage) -> true);
	}

	private Vector2i doAssertScreenshotContains(
			TestScreenshotComparisonOptions options,
			BiPredicate<TestScreenshotComparisonAlgorithm.RawImage<?>, TestScreenshotComparisonAlgorithm.RawImage<?>> preCheck
	) {
		TestScreenshotComparisonOptionsImpl optionsImpl = (TestScreenshotComparisonOptionsImpl) options;
		return doTakeScreenshot(optionsImpl, screenshot -> {
			class_768 region = optionsImpl.region == null ? new class_768(0, 0, screenshot.method_4307(), screenshot.method_4323()) : optionsImpl.region;
			Preconditions.checkState(region.method_3321() + region.method_3319() <= screenshot.method_4307() && region.method_3322() + region.method_3320() <= screenshot.method_4323(), "Screenshot comparison region extends outside the screenshot");

			try (class_1011 subScreenshot = new class_1011(region.method_3319(), region.method_3320(), false)) {
				screenshot.method_4300(region.method_3321(), region.method_3322(), region.method_3319(), region.method_3320(), subScreenshot);

				if (optionsImpl.savedFileName != null) {
					saveScreenshot(subScreenshot, optionsImpl.savedFileName, optionsImpl);
				}

				Vector2i result;

				if (optionsImpl.grayscale) {
					TestScreenshotComparisonAlgorithm.RawImage<byte[]> templateImage = optionsImpl.getGrayscaleTemplateImage();

					if (templateImage == null) {
						onTemplateImageDoesntExist(subScreenshot, optionsImpl);
						return new Vector2i(region.method_3321(), region.method_3322());
					}

					TestScreenshotComparisonAlgorithm.RawImage<byte[]> haystackImage = TestScreenshotComparisonAlgorithms.RawImageImpl.fromGrayscaleNativeImage(subScreenshot);

					if (preCheck.test(haystackImage, templateImage)) {
						result = optionsImpl.algorithm.findGrayscale(haystackImage, templateImage);
					} else {
						result = null;
					}
				} else {
					TestScreenshotComparisonAlgorithm.RawImage<int[]> templateImage = optionsImpl.getColorTemplateImage();

					if (templateImage == null) {
						onTemplateImageDoesntExist(subScreenshot, optionsImpl);
						return new Vector2i(region.method_3321(), region.method_3322());
					}

					TestScreenshotComparisonAlgorithm.RawImage<int[]> haystackImage = TestScreenshotComparisonAlgorithms.RawImageImpl.fromColorNativeImage(subScreenshot);

					if (preCheck.test(haystackImage, templateImage)) {
						result = optionsImpl.algorithm.findColor(haystackImage, templateImage);
					} else {
						result = null;
					}
				}

				if (result == null) {
					throw new AssertionError("Screenshot does not contain template" + optionsImpl.getTemplateImagePath().map(" '%s'"::formatted).orElse(""));
				}

				return result.add(region.method_3321(), region.method_3322());
			}
		});
	}

	private <T> T doTakeScreenshot(TestScreenshotCommonOptionsImpl<?> options, Function<class_1011, T> screenshotConsumer) {
		ThreadingImpl.checkOnGametestThread("doTakeScreenshot");

		Vector2i prevSize = computeOnClient(client -> {
			int prevWidth = client.method_22683().method_4489();
			int prevHeight = client.method_22683().method_4506();

			if (options.size != null) {
				client.method_22683().method_35642(options.size.x);
				client.method_22683().method_35643(options.size.y);
				client.method_1522().method_1234(options.size.x, options.size.y);
			}

			return new Vector2i(prevWidth, prevHeight);
		});

		try {
			CompletableFuture<T> future = computeOnClient(client -> {
				client.field_1773.method_3192(RenderTickCounterConstantAccessor.create(options.tickDelta), true);
				CompletableFuture<T> resultFuture = new CompletableFuture<>();

				class_318.method_1663(client.method_1522(), screenshot -> {
					try {
						resultFuture.complete(screenshotConsumer.apply(screenshot));
					} catch (Throwable e) {
						resultFuture.completeExceptionally(e);
					}
				});

				return resultFuture;
			});

			// Keep ticking until the screenshot is done
			while (!future.isDone()) {
				waitTick();
			}

			return future.get();
		} catch (ExecutionException | InterruptedException e) {
			throw new RuntimeException(e);
		} finally {
			if (options.size != null) {
				computeOnClient(client -> {
					client.method_22683().method_35642(prevSize.x);
					client.method_22683().method_35643(prevSize.y);
					client.method_1522().method_1234(prevSize.x, prevSize.y);
					return null;
				});
			}
		}
	}

	private static Path saveScreenshot(class_1011 screenshot, String fileName, TestScreenshotCommonOptionsImpl<?> options) {
		Path destinationDir = Objects.requireNonNullElseGet(options.destinationDir, () -> FabricLoader.getInstance().getGameDir().resolve("screenshots"));

		try {
			Files.createDirectories(destinationDir);
		} catch (IOException e) {
			throw new AssertionError("Failed to create screenshots directory", e);
		}

		String counterPrefix = options.counterPrefix ? String.format(Locale.ROOT, "%04d_", screenshotCounter++) : "";
		Path screenshotFile = destinationDir.resolve(counterPrefix + fileName + ".png");

		try {
			screenshot.method_4314(screenshotFile);
		} catch (IOException e) {
			throw new AssertionError("Failed to write screenshot file", e);
		}

		return screenshotFile;
	}

	private static void onTemplateImageDoesntExist(class_1011 subScreenshot, TestScreenshotComparisonOptionsImpl options) {
		if (TestSystemProperties.TEST_MOD_RESOURCES_PATH != null) {
			Path savePath = Path.of(TestSystemProperties.TEST_MOD_RESOURCES_PATH).resolve("templates").resolve(options.getTemplateImagePathOrThrow() + ".png");

			try {
				Files.createDirectories(savePath.getParent());
				subScreenshot.method_4314(savePath);
			} catch (IOException e) {
				throw new AssertionError("Failed to write screenshot file", e);
			}

			LOGGER.info("Written absent screenshot template to {}", savePath);
		} else {
			LOGGER.error("The template image does not exist. Set the fabric.client.gametest.testModResourcesPath system property to your test mod resources file path to automatically save it");
			throw new AssertionError("Template image does not exist");
		}
	}

	@Override
	public TestInputImpl getInput() {
		return input;
	}

	@Override
	public TestWorldBuilder worldBuilder() {
		return new TestWorldBuilderImpl(this);
	}

	@Override
	public void restoreDefaultGameOptions() {
		ThreadingImpl.checkOnGametestThread("restoreDefaultGameOptions");

		runOnClient(client -> {
			((GameOptionsAccessor) class_310.method_1551().field_1690).invokeAccept(new class_315.class_5823() {
				@Override
				public int method_33680(String key, int current) {
					return (Integer) DEFAULT_GAME_OPTIONS.get(key);
				}

				@Override
				public boolean method_33684(String key, boolean current) {
					return (Boolean) DEFAULT_GAME_OPTIONS.get(key);
				}

				@Override
				public String method_33683(String key, String current) {
					return (String) DEFAULT_GAME_OPTIONS.get(key);
				}

				@Override
				public float method_33679(String key, float current) {
					return (Float) DEFAULT_GAME_OPTIONS.get(key);
				}

				@SuppressWarnings("unchecked")
				@Override
				public <T> T method_33681(String key, T current, Function<String, T> decoder, Function<T, String> encoder) {
					return (T) DEFAULT_GAME_OPTIONS.get(key);
				}

				@SuppressWarnings("unchecked")
				@Override
				public <T> void method_42570(String key, class_7172<T> option) {
					option.method_41748((T) DEFAULT_GAME_OPTIONS.get(key));
				}
			});
		});
	}

	@Override
	public <E extends Throwable> void runOnClient(FailableConsumer<class_310, E> action) throws E {
		ThreadingImpl.checkOnGametestThread("runOnClient");
		Preconditions.checkNotNull(action, "action");

		ThreadingImpl.runOnClient(() -> action.accept(class_310.method_1551()));
	}

	@Override
	public <T, E extends Throwable> T computeOnClient(FailableFunction<class_310, T, E> function) throws E {
		ThreadingImpl.checkOnGametestThread("computeOnClient");
		Preconditions.checkNotNull(function, "function");

		MutableObject<T> result = new MutableObject<>();
		ThreadingImpl.runOnClient(() -> result.setValue(function.apply(class_310.method_1551())));
		return result.getValue();
	}
}
