/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.registry;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import net.fabricmc.fabric.api.registry.FlammableBlockRegistry;
import net.fabricmc.fabric.api.resource.ResourceManagerHelper;
import net.fabricmc.fabric.api.resource.ResourceReloadListenerKeys;
import net.fabricmc.fabric.api.resource.SimpleSynchronousResourceReloadListener;
import net.minecraft.class_2248;
import net.minecraft.class_2960;
import net.minecraft.class_3264;
import net.minecraft.class_3300;
import net.minecraft.class_3494;

public class FlammableBlockRegistryImpl implements FlammableBlockRegistry, SimpleSynchronousResourceReloadListener {
	private static final FlammableBlockRegistry.Entry REMOVED = new FlammableBlockRegistry.Entry(0, 0);
	private static final Map<class_2248, FlammableBlockRegistryImpl> REGISTRIES = new HashMap<>();
	private static final Collection<class_2960> RELOAD_DEPS = Collections.singletonList(ResourceReloadListenerKeys.TAGS);
	private static int idCounter = 0;

	private final Map<class_2248, FlammableBlockRegistry.Entry> registeredEntriesBlock = new HashMap<>();
	private final Map<class_3494<class_2248>, FlammableBlockRegistry.Entry> registeredEntriesTag = new HashMap<>();
	private final Map<class_2248, FlammableBlockRegistry.Entry> computedEntries = new HashMap<>();
	private final class_2960 id;
	private final class_2248 key;
	private boolean tagsPresent = false;

	private FlammableBlockRegistryImpl(class_2248 key) {
		ResourceManagerHelper.get(class_3264.field_14190).registerReloadListener(this);
		this.id = new class_2960("fabric:private/fire_registry_" + (++idCounter));
		this.key = key;
	}

	// TODO: Asynchronous?
	@Override
	public void method_14491(class_3300 var1) {
		reload();
		tagsPresent = true;
	}

	private void reload() {
		computedEntries.clear();

		// tags take precedence before blocks
		for (class_3494<class_2248> tag : registeredEntriesTag.keySet()) {
			FlammableBlockRegistry.Entry entry = registeredEntriesTag.get(tag);

			for (class_2248 block : tag.method_15138()) {
				computedEntries.put(block, entry);
			}
		}

		computedEntries.putAll(registeredEntriesBlock);

		/* computedBurnChances.clear();
		computedSpreadChances.clear();

		for (Block block : computedEntries.keySet()) {
			FlammableBlockRegistry.Entry entry = computedEntries.get(block);
			computedBurnChances.put(block, entry.getBurnChance());
			computedSpreadChances.put(block, entry.getSpreadChance());
		} */
	}

	// User-facing fire registry interface - queries vanilla fire block
	@Override
	public Entry get(class_2248 block) {
		Entry entry = computedEntries.get(block);

		if (entry != null) {
			return entry;
		} else {
			return ((FireBlockHooks) key).fabric_getVanillaEntry(block.method_9564());
		}
	}

	public Entry getFabric(class_2248 block) {
		return computedEntries.get(block);
	}

	@Override
	public void add(class_2248 block, Entry value) {
		registeredEntriesBlock.put(block, value);

		if (tagsPresent) {
			reload();
		}
	}

	@Override
	public void add(class_3494<class_2248> tag, Entry value) {
		registeredEntriesTag.put(tag, value);

		if (tagsPresent) {
			reload();
		}
	}

	@Override
	public void remove(class_2248 block) {
		add(block, REMOVED);
	}

	@Override
	public void remove(class_3494<class_2248> tag) {
		add(tag, REMOVED);
	}

	@Override
	public void clear(class_2248 block) {
		registeredEntriesBlock.remove(block);

		if (tagsPresent) {
			reload();
		}
	}

	@Override
	public void clear(class_3494<class_2248> tag) {
		registeredEntriesTag.remove(tag);

		if (tagsPresent) {
			reload();
		}
	}

	public static FlammableBlockRegistryImpl getInstance(class_2248 block) {
		if (!(block instanceof FireBlockHooks)) {
			throw new RuntimeException("Not a hookable fire block: " + block);
		}

		return REGISTRIES.computeIfAbsent(block, FlammableBlockRegistryImpl::new);
	}

	@Override
	public class_2960 getFabricId() {
		return id;
	}

	@Override
	public Collection<class_2960> getFabricDependencies() {
		return RELOAD_DEPS;
	}
}
