/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.content.registry;

import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.Map;
import net.fabricmc.fabric.api.event.lifecycle.v1.CommonLifecycleEvents;
import net.fabricmc.fabric.api.registry.FlammableBlockRegistry;
import net.minecraft.class_2248;
import net.minecraft.class_6862;
import net.minecraft.class_6880;
import net.minecraft.class_7923;

public class FlammableBlockRegistryImpl implements FlammableBlockRegistry {
	private static final FlammableBlockRegistry.Entry REMOVED = new FlammableBlockRegistry.Entry(0, 0);
	private static final Map<class_2248, FlammableBlockRegistryImpl> REGISTRIES = new HashMap<>();

	private final Map<class_2248, FlammableBlockRegistry.Entry> registeredEntriesBlock = new HashMap<>();
	private final Map<class_6862<class_2248>, FlammableBlockRegistry.Entry> registeredEntriesTag = new HashMap<>();
	private volatile Map<class_2248, FlammableBlockRegistry.Entry> computedEntries = null;
	private final class_2248 key;

	private FlammableBlockRegistryImpl(class_2248 key) {
		this.key = key;

		// Reset computed values after tags change since they depend on tags.
		CommonLifecycleEvents.TAGS_LOADED.register((registries, client) -> {
			computedEntries = null;
		});
	}

	private Map<class_2248, FlammableBlockRegistry.Entry> getEntryMap() {
		Map<class_2248, FlammableBlockRegistry.Entry> ret = computedEntries;

		if (ret == null) {
			ret = new IdentityHashMap<>();

			// tags take precedence over blocks
			for (class_6862<class_2248> tag : registeredEntriesTag.keySet()) {
				FlammableBlockRegistry.Entry entry = registeredEntriesTag.get(tag);

				for (class_6880<class_2248> block : class_7923.field_41175.method_40286(tag)) {
					ret.put(block.comp_349(), entry);
				}
			}

			ret.putAll(registeredEntriesBlock);

			computedEntries = ret;
		}

		return ret;
	}

	// User-facing fire registry interface - queries vanilla fire block
	@Override
	public Entry get(class_2248 block) {
		Entry entry = getEntryMap().get(block);

		if (entry != null) {
			return entry;
		} else {
			return ((FireBlockHooks) key).fabric_getVanillaEntry(block.method_9564());
		}
	}

	public Entry getFabric(class_2248 block) {
		return getEntryMap().get(block);
	}

	@Override
	public void add(class_2248 block, Entry value) {
		registeredEntriesBlock.put(block, value);

		computedEntries = null;
	}

	@Override
	public void add(class_6862<class_2248> tag, Entry value) {
		registeredEntriesTag.put(tag, value);

		computedEntries = null;
	}

	@Override
	public void remove(class_2248 block) {
		add(block, REMOVED);
	}

	@Override
	public void remove(class_6862<class_2248> tag) {
		add(tag, REMOVED);
	}

	@Override
	public void clear(class_2248 block) {
		registeredEntriesBlock.remove(block);

		computedEntries = null;
	}

	@Override
	public void clear(class_6862<class_2248> tag) {
		registeredEntriesTag.remove(tag);

		computedEntries = null;
	}

	public static FlammableBlockRegistryImpl getInstance(class_2248 block) {
		if (!(block instanceof FireBlockHooks)) {
			throw new RuntimeException("Not a hookable fire block: " + block);
		}

		return REGISTRIES.computeIfAbsent(block, FlammableBlockRegistryImpl::new);
	}
}
