/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.registry;

import org.jspecify.annotations.Nullable;
import net.fabricmc.fabric.impl.content.registry.StrippableBlockRegistryImpl;
import net.minecraft.class_2248;
import net.minecraft.class_2350;
import net.minecraft.class_2680;
import net.minecraft.class_2741;
import net.minecraft.class_2769;

/**
 * A registry for axe stripping interactions. A vanilla example is turning logs to stripped logs.
 */
public final class StrippableBlockRegistry {
	private StrippableBlockRegistry() {
	}

	/**
	 * Registers a stripping interaction.
	 * The resulting BlockState of stripping of input will only copy the {@link class_2741#field_12496 axis} property, if it's present.
	 *
	 * @param input    the input block that can be stripped
	 * @param stripped the stripped result block
	 */
	public static void register(class_2248 input, class_2248 stripped) {
		StrippingTransformer transformer;

		if (input.method_9564().method_28498(class_2741.field_12496) && stripped.method_9564().method_28498(class_2741.field_12496)) {
			transformer = StrippingTransformer.VANILLA;
		} else {
			transformer = StrippingTransformer.DEFAULT_STATE;
		}

		StrippableBlockRegistryImpl.register(input, stripped, transformer);
	}

	/**
	 * Registers a stripping interaction.
	 * The resulting BlockState of stripping of input will copy all present properties.
	 *
	 * @param input    the input block that can be stripped
	 * @param stripped the stripped result block
	 */
	public static void registerCopyState(class_2248 input, class_2248 stripped) {
		StrippableBlockRegistryImpl.register(input, stripped, StrippingTransformer.COPY);
	}

	/**
	 * Registers a stripping interaction.
	 * The resulting BlockState of stripping of input will depend on provided transformer.
	 *
	 * @param input       the input block that can be stripped
	 * @param stripped    the stripped result block
	 * @param transformer the transformer used to provide the resulting block state
	 */
	public static void register(class_2248 input, class_2248 stripped, StrippingTransformer transformer) {
		StrippableBlockRegistryImpl.register(input, stripped, transformer);
	}

	/**
	 * Provides result of stripping interaction.
	 *
	 * @param blockState original block state
	 * @return stripped block state if successful, otherwise null
	 */
	@Nullable
	public static class_2680 getStrippedBlockState(class_2680 blockState) {
		return StrippableBlockRegistryImpl.getStrippedBlockState(blockState);
	}

	public interface StrippingTransformer {
		StrippingTransformer DEFAULT_STATE = (strippedBlock, originalState) -> strippedBlock.method_9564();
		StrippingTransformer VANILLA = (strippedBlock, originalState) -> strippedBlock.method_9564().method_47968(class_2741.field_12496, originalState.method_61767(class_2741.field_12496, class_2350.class_2351.field_11052));
		StrippingTransformer COPY = class_2248::method_34725;

		@Nullable
		class_2680 getStrippedBlockState(class_2248 strippedBlock, class_2680 originalState);

		static StrippingTransformer copyOf(class_2769<?>... properties) {
			if (properties.length == 0) {
				return DEFAULT_STATE;
			}

			if (properties.length == 1 && properties[0] == class_2741.field_12496) {
				return VANILLA;
			}

			return ((strippedBlock, originalState) -> {
				class_2680 state = strippedBlock.method_9564();

				//noinspection rawtypes
				for (class_2769 property : properties) {
					if (originalState.method_28498(property)) {
						//noinspection unchecked
						state = state.method_47968(property, originalState.method_11654(property));
					}
				}

				return state;
			});
		}
	}
}
