/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.attachment;

import java.util.IdentityHashMap;
import java.util.Map;

import com.mojang.serialization.Codec;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import net.fabricmc.fabric.api.attachment.v1.AttachmentTarget;
import net.fabricmc.fabric.api.attachment.v1.AttachmentType;
import net.minecraft.class_2487;
import net.minecraft.class_2509;
import net.minecraft.class_2520;
import net.minecraft.class_2960;
import net.minecraft.class_6903;
import net.minecraft.class_7225;

public class AttachmentSerializingImpl {
	private static final Logger LOGGER = LoggerFactory.getLogger("fabric-data-attachment-api-v1");

	@SuppressWarnings("unchecked")
	public static void serializeAttachmentData(class_2487 nbt, class_7225.class_7874 wrapperLookup, @Nullable IdentityHashMap<AttachmentType<?>, ?> attachments) {
		if (attachments == null || attachments.isEmpty()) {
			return;
		}

		var compound = new class_2487();

		for (Map.Entry<AttachmentType<?>, ?> entry : attachments.entrySet()) {
			AttachmentType<?> type = entry.getKey();
			Codec<Object> codec = (Codec<Object>) type.persistenceCodec();

			if (codec != null) {
				class_6903<class_2520> registryOps = class_6903.method_46632(class_2509.field_11560, wrapperLookup);
				codec.encodeStart(registryOps, entry.getValue())
						.get()
						.ifRight(partial -> {
							LOGGER.warn("Couldn't serialize attachment " + type.identifier() + ", skipping. Error:");
							LOGGER.warn(partial.message());
						})
						.ifLeft(serialized -> compound.method_10566(type.identifier().toString(), serialized));
			}
		}

		nbt.method_10566(AttachmentTarget.NBT_ATTACHMENT_KEY, compound);
	}

	@Nullable
	public static IdentityHashMap<AttachmentType<?>, Object> deserializeAttachmentData(class_2487 nbt, class_7225.class_7874 wrapperLookup) {
		if (nbt.method_10573(AttachmentTarget.NBT_ATTACHMENT_KEY, class_2520.field_33260)) {
			var attachments = new IdentityHashMap<AttachmentType<?>, Object>();
			class_2487 compound = nbt.method_10562(AttachmentTarget.NBT_ATTACHMENT_KEY);

			for (String key : compound.method_10541()) {
				AttachmentType<?> type = AttachmentRegistryImpl.get(new class_2960(key));

				if (type == null) {
					LOGGER.warn("Unknown attachment type " + key + " found when deserializing, skipping");
					continue;
				}

				Codec<?> codec = type.persistenceCodec();

				if (codec != null) {
					class_6903<class_2520> registryOps = class_6903.method_46632(class_2509.field_11560, wrapperLookup);
					codec.parse(registryOps, compound.method_10580(key))
							.get()
							.ifRight(partial -> {
								LOGGER.warn("Couldn't deserialize attachment " + type.identifier() + ", skipping. Error:");
								LOGGER.warn(partial.message());
							})
							.ifLeft(
									deserialized -> attachments.put(type, deserialized)
							);
				}
			}

			if (attachments.isEmpty()) {
				return null;
			}

			return attachments;
		}

		return null;
	}

	public static boolean hasPersistentAttachments(@Nullable IdentityHashMap<AttachmentType<?>, ?> map) {
		if (map == null) {
			return false;
		}

		for (AttachmentType<?> type : map.keySet()) {
			if (type.isPersistent()) {
				return true;
			}
		}

		return false;
	}
}
