/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.datagen.v1.provider;

import java.nio.file.Path;
import java.util.Objects;
import java.util.function.Function;

import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.api.datagen.v1.FabricDataGenerator;
import net.minecraft.class_1299;
import net.minecraft.class_1792;
import net.minecraft.class_2248;
import net.minecraft.class_2378;
import net.minecraft.class_2405;
import net.minecraft.class_2474;
import net.minecraft.class_2960;
import net.minecraft.class_3494;
import net.minecraft.class_3611;
import net.minecraft.class_5712;

/**
 * Implement this class (or one of the inner classes) to generate a tag list.
 *
 * <p>Register your implementation using {@link FabricDataGenerator#addProvider} in a {@link net.fabricmc.fabric.api.datagen.v1.DataGeneratorEntrypoint}
 *
 * <p>Commonly used implementations of this class are provided:
 *
 * @see BlockTagProvider
 * @see ItemTagProvider
 * @see FluidTagProvider
 * @see EntityTypeTagProvider
 * @see GameEventTagProvider
 */
public abstract class FabricTagProvider<T> extends class_2474<T> {
	private final String path;
	private final String name;

	/**
	 * Construct a new {@link FabricTagProvider}.
	 *
	 * <p>Common implementations of this class are provided. For example @see BlockTagProvider
	 *
	 * @param dataGenerator The data generator instance
	 * @param registry The backing registry for the Tag type.
	 * @param path The directory name to write the tag file names. Example: "blocks" or "items"
	 * @param name The name used for {@link class_2405#method_10321()}
	 */
	protected FabricTagProvider(FabricDataGenerator dataGenerator, class_2378<T> registry, String path, String name) {
		super(dataGenerator, registry);
		this.path = path;
		this.name = name;
	}

	/**
	 * Implement this method and then use {@link FabricTagProvider#method_10512} to get and register new tag builders.
	 */
	protected abstract void generateTags();

	/**
	 * Creates a new instance of {@link FabricTagBuilder} for the given {@link net.minecraft.class_3494.class_5123} tag.
	 *
	 * @param tag The {@link net.minecraft.class_3494.class_5123} tag to create the builder for
	 * @return The {@link FabricTagBuilder} instance
	 */
	@Override
	protected FabricTagBuilder<T> method_10512(class_3494.class_5123<T> tag) {
		return new FabricTagBuilder<>(super.method_10512(tag));
	}

	@Override
	protected Path method_10510(class_2960 id) {
		return this.field_11483.method_10313().resolve("data/%s/tags/%s/%s.json".formatted(id.method_12836(), path, id.method_12832()));
	}

	@Override
	protected final void method_10514() {
		generateTags();
	}

	@Override
	public String method_10321() {
		return name;
	}

	/**
	 * Extend this class to create {@link class_2248} tags in the "/blocks" tag directory.
	 */
	public abstract static class BlockTagProvider extends FabricTagProvider<class_2248> {
		public BlockTagProvider(FabricDataGenerator dataGenerator) {
			super(dataGenerator, class_2378.field_11146, "blocks", "Block Tags");
		}
	}

	/**
	 * Extend this class to create {@link class_1792} tags in the "/items" tag directory.
	 */
	public abstract static class ItemTagProvider extends FabricTagProvider<class_1792> {
		@Nullable
		private final Function<class_3494.class_5123<class_2248>, class_3494.class_3495> blockTagBuilderProvider;

		/**
		 * Construct an {@link ItemTagProvider} tag provider <b>with</b> an associated {@link BlockTagProvider} tag provider.
		 *
		 * @param dataGenerator a {@link ItemTagProvider} tag provider
		 */
		public ItemTagProvider(FabricDataGenerator dataGenerator, @Nullable FabricTagProvider.BlockTagProvider blockTagProvider) {
			super(dataGenerator, class_2378.field_11142, "items", "Item Tags");

			this.blockTagBuilderProvider = blockTagProvider == null ? null : blockTagProvider::method_27169;
		}

		/**
		 * Construct an {@link ItemTagProvider} tag provider <b>without</b> an associated {@link BlockTagProvider} tag provider.
		 *
		 * @param dataGenerator a {@link ItemTagProvider} tag provider
		 */
		public ItemTagProvider(FabricDataGenerator dataGenerator) {
			this(dataGenerator, null);
		}

		/**
		 * Copy the entries from a tag with the {@link class_2248} type into this item tag.
		 *
		 * <p>The {@link ItemTagProvider} tag provider must be constructed with an associated {@link BlockTagProvider} tag provider to use this method.
		 *
		 * @param blockTag The block tag to copy from.
		 * @param itemTag The item tag to copy to.
		 */
		public void copy(class_3494.class_5123<class_2248> blockTag, class_3494.class_5123<class_1792> itemTag) {
			class_3494.class_3495 blockTagBuilder = Objects.requireNonNull(this.blockTagBuilderProvider, "Pass Block tag provider via constructor to use copy").apply(blockTag);
			class_3494.class_3495 itemTagBuilder = this.method_27169(itemTag);
			blockTagBuilder.method_26785().forEach(itemTagBuilder::method_27064);
		}
	}

	/**
	 * Extend this class to create {@link class_3611} tags in the "/fluids" tag directory.
	 */
	public abstract static class FluidTagProvider extends FabricTagProvider<class_3611> {
		public FluidTagProvider(FabricDataGenerator dataGenerator) {
			super(dataGenerator, class_2378.field_11154, "fluids", "Fluid Tags");
		}
	}

	/**
	 * Extend this class to create {@link class_1299} tags in the "/entity_types" tag directory.
	 */
	public abstract static class EntityTypeTagProvider extends FabricTagProvider<class_1299<?>> {
		public EntityTypeTagProvider(FabricDataGenerator dataGenerator) {
			super(dataGenerator, class_2378.field_11145, "entity_types", "Entity Type Tags");
		}
	}

	/**
	 * Extend this class to create {@link class_5712} tags in the "/game_events" tag directory.
	 */
	public abstract static class GameEventTagProvider extends FabricTagProvider<class_5712> {
		public GameEventTagProvider(FabricDataGenerator dataGenerator) {
			super(dataGenerator, class_2378.field_28264, "game_events", "Game Event Tags");
		}
	}

	/**
	 * An extension to {@link net.minecraft.class_2474.class_5124} that provides additional functionality.
	 */
	public static class FabricTagBuilder<T> extends class_5124<T> {
		private final class_2474.class_5124<T> parent;

		private FabricTagBuilder(class_5124<T> parent) {
			super(parent.field_23960, parent.field_23961, parent.field_23962);
			this.parent = parent;
		}

		/**
		 * Set the value of the `replace` flag in a Tag.
		 *
		 * <p>When set to true the tag will replace any existing tag entries.
		 *
		 * @return the {@link FabricTagBuilder} instance
		 */
		public FabricTagBuilder<T> setReplace(boolean replace) {
			((net.fabricmc.fabric.impl.datagen.FabricTagBuilder) field_23960).fabric_setReplace(replace);
			return this;
		}

		/**
		 * Add a single element to the tag.
		 *
		 * @return the {@link FabricTagBuilder} instance
		 */
		@Override
		public FabricTagBuilder<T> method_26793(T element) {
			parent.method_26793(element);
			return this;
		}

		/**
		 * Add an optional {@link class_2960} to the tag.
		 *
		 * @return the {@link FabricTagBuilder} instance
		 */
		@Override
		public FabricTagBuilder<T> method_35922(class_2960 id) {
			parent.method_35922(id);
			return this;
		}

		/**
		 * Add another tag to this tag.
		 *
		 * @return the {@link FabricTagBuilder} instance
		 */
		@Override
		public FabricTagBuilder<T> method_26792(class_3494.class_5123<T> tag) {
			parent.method_26792(tag);
			return this;
		}

		/**
		 * Add another optional tag to this tag.
		 *
		 * @return the {@link FabricTagBuilder} instance
		 */
		@Override
		public FabricTagBuilder<T> method_35923(class_2960 id) {
			parent.method_35923(id);
			return this;
		}
	}
}
