/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.datagen.v1.provider;

import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Stream;

import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.api.datagen.v1.FabricDataGenerator;
import net.fabricmc.fabric.api.datagen.v1.FabricDataOutput;
import net.fabricmc.fabric.impl.datagen.ForcedTagEntry;
import net.minecraft.class_1299;
import net.minecraft.class_1792;
import net.minecraft.class_1887;
import net.minecraft.class_2248;
import net.minecraft.class_2378;
import net.minecraft.class_2474;
import net.minecraft.class_2960;
import net.minecraft.class_3481;
import net.minecraft.class_3483;
import net.minecraft.class_3486;
import net.minecraft.class_3489;
import net.minecraft.class_3495;
import net.minecraft.class_3497;
import net.minecraft.class_3611;
import net.minecraft.class_5321;
import net.minecraft.class_5698;
import net.minecraft.class_5712;
import net.minecraft.class_6862;
import net.minecraft.class_6880;
import net.minecraft.class_7225;
import net.minecraft.class_7923;
import net.minecraft.class_7924;

/**
 * Implement this class (or one of the inner classes) to generate a tag list.
 *
 * <p>Register your implementation using {@link FabricDataGenerator.Pack#addProvider} in a {@link net.fabricmc.fabric.api.datagen.v1.DataGeneratorEntrypoint}
 *
 * <p>Commonly used implementations of this class are provided:
 *
 * @see BlockTagProvider
 * @see ItemTagProvider
 * @see FluidTagProvider
 * @see EntityTypeTagProvider
 * @see GameEventTagProvider
 */
public abstract class FabricTagProvider<T> extends class_2474<T> {
	/**
	 * Construct a new {@link FabricTagProvider} with the default computed path.
	 *
	 * <p>Common implementations of this class are provided. For example @see BlockTagProvider
	 *
	 * @param output        The {@link FabricDataOutput} instance
	 * @param registriesFuture      The backing registry for the Tag type.
	 */
	public FabricTagProvider(FabricDataOutput output, class_5321<? extends class_2378<T>> registryKey, CompletableFuture<class_7225.class_7874> registriesFuture) {
		super(output, registryKey, registriesFuture);
	}

	/**
	 * Implement this method and then use {@link FabricTagProvider#method_10512} to get and register new tag builders.
	 */
	protected abstract void method_10514(class_7225.class_7874 arg);

	/**
	 * Override to enable adding objects to the tag builder directly.
	 */
	@SuppressWarnings({"unchecked", "rawtypes"})
	protected class_5321<T> reverseLookup(T element) {
		class_2378 registry = class_7923.field_41167.method_29107((class_5321) field_40957);

		if (registry != null) {
			Optional<class_6880<T>> key = registry.method_29113(element);

			if (key.isPresent()) {
				return (class_5321<T>) key.get();
			}
		}

		throw new UnsupportedOperationException("Adding objects is not supported by " + getClass());
	}

	/**
	 * Creates a new instance of {@link FabricTagBuilder} for the given {@link class_6862} tag.
	 *
	 * @param tag The {@link class_6862} tag to create the builder for
	 * @return The {@link FabricTagBuilder} instance
	 */
	@Override
	protected FabricTagBuilder method_10512(class_6862<T> tag) {
		return new FabricTagBuilder(super.method_10512(tag));
	}

	/**
	 * Extend this class to create {@link class_2248} tags in the "/blocks" tag directory.
	 */
	public abstract static class BlockTagProvider extends FabricTagProvider<class_2248> {
		public BlockTagProvider(FabricDataOutput output, CompletableFuture<class_7225.class_7874> registriesFuture) {
			super(output, class_7924.field_41254, registriesFuture);
		}

		@Override
		protected class_5321<class_2248> reverseLookup(class_2248 element) {
			return element.method_40142().method_40237();
		}
	}

	/**
	 * Extend this class to create {@link class_1792} tags in the "/items" tag directory.
	 */
	public abstract static class ItemTagProvider extends FabricTagProvider<class_1792> {
		@Nullable
		private final Function<class_6862<class_2248>, class_3495> blockTagBuilderProvider;

		/**
		 * Construct an {@link ItemTagProvider} tag provider <b>with</b> an associated {@link BlockTagProvider} tag provider.
		 *
		 * @param output The {@link FabricDataOutput} instance
		 */
		public ItemTagProvider(FabricDataOutput output, CompletableFuture<class_7225.class_7874> completableFuture, @Nullable FabricTagProvider.BlockTagProvider blockTagProvider) {
			super(output, class_7924.field_41197, completableFuture);

			this.blockTagBuilderProvider = blockTagProvider == null ? null : blockTagProvider::method_27169;
		}

		/**
		 * Construct an {@link ItemTagProvider} tag provider <b>without</b> an associated {@link BlockTagProvider} tag provider.
		 *
		 * @param output The {@link FabricDataOutput} instance
		 */
		public ItemTagProvider(FabricDataOutput output, CompletableFuture<class_7225.class_7874> completableFuture) {
			this(output, completableFuture, null);
		}

		/**
		 * Copy the entries from a tag with the {@link class_2248} type into this item tag.
		 *
		 * <p>The {@link ItemTagProvider} tag provider must be constructed with an associated {@link BlockTagProvider} tag provider to use this method.
		 *
		 *
		 * @param blockTag The block tag to copy from.
		 * @param itemTag  The item tag to copy to.
		 */
		public void copy(class_6862<class_2248> blockTag, class_6862<class_1792> itemTag) {
			class_3495 blockTagBuilder = Objects.requireNonNull(this.blockTagBuilderProvider, "Pass Block tag provider via constructor to use copy").apply(blockTag);
			class_3495 itemTagBuilder = this.method_27169(itemTag);
			blockTagBuilder.method_26782().forEach(itemTagBuilder::method_27064);
		}

		@Override
		protected class_5321<class_1792> reverseLookup(class_1792 element) {
			return element.method_40131().method_40237();
		}
	}

	/**
	 * Extend this class to create {@link class_3611} tags in the "/fluids" tag directory.
	 */
	public abstract static class FluidTagProvider extends FabricTagProvider<class_3611> {
		public FluidTagProvider(FabricDataOutput output, CompletableFuture<class_7225.class_7874> completableFuture) {
			super(output, class_7924.field_41270, completableFuture);
		}

		@Override
		protected class_5321<class_3611> reverseLookup(class_3611 element) {
			return element.method_40178().method_40237();
		}
	}

	/**
	 * Extend this class to create {@link class_1887} tags in the "/enchantments" tag directory.
	 */
	public abstract static class EnchantmentTagProvider extends FabricTagProvider<class_1887> {
		public EnchantmentTagProvider(FabricDataOutput output, CompletableFuture<class_7225.class_7874> completableFuture) {
			super(output, class_7924.field_41265, completableFuture);
		}

		@Override
		protected class_5321<class_1887> reverseLookup(class_1887 element) {
			return class_7923.field_41176.method_29113(element)
					.orElseThrow(() -> new IllegalArgumentException("Enchantment " + element + " is not registered"));
		}
	}

	/**
	 * Extend this class to create {@link class_1299} tags in the "/entity_types" tag directory.
	 */
	public abstract static class EntityTypeTagProvider extends FabricTagProvider<class_1299<?>> {
		public EntityTypeTagProvider(FabricDataOutput output, CompletableFuture<class_7225.class_7874> completableFuture) {
			super(output, class_7924.field_41266, completableFuture);
		}

		@Override
		protected class_5321<class_1299<?>> reverseLookup(class_1299<?> element) {
			return element.method_40124().method_40237();
		}
	}

	/**
	 * Extend this class to create {@link class_5712} tags in the "/game_events" tag directory.
	 */
	public abstract static class GameEventTagProvider extends FabricTagProvider<class_5712> {
		public GameEventTagProvider(FabricDataOutput output, CompletableFuture<class_7225.class_7874> completableFuture) {
			super(output, class_7924.field_41273, completableFuture);
		}

		@Override
		protected class_5321<class_5712> reverseLookup(class_5712 element) {
			return element.method_40157().method_40237();
		}
	}

	/**
	 * An extension to {@link class_5124} that provides additional functionality.
	 */
	public final class FabricTagBuilder extends class_5124<T> {
		private final class_2474.class_5124<T> parent;

		private FabricTagBuilder(class_5124<T> parent) {
			super(parent.field_23960);
			this.parent = parent;
		}

		/**
		 * Set the value of the `replace` flag in a Tag.
		 *
		 * <p>When set to true the tag will replace any existing tag entries.
		 *
		 * @return the {@link FabricTagBuilder} instance
		 */
		public FabricTagBuilder setReplace(boolean replace) {
			((net.fabricmc.fabric.impl.datagen.FabricTagBuilder) field_23960).fabric_setReplace(replace);
			return this;
		}

		/**
		 * Add an element to the tag.
		 *
		 * @return the {@link FabricTagBuilder} instance
		 */
		public FabricTagBuilder add(T element) {
			method_46835(reverseLookup(element));
			return this;
		}

		/**
		 * Add multiple elements to the tag.
		 *
		 * @return the {@link FabricTagBuilder} instance
		 */
		@SafeVarargs
		public final FabricTagBuilder add(T... element) {
			Stream.of(element).map(FabricTagProvider.this::reverseLookup).forEach(this::method_46835);
			return this;
		}

		/**
		 * Add an element to the tag.
		 *
		 * @return the {@link FabricTagBuilder} instance
		 * @see #add(class_2960)
		 */
		@Override
		public FabricTagBuilder method_46835(class_5321<T> registryKey) {
			parent.method_46835(registryKey);
			return this;
		}

		/**
		 * Add a single element to the tag.
		 *
		 * @return the {@link FabricTagBuilder} instance
		 */
		public FabricTagBuilder add(class_2960 id) {
			field_23960.add(id);
			return this;
		}

		/**
		 * Add an optional {@link class_2960} to the tag.
		 *
		 * @return the {@link FabricTagBuilder} instance
		 */
		@Override
		public FabricTagBuilder method_35922(class_2960 id) {
			parent.method_35922(id);
			return this;
		}

		/**
		 * Add an optional {@link class_5321} to the tag.
		 *
		 * @return the {@link FabricTagBuilder} instance
		 */
		public FabricTagBuilder addOptional(class_5321<? extends T> registryKey) {
			return method_35922(registryKey.method_29177());
		}

		/**
		 * Add another tag to this tag.
		 *
		 * <p><b>Note:</b> any vanilla tags can be added to the builder,
		 * but other tags can only be added if it has a builder registered in the same provider.
		 *
		 * <p>Use {@link #forceAddTag(class_6862)} to force add any tag.
		 *
		 * @return the {@link FabricTagBuilder} instance
		 * @see class_3481
		 * @see class_3483
		 * @see class_3486
		 * @see class_5698
		 * @see class_3489
		 */
		@Override
		public FabricTagBuilder method_26792(class_6862<T> tag) {
			field_23960.addTag(tag.comp_327());
			return this;
		}

		/**
		 * Add another optional tag to this tag.
		 *
		 * @return the {@link FabricTagBuilder} instance
		 */
		@Override
		public FabricTagBuilder method_35923(class_2960 id) {
			parent.method_35923(id);
			return this;
		}

		/**
		 * Add another optional tag to this tag.
		 *
		 * @return the {@link FabricTagBuilder} instance
		 */
		public FabricTagBuilder addOptionalTag(class_6862<T> tag) {
			return method_35923(tag.comp_327());
		}

		/**
		 * Add another tag to this tag, ignoring any warning.
		 *
		 * <p><b>Note:</b> only use this method if you sure that the tag will be always available at runtime.
		 * If not, use {@link #method_35923(class_2960)} instead.
		 *
		 * @return the {@link FabricTagBuilder} instance
		 */
		public FabricTagBuilder forceAddTag(class_6862<T> tag) {
			field_23960.add(new ForcedTagEntry(class_3497.method_43937(tag.comp_327())));
			return this;
		}

		/**
		 * Add multiple elements to this tag.
		 *
		 * @return the {@link FabricTagBuilder} instance
		 */
		public FabricTagBuilder add(class_2960... ids) {
			for (class_2960 id : ids) {
				add(id);
			}

			return this;
		}

		/**
		 * Add multiple elements to this tag.
		 *
		 * @return the {@link FabricTagBuilder} instance
		 */
		@SafeVarargs
		@Override
		public final FabricTagBuilder method_40565(class_5321<T>... registryKeys) {
			for (class_5321<T> registryKey : registryKeys) {
				method_46835(registryKey);
			}

			return this;
		}
	}
}
