/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.datagen.v1.provider;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.gson.JsonElement;
import com.mojang.serialization.Codec;
import com.mojang.serialization.DynamicOps;
import com.mojang.serialization.Encoder;
import com.mojang.serialization.JsonOps;
import org.jetbrains.annotations.ApiStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import net.fabricmc.fabric.api.datagen.v1.FabricDataOutput;
import net.minecraft.class_2378;
import net.minecraft.class_2405;
import net.minecraft.class_2922;
import net.minecraft.class_2960;
import net.minecraft.class_5321;
import net.minecraft.class_6796;
import net.minecraft.class_6880;
import net.minecraft.class_6903;
import net.minecraft.class_7225;
import net.minecraft.class_7403;
import net.minecraft.class_7655;
import net.minecraft.class_7784;
import net.minecraft.class_7871;
import net.minecraft.class_7876;
import net.minecraft.class_7924;

/**
 * A provider to help with data-generation of dynamic registry objects,
 * such as biomes, features, or message types.
 */
@ApiStatus.Experimental
public abstract class FabricDynamicRegistryProvider implements class_2405 {
	private static final Logger LOGGER = LoggerFactory.getLogger(FabricDynamicRegistryProvider.class);

	private final FabricDataOutput output;
	private final CompletableFuture<class_7225.class_7874> registriesFuture;

	public FabricDynamicRegistryProvider(FabricDataOutput output, CompletableFuture<class_7225.class_7874> registriesFuture) {
		this.output = output;
		this.registriesFuture = registriesFuture;
	}

	protected abstract void configure(class_7225.class_7874 registries, Entries entries);

	public static final class Entries {
		private final class_7225.class_7874 registries;
		// Registry ID -> Entries for that registry
		private final Map<class_2960, RegistryEntries<?>> queuedEntries;
		private final String modId;

		@ApiStatus.Internal
		Entries(class_7225.class_7874 registries, String modId) {
			this.registries = registries;
			this.queuedEntries = class_7655.field_39968.stream()
					.collect(Collectors.toMap(
							e -> e.comp_985().method_29177(),
							e -> RegistryEntries.create(registries, e)
					));
			this.modId = modId;
		}

		/**
		 * Gets access to all registry lookups.
		 */
		public class_7225.class_7874 getLookups() {
			return registries;
		}

		/**
		 * Gets a lookup for entries from the given registry.
		 */
		public <T> class_7871<T> getLookup(class_5321<? extends class_2378<T>> registryKey) {
			return registries.method_46762(registryKey);
		}

		/**
		 * Returns a lookup for placed features. Useful when creating biomes.
		 */
		public class_7871<class_6796> placedFeatures() {
			return getLookup(class_7924.field_41245);
		}

		/**
		 * Returns a lookup for configured carvers. Useful when creating biomes.
		 */
		public class_7871<class_2922<?>> configuredCarvers() {
			return getLookup(class_7924.field_41238);
		}

		/**
		 * Gets a reference to a registry entry for use in other registrations.
		 */
		public <T> class_6880<T> ref(class_5321<T> key) {
			RegistryEntries<T> entries = getQueuedEntries(key);
			return class_6880.class_6883.method_40234(entries.lookup, key);
		}

		/**
		 * Adds a new object to be data generated.
		 *
		 * @return a reference to it for use in other objects.
		 */
		public <T> class_6880<T> add(class_5321<T> registry, T object) {
			return getQueuedEntries(registry).add(registry.method_29177(), object);
		}

		/**
		 * Adds a new {@link class_5321} from a given {@link class_7225.class_7226} to be data generated.
		 *
		 * @return a reference to it for use in other objects.
		 */
		public <T> class_6880<T> add(class_7225.class_7226<T> registry, class_5321<T> valueKey) {
			return add(valueKey, registry.method_46747(valueKey).comp_349());
		}

		/**
		 * All the registry entries whose namespace matches the current effective mod ID will be data generated.
		 */
		public <T> List<class_6880<T>> addAll(class_7225.class_7226<T> registry) {
			return registry.method_46754()
					.filter(registryKey -> registryKey.method_29177().method_12836().equals(modId))
					.map(key -> add(registry, key))
					.toList();
		}

		@SuppressWarnings("unchecked")
		<T> RegistryEntries<T> getQueuedEntries(class_5321<T> key) {
			RegistryEntries<?> regEntries = queuedEntries.get(key.method_41185());

			if (regEntries == null) {
				throw new IllegalArgumentException("Registry " + key.method_41185() + " is not loaded from datapacks");
			}

			return (RegistryEntries<T>) regEntries;
		}
	}

	private static class RegistryEntries<T> {
		final class_7876<T> lookup;
		final class_5321<? extends class_2378<T>> registry;
		final Codec<T> elementCodec;
		Map<class_5321<T>, T> entries = new IdentityHashMap<>();

		RegistryEntries(class_7876<T> lookup,
						class_5321<? extends class_2378<T>> registry,
						Codec<T> elementCodec) {
			this.lookup = lookup;
			this.registry = registry;
			this.elementCodec = elementCodec;
		}

		static <T> RegistryEntries<T> create(class_7225.class_7874 lookups, class_7655.class_7657<T> loaderEntry) {
			class_7225.class_7226<T> lookup = lookups.method_46762(loaderEntry.comp_985());
			return new RegistryEntries<>(lookup, loaderEntry.comp_985(), loaderEntry.comp_986());
		}

		public class_6880<T> add(class_5321<T> key, T value) {
			if (entries.put(key, value) != null) {
				throw new IllegalArgumentException("Trying to add registry key " + key + " more than once.");
			}

			return class_6880.class_6883.method_40234(lookup, key);
		}

		public class_6880<T> add(class_2960 id, T value) {
			return add(class_5321.method_29179(registry, id), value);
		}
	}

	@Override
	public CompletableFuture<?> method_10319(class_7403 writer) {
		return registriesFuture.thenCompose(registries -> {
			return CompletableFuture
					.supplyAsync(() -> {
						Entries entries = new Entries(registries, output.getModId());
						configure(registries, entries);
						return entries;
					})
					.thenCompose(entries -> {
						final class_6903<JsonElement> dynamicOps = class_6903.method_46632(JsonOps.INSTANCE, registries);
						ArrayList<CompletableFuture<?>> futures = new ArrayList<>();

						for (RegistryEntries<?> registryEntries : entries.queuedEntries.values()) {
							futures.add(writeRegistryEntries(writer, dynamicOps, registryEntries));
						}

						return CompletableFuture.allOf(futures.toArray(CompletableFuture[]::new));
					});
		});
	}

	private <T> CompletableFuture<?> writeRegistryEntries(class_7403 writer, class_6903<JsonElement> ops, RegistryEntries<T> entries) {
		final class_5321<? extends class_2378<T>> registry = entries.registry;
		final class_7784.class_7489 pathResolver = output.method_45973(class_7784.class_7490.field_39367, registry.method_29177().method_12832());
		final List<CompletableFuture<?>> futures = new ArrayList<>();

		for (Map.Entry<class_5321<T>, T> entry : entries.entries.entrySet()) {
			Path path = pathResolver.method_44107(entry.getKey().method_29177());
			futures.add(writeToPath(path, writer, ops, entries.elementCodec, entry.getValue()));
		}

		return CompletableFuture.allOf(futures.toArray(CompletableFuture[]::new));
	}

	private static <E> CompletableFuture<?> writeToPath(Path path, class_7403 cache, DynamicOps<JsonElement> json, Encoder<E> encoder, E value) {
		Optional<JsonElement> optional = encoder.encodeStart(json, value).resultOrPartial((error) -> {
			field_40831.error("Couldn't serialize element {}: {}", path, error);
		});

		if (optional.isPresent()) {
			return class_2405.method_10320(cache, optional.get(), path);
		}

		return CompletableFuture.completedFuture(null);
	}
}
