/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.datagen.v1.provider;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.gson.JsonObject;
import org.jetbrains.annotations.ApiStatus;
import net.minecraft.class_176;
import net.minecraft.class_2405;
import net.minecraft.class_2960;
import net.minecraft.class_52;
import net.minecraft.class_7403;
import net.minecraft.class_7784;
import net.minecraft.class_8490;
import net.fabricmc.fabric.api.datagen.v1.FabricDataOutput;
import net.fabricmc.fabric.api.datagen.v1.loot.FabricBlockLootTableGenerator;
import net.fabricmc.fabric.api.resource.conditions.v1.ConditionJsonProvider;
import net.fabricmc.fabric.impl.datagen.FabricDataGenHelper;

/**
 * A base interface for Loot table providers. You should not implement this class directly.
 *
 * <p>{@link FabricBlockLootTableProvider} provides additional features specific to block drop loot tables.
 *
 * <p>Use {@link SimpleFabricLootTableProvider} for a simple abstract class that you can implement to handle standard loot table functions.
 */
@ApiStatus.NonExtendable
public interface FabricLootTableProvider extends Consumer<BiConsumer<class_2960, class_52.class_53>>, class_2405 {
	class_176 getLootContextType();

	FabricDataOutput getFabricDataOutput();

	/**
	 * Return a new exporter that applies the specified conditions to any loot table it receives.
	 *
	 * <p>For block loot tables, use {@link FabricBlockLootTableGenerator#withConditions} instead.
	 */
	default BiConsumer<class_2960, class_52.class_53> withConditions(BiConsumer<class_2960, class_52.class_53> exporter, ConditionJsonProvider... conditions) {
		Preconditions.checkArgument(conditions.length > 0, "Must add at least one condition.");
		return (id, table) -> {
			FabricDataGenHelper.addConditions(table, conditions);
			exporter.accept(id, table);
		};
	}

	@ApiStatus.Internal
	@Override
	default CompletableFuture<?> method_10319(class_7403 writer) {
		HashMap<class_2960, class_52> builders = Maps.newHashMap();
		HashMap<class_2960, ConditionJsonProvider[]> conditionMap = new HashMap<>();

		accept((identifier, builder) -> {
			ConditionJsonProvider[] conditions = FabricDataGenHelper.consumeConditions(builder);
			conditionMap.put(identifier, conditions);

			if (builders.put(identifier, builder.method_334(getLootContextType()).method_338()) != null) {
				throw new IllegalStateException("Duplicate loot table " + identifier);
			}
		});

		final List<CompletableFuture<?>> futures = new ArrayList<>();

		for (Map.Entry<class_2960, class_52> entry : builders.entrySet()) {
			JsonObject tableJson = (JsonObject) class_8490.field_44498.method_51203().toJsonTree(entry.getValue());
			ConditionJsonProvider.write(tableJson, conditionMap.remove(entry.getKey()));

			futures.add(class_2405.method_10320(writer, tableJson, getOutputPath(entry.getKey())));
		}

		return CompletableFuture.allOf(futures.toArray(CompletableFuture[]::new));
	}

	private Path getOutputPath(class_2960 lootTableId) {
		return getFabricDataOutput().method_45973(class_7784.class_7490.field_39367, "loot_tables").method_44107(lootTableId);
	}
}
