/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.dimension;

import io.netty.buffer.Unpooled;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import net.fabricmc.fabric.api.dimension.v1.FabricDimensionType;
import net.fabricmc.fabric.api.network.ServerSidePacketRegistry;
import net.fabricmc.fabric.impl.registry.RemapException;
import net.minecraft.class_2378;
import net.minecraft.class_2487;
import net.minecraft.class_2540;
import net.minecraft.class_2596;
import net.minecraft.class_2874;
import net.minecraft.class_2960;
import net.minecraft.class_31;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Handles fixing raw dimension ids between saves and servers,
 * and synchronizes said ids.
 */
public class DimensionIdsFixer {
	private static final Field FABRIC_DIMENSION_TYPE$RAW_ID;
	static final class_2960 ID = new class_2960("fabric", "dimension/sync");

	/**
	 * Assigns a unique id to every registered {@link FabricDimensionType}, keeping the known ids
	 * from {@code savedIds}.
	 *
	 * @param savedIds a compound tag mapping dimension ids to raw ids
	 * @return id to raw id mappings of the current instance
	 * @throws RemapException if dimensions IDs conflict irredeemably
	 */
	public static class_2487 apply(class_2487 savedIds) throws RemapException {
		/*
		 * We want to give to each fabric dimension a unique ID. We also want to give back previously assigned ids.
		 * And we have to take into account non-fabric dimensions, which raw IDs cannot change.
		 * So we iterate over every dimension, note the ones which id cannot change, then update the free ones.
		 */
		Int2ObjectMap<class_2960> fixedIds = new Int2ObjectOpenHashMap<>();
		List<FabricDimensionType> fabricDimensions = new ArrayList<>();
		class_2487 fabricDimensionIds = new class_2487();

		// step 1: detect all fabric and non-fabric dimensions
		for (class_2960 id : class_2378.field_11155.method_10235()) {
			class_2874 dimensionType = Objects.requireNonNull(class_2874.method_12483(id));

			if (dimensionType instanceof FabricDimensionType) {
				FabricDimensionType fabricDimension = (FabricDimensionType) dimensionType;
				fabricDimensions.add(fabricDimension);
				// reset the fixed raw id to the preferred raw id
				setFixedRawId(fabricDimension, fabricDimension.getDesiredRawId());
			} else {
				class_2960 existing = fixedIds.put(dimensionType.method_12484(), id);
				if (existing != null) {
					throw new RemapException("Two non-fabric dimensions have the same raw dim id (" + dimensionType.method_12484() + ") : " + existing + " and " + id);
				}
			}
		}

		// step 2: read saved ids
		for (String key : savedIds.method_10541()) {
			int savedRawId = savedIds.method_10550(key);
			class_2960 dimId = new class_2960(key);
			class_2960 existing = fixedIds.putIfAbsent(savedRawId, dimId);

			if (existing != null && !existing.equals(dimId)) {
				throw new RemapException("Saved fabric dimension got replaced with a non-fabric one! " + dimId + " replaced with " + existing + " (raw id: " + savedRawId + ")");
			}

			class_2874 dim = class_2874.method_12483(dimId);
			if (dim instanceof FabricDimensionType) {
				setFixedRawId((FabricDimensionType) dim, savedRawId);
			} else {
				FabricDimensionInternals.LOGGER.warn("A saved dimension has {}: {}", dim == null ? "been removed" : "stopped using the dimensions API", dimId);
				// Preserve saved ids in case the mod is eventually added back
				fabricDimensionIds.method_10569(dimId.toString(), savedRawId);
			}
		}

		// step 3: de-duplicate raw ids for dimensions which ids are not fixed yet
		int nextFreeId = 0;
		for (FabricDimensionType fabricDimension : fabricDimensions) {
			int rawDimId = fabricDimension.method_12484();
			class_2960 dimId = Objects.requireNonNull(class_2874.method_12485(fabricDimension));

			if (fixedIds.containsKey(rawDimId) && !fixedIds.get(rawDimId).equals(dimId)) {
				while (fixedIds.containsKey(nextFreeId)) ++nextFreeId;
				setFixedRawId(fabricDimension, nextFreeId);
				rawDimId = nextFreeId;
			}

			fixedIds.put(rawDimId, dimId);
			fabricDimensionIds.method_10569(dimId.toString(), rawDimId);
		}

		return fabricDimensionIds;
	}

	/**
	 * Reflectively set the fixed raw id on a {@link FabricDimensionType}.
	 *
	 * @see FabricDimensionType#getRawId()
	 */
	private static void setFixedRawId(FabricDimensionType fabricDimension, int rawId) {
		try {
			FABRIC_DIMENSION_TYPE$RAW_ID.setInt(fabricDimension, rawId);
		} catch (IllegalAccessException e) {
			throw new RuntimeException("Failed to fix a raw id on a FabricDimensionType", e);
		}
	}

	public static class_2596<?> createPacket(class_31 levelProperties) {
		class_2540 buf = new class_2540(Unpooled.buffer());
		buf.method_10794(((DimensionIdsHolder) levelProperties).fabric_getDimensionIds());
		return ServerSidePacketRegistry.INSTANCE.toPacket(ID, buf);
	}

	static {
		try {
			FABRIC_DIMENSION_TYPE$RAW_ID = FabricDimensionType.class.getDeclaredField("fixedRawId");
			FABRIC_DIMENSION_TYPE$RAW_ID.setAccessible(true);
		} catch (NoSuchFieldException e) {
			throw new RuntimeException(e);
		}
	}
}
