/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.dimension;

import com.google.common.base.Preconditions;
import net.fabricmc.fabric.api.dimension.v1.EntityPlacer;
import net.fabricmc.fabric.api.dimension.v1.FabricDimensionType;
import net.fabricmc.fabric.api.dimension.v1.FabricDimensions;
import net.fabricmc.fabric.mixin.EntityHooks;
import net.minecraft.class_1297;
import net.minecraft.class_2350;
import net.minecraft.class_2700;
import net.minecraft.class_2874;
import net.minecraft.class_3218;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public final class FabricDimensionInternals {
	private FabricDimensionInternals() {
		throw new AssertionError();
	}

	public static final boolean DEBUG = System.getProperty("fabric.dimension.debug", "false").equalsIgnoreCase("true");
	public static final Logger LOGGER = LogManager.getLogger();

	/**
	 * The entity currently being transported to another dimension
	 */
	private static final ThreadLocal<class_1297> PORTAL_ENTITY = new ThreadLocal<>();
	/**
	 * The custom placement logic passed from {@link FabricDimensions#teleport(Entity, DimensionType, EntityPlacer)}
	 */
	private static EntityPlacer customPlacement;

	/*
	 * The dimension change hooks consist of two steps:
	 * - First, we memorize the currently teleported entity, and set required fields
	 * - Then, we retrieve the teleported entity in the placement logic in PortalForcer#getPortal
	 *   and use it to call the entity placers
	 * This lets us use the exact same logic for any entity and prevent the vanilla getPortal (which has unwanted
	 * side effects) from running, while keeping the patches minimally invasive.
	 *
	 * Shortcomings: bugs may arise if another patch cancels the teleportation method between
	 * #prepareDimensionalTeleportation and #tryFindPlacement, AND a mod calls PortalForcer#getPortal directly
	 * right after.
	 */

	public static void prepareDimensionalTeleportation(class_1297 entity) {
		Preconditions.checkNotNull(entity);
		PORTAL_ENTITY.set(entity);

		// Set values used by `PortalForcer#changeDimension` to prevent a NPE crash.
		EntityHooks access = ((EntityHooks) entity);
		if (entity.method_5656() == null) {
			access.setLastPortalDirectionVector(entity.method_5720());
		}
		if (entity.method_5843() == null) {
			access.setLastPortalDirection(entity.method_5735());
		}
	}

	/* Nullable */
	public static class_2700.class_4297 tryFindPlacement(class_3218 destination, class_2350 portalDir, double portalX, double portalY) {
		Preconditions.checkNotNull(destination);
		class_1297 teleported = PORTAL_ENTITY.get();
		PORTAL_ENTITY.set(null);

		// If the entity is null, the call does not come from a vanilla context
		if (teleported == null) {
			return null;
		}

		// Custom placement logic, falls back to default dimension placement if no placement or target found
		EntityPlacer customPlacement = FabricDimensionInternals.customPlacement;
		if (customPlacement != null) {
			class_2700.class_4297 customTarget = customPlacement.placeEntity(teleported, destination, portalDir, portalX, portalY);

			if (customTarget != null) {
				return customTarget;
			}
		}

		// Default placement logic, falls back to vanilla if not a fabric dimension
		class_2874 dimType = destination.method_8597().method_12460();
		if (dimType instanceof FabricDimensionType) {
			class_2700.class_4297 defaultTarget = ((FabricDimensionType) dimType).getDefaultPlacement().placeEntity(teleported, destination, portalDir, portalX, portalY);

			if (defaultTarget == null) {
				throw new IllegalStateException("Mod dimension " + class_2874.method_12485(dimType) + " returned an invalid teleport target");
			}
			return defaultTarget;
		}

		// Vanilla / other implementations logic, undefined behaviour on custom dimensions
		return null;
	}

	@SuppressWarnings("unchecked")
	public static <E extends class_1297> E changeDimension(E teleported, class_2874 dimension, EntityPlacer placement) {
		assert !teleported.field_6002.field_9236 : "Entities can only be teleported on the server side";
		assert Thread.currentThread() == ((class_3218) teleported.field_6002).method_8503().method_3777() : "Entities must be teleported from the main server thread";

		try {
			customPlacement = placement;
			return (E) teleported.method_5731(dimension);
		} finally {
			customPlacement = null;
		}
	}

}
