/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.gamerule.v1;

import static com.google.common.base.Preconditions.checkNotNull;

import java.util.function.BiConsumer;

import com.mojang.brigadier.arguments.DoubleArgumentType;
import com.mojang.brigadier.arguments.IntegerArgumentType;
import org.jetbrains.annotations.Nullable;

import net.minecraft.class_10961;
import net.minecraft.class_1928;
import net.minecraft.class_7699;
import net.minecraft.server.MinecraftServer;
import net.fabricmc.fabric.api.gamerule.v1.rule.DoubleRule;
import net.fabricmc.fabric.api.gamerule.v1.rule.EnumRule;
import net.fabricmc.fabric.impl.gamerule.EnumRuleType;
import net.fabricmc.fabric.impl.gamerule.rule.BoundedIntRule;
import net.fabricmc.fabric.mixin.gamerule.GameRulesBooleanRuleAccessor;

/**
 * A utility class containing factory methods to create game rule types.
 * A game rule is a persisted, per server data value which may control gameplay aspects.
 *
 * <p>Some factory methods allow specification of a callback that is invoked when the value of a game rule has changed.
 * Typically, the callback is used for game rules which may influence game logic, such as {@link class_1928#field_19422 disabling raids}.
 *
 * <p>To register a game rule, you can use {@link GameRuleRegistry#register(String, class_1928.class_5198, class_1928.class_4314)}.
 * For example, to register a game rule that is an integer where the acceptable values are between 0 and 10, one would use the following:
 * <blockquote><pre>
 * public static final GameRules.Key&lt;GameRules.IntRule&gt; EXAMPLE_INT_RULE = GameRuleRegistry.register("exampleIntRule", GameRules.Category.UPDATES, GameRuleFactory.createIntRule(1, 10));
 * </pre></blockquote>
 *
 * <p>To register a game rule in a custom category, {@link GameRuleRegistry#register(String, CustomGameRuleCategory, class_1928.class_4314)} should be used.
 *
 * @see GameRuleRegistry
 */
public final class GameRuleFactory {
	/**
	 * Creates a boolean rule type.
	 *
	 * @param defaultValue the default value of the game rule
	 * @return a boolean rule type
	 */
	public static class_1928.class_4314<class_1928.class_4310> createBooleanRule(boolean defaultValue) {
		return createBooleanRule(defaultValue, (server, rule) -> {
		});
	}

	/**
	 * Creates a boolean rule type.
	 *
	 * @param defaultValue the default value of the game rule
	 * @param changedCallback a callback that is invoked when the value of a game rule has changed
	 * @return a boolean rule type
	 */
	public static class_1928.class_4314<class_1928.class_4310> createBooleanRule(boolean defaultValue, BiConsumer<MinecraftServer, class_1928.class_4310> changedCallback) {
		return GameRulesBooleanRuleAccessor.invokeCreate(defaultValue, changedCallback);
	}

	/**
	 * Creates an integer rule type.
	 *
	 * @param defaultValue the default value of the game rule
	 * @return an integer rule type
	 */
	public static class_1928.class_4314<class_1928.class_4312> createIntRule(int defaultValue) {
		return createIntRule(defaultValue, (server, rule) -> {
		});
	}

	/**
	 * Creates an integer rule type.
	 *
	 * @param defaultValue the default value of the game rule
	 * @param minimumValue the minimum value the game rule may accept
	 * @return an integer rule type
	 */
	public static class_1928.class_4314<class_1928.class_4312> createIntRule(int defaultValue, int minimumValue) {
		return createIntRule(defaultValue, minimumValue, Integer.MAX_VALUE, (server, rule) -> {
		});
	}

	/**
	 * Creates an integer rule type.
	 *
	 * @param defaultValue the default value of the game rule
	 * @param minimumValue the minimum value the game rule may accept
	 * @param changedCallback a callback that is invoked when the value of a game rule has changed
	 * @return an integer rule type
	 */
	public static class_1928.class_4314<class_1928.class_4312> createIntRule(int defaultValue, int minimumValue, BiConsumer<class_10961, class_1928.class_4312> changedCallback) {
		return createIntRule(defaultValue, minimumValue, Integer.MAX_VALUE, changedCallback);
	}

	/**
	 * Creates an integer rule type.
	 *
	 * @param defaultValue the default value of the game rule
	 * @param minimumValue the minimum value the game rule may accept
	 * @param maximumValue the maximum value the game rule may accept
	 * @return an integer rule type
	 */
	public static class_1928.class_4314<class_1928.class_4312> createIntRule(int defaultValue, int minimumValue, int maximumValue) {
		return createIntRule(defaultValue, minimumValue, maximumValue, (server, rule) -> {
		});
	}

	/**
	 * Creates an integer rule type.
	 *
	 * @param defaultValue the default value of the game rule
	 * @param changedCallback a callback that is invoked when the value of a game rule has changed
	 * @return an integer rule type
	 */
	public static class_1928.class_4314<class_1928.class_4312> createIntRule(int defaultValue, BiConsumer<class_10961, class_1928.class_4312> changedCallback) {
		return createIntRule(defaultValue, Integer.MIN_VALUE, Integer.MAX_VALUE, changedCallback);
	}

	/**
	 * Creates an integer rule type.
	 *
	 * @param defaultValue the default value of the game rule
	 * @param minimumValue the minimum value the game rule may accept
	 * @param maximumValue the maximum value the game rule may accept
	 * @param changedCallback a callback that is invoked when the value of a game rule has changed
	 * @return an integer rule type
	 */
	public static class_1928.class_4314<class_1928.class_4312> createIntRule(int defaultValue, int minimumValue, int maximumValue, @Nullable BiConsumer<class_10961, class_1928.class_4312> changedCallback) {
		return new class_1928.class_4314<>(
				() -> IntegerArgumentType.integer(minimumValue, maximumValue),
				type -> new BoundedIntRule(type, defaultValue, minimumValue, maximumValue), // Internally use a bounded int rule
				changedCallback,
				class_1928.class_4311::method_27330,
				class_1928.class_4312.class,
				class_7699.method_45397()
		);
	}

	/**
	 * Creates a double rule type.
	 *
	 * @param defaultValue the default value of the game rule
	 * @return a double rule type
	 */
	public static class_1928.class_4314<DoubleRule> createDoubleRule(double defaultValue) {
		return createDoubleRule(defaultValue, (server, rule) -> {
		});
	}

	/**
	 * Creates a double rule type.
	 *
	 * @param defaultValue the default value of the game rule
	 * @param minimumValue the minimum value the game rule may accept
	 * @return a double rule type
	 */
	public static class_1928.class_4314<DoubleRule> createDoubleRule(double defaultValue, double minimumValue) {
		return createDoubleRule(defaultValue, minimumValue, Double.MAX_VALUE, (server, rule) -> {
		});
	}

	/**
	 * Creates a double rule type.
	 *
	 * @param defaultValue the default value of the game rule
	 * @param minimumValue the minimum value the game rule may accept
	 * @param changedCallback a callback that is invoked when the value of a game rule has changed
	 * @return a double rule type
	 */
	public static class_1928.class_4314<DoubleRule> createDoubleRule(double defaultValue, double minimumValue, BiConsumer<class_10961, DoubleRule> changedCallback) {
		return createDoubleRule(defaultValue, minimumValue, Double.MAX_VALUE, changedCallback);
	}

	/**
	 * Creates a double rule type.
	 *
	 * @param defaultValue the default value of the game rule
	 * @param minimumValue the minimum value the game rule may accept
	 * @param maximumValue the maximum value the game rule may accept
	 * @return a double rule type
	 */
	public static class_1928.class_4314<DoubleRule> createDoubleRule(double defaultValue, double minimumValue, double maximumValue) {
		return createDoubleRule(defaultValue, minimumValue, maximumValue, (server, rule) -> {
		});
	}

	/**
	 * Creates a double rule type.
	 *
	 * @param defaultValue the default value of the game rule
	 * @param changedCallback a callback that is invoked when the value of a game rule has changed
	 * @return a double rule type
	 */
	public static class_1928.class_4314<DoubleRule> createDoubleRule(double defaultValue, BiConsumer<class_10961, DoubleRule> changedCallback) {
		return createDoubleRule(defaultValue, Double.MIN_VALUE, Double.MAX_VALUE, changedCallback);
	}

	/**
	 * Creates a double rule type.
	 *
	 * @param defaultValue the default value of the game rule
	 * @param minimumValue the minimum value the game rule may accept
	 * @param maximumValue the maximum value the game rule may accept
	 * @param changedCallback a callback that is invoked when the value of a game rule has changed
	 * @return a double rule type
	 */
	public static class_1928.class_4314<DoubleRule> createDoubleRule(double defaultValue, double minimumValue, double maximumValue, BiConsumer<class_10961, DoubleRule> changedCallback) {
		return new class_1928.class_4314<>(
				() -> DoubleArgumentType.doubleArg(minimumValue, maximumValue),
				type -> new DoubleRule(type, defaultValue, minimumValue, maximumValue),
				changedCallback,
				GameRuleFactory::visitDouble,
				DoubleRule.class,
				class_7699.method_45397()
		);
	}

	/**
	 * Creates an enum rule type.
	 *
	 * <p>All enum values are supported.
	 *
	 * @param defaultValue the default value of the game rule
	 * @param <E> the type of enum this game rule stores
	 * @return an enum rule type
	 */
	public static <E extends Enum<E>> class_1928.class_4314<EnumRule<E>> createEnumRule(E defaultValue) {
		return createEnumRule(defaultValue, (server, rule) -> {
		});
	}

	/**
	 * Creates an enum rule type.
	 *
	 * <p>All enum values are supported.
	 *
	 * @param defaultValue the default value of the game rule
	 * @param changedCallback a callback that is invoked when the value of a game rule has changed
	 * @param <E> the type of enum this game rule stores
	 * @return an enum rule type
	 */
	public static <E extends Enum<E>> class_1928.class_4314<EnumRule<E>> createEnumRule(E defaultValue, BiConsumer<class_10961, EnumRule<E>> changedCallback) {
		return createEnumRule(defaultValue, defaultValue.getDeclaringClass().getEnumConstants(), changedCallback);
	}

	/**
	 * Creates an enum rule type.
	 *
	 * @param defaultValue the default value of the game rule
	 * @param supportedValues the values the game rule may support
	 * @param <E> the type of enum this game rule stores
	 * @return an enum rule type
	 */
	public static <E extends Enum<E>> class_1928.class_4314<EnumRule<E>> createEnumRule(E defaultValue, E[] supportedValues) {
		return createEnumRule(defaultValue, supportedValues, (server, rule) -> {
		});
	}

	/**
	 * Creates an enum rule type.
	 *
	 * @param defaultValue the default value of the game rule
	 * @param supportedValues the values the game rule may support
	 * @param changedCallback a callback that is invoked when the value of a game rule has changed.
	 * @param <E> the type of enum this game rule stores
	 * @return an enum rule type
	 */
	public static <E extends Enum<E>> class_1928.class_4314<EnumRule<E>> createEnumRule(E defaultValue, E[] supportedValues, BiConsumer<class_10961, EnumRule<E>> changedCallback) {
		checkNotNull(defaultValue, "Default rule value cannot be null");
		checkNotNull(supportedValues, "Supported Values cannot be null");

		if (supportedValues.length == 0) {
			throw new IllegalArgumentException("Cannot register an enum rule where no values are supported");
		}

		return new EnumRuleType<>(
				type -> new EnumRule<>(type, defaultValue, supportedValues),
				changedCallback,
				supportedValues,
				GameRuleFactory::visitEnum,
				(Class<EnumRule<E>>) (Object) EnumRule.class
		);
	}

	// RULE VISITORS - INTERNAL

	private static void visitDouble(class_1928.class_4311 visitor, class_1928.class_4313<DoubleRule> key, class_1928.class_4314<DoubleRule> type) {
		if (visitor instanceof FabricGameRuleVisitor) {
			((FabricGameRuleVisitor) visitor).visitDouble(key, type);
		}
	}

	private static <E extends Enum<E>> void visitEnum(class_1928.class_4311 visitor, class_1928.class_4313<EnumRule<E>> key, class_1928.class_4314<EnumRule<E>> type) {
		if (visitor instanceof FabricGameRuleVisitor) {
			((FabricGameRuleVisitor) visitor).visitEnum(key, type);
		}
	}

	private GameRuleFactory() {
	}
}
