/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.mixin.itemgroup.client;

import java.util.Comparator;
import java.util.List;
import java.util.Objects;

import org.lwjgl.glfw.GLFW;
import org.spongepowered.asm.mixin.Mixin;
import org.spongepowered.asm.mixin.Shadow;
import org.spongepowered.asm.mixin.Unique;
import org.spongepowered.asm.mixin.injection.At;
import org.spongepowered.asm.mixin.injection.Inject;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfo;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfoReturnable;
import net.fabricmc.fabric.api.client.itemgroup.v1.FabricCreativeInventoryScreen;
import net.fabricmc.fabric.impl.client.itemgroup.FabricCreativeGuiComponents;
import net.fabricmc.fabric.impl.itemgroup.FabricItemGroupImpl;
import net.minecraft.class_11908;
import net.minecraft.class_1661;
import net.minecraft.class_1761;
import net.minecraft.class_2561;
import net.minecraft.class_332;
import net.minecraft.class_465;
import net.minecraft.class_481;
import net.minecraft.class_481.class_483;
import net.minecraft.class_7706;

@Mixin(class_481.class)
public abstract class CreativeInventoryScreenMixin extends class_465<class_483> implements FabricCreativeInventoryScreen {
	public CreativeInventoryScreenMixin(class_483 screenHandler, class_1661 playerInventory, class_2561 text) {
		super(screenHandler, playerInventory, text);
	}

	@Shadow
	protected abstract void setSelectedTab(class_1761 itemGroup_1);

	@Shadow
	private static class_1761 selectedTab;

	// "static" matches selectedTab
	@Unique
	private static int currentPage = 0;

	@Unique
	private void updateSelection() {
		if (!isGroupVisible(selectedTab)) {
			class_7706.method_47341()
					.stream()
					.filter(this::isGroupVisible)
					.min((a, b) -> Boolean.compare(a.method_7752(), b.method_7752()))
					.ifPresent(this::setSelectedTab);
		}
	}

	@Inject(method = "init", at = @At(value = "INVOKE", target = "Lnet/minecraft/client/gui/widget/TextFieldWidget;setEditableColor(I)V", shift = At.Shift.AFTER))
	private void init(CallbackInfo info) {
		currentPage = getPage(selectedTab);

		int xpos = field_2776 + 171;
		int ypos = field_2800 + 4;

		class_481 self = (class_481) (Object) this;
		method_37063(new FabricCreativeGuiComponents.ItemGroupButtonWidget(xpos + 10, ypos, FabricCreativeGuiComponents.Type.NEXT, self));
		method_37063(new FabricCreativeGuiComponents.ItemGroupButtonWidget(xpos, ypos, FabricCreativeGuiComponents.Type.PREVIOUS, self));
	}

	@Inject(method = "setSelectedTab", at = @At("HEAD"), cancellable = true)
	private void setSelectedTab(class_1761 itemGroup, CallbackInfo info) {
		if (!isGroupVisible(itemGroup)) {
			info.cancel();
		}
	}

	@Inject(method = "renderTabTooltipIfHovered", at = @At("HEAD"), cancellable = true)
	private void renderTabTooltipIfHovered(class_332 drawContext, class_1761 itemGroup, int mx, int my, CallbackInfoReturnable<Boolean> info) {
		if (!isGroupVisible(itemGroup)) {
			info.setReturnValue(false);
		}
	}

	@Inject(method = "isClickInTab", at = @At("HEAD"), cancellable = true)
	private void isClickInTab(class_1761 itemGroup, double mx, double my, CallbackInfoReturnable<Boolean> info) {
		if (!isGroupVisible(itemGroup)) {
			info.setReturnValue(false);
		}
	}

	@Inject(method = "renderTabIcon", at = @At("HEAD"), cancellable = true)
	private void renderTabIcon(class_332 drawContext, class_1761 itemGroup, CallbackInfo info) {
		if (!isGroupVisible(itemGroup)) {
			info.cancel();
		}
	}

	@Inject(method = "keyPressed", at = @At("HEAD"), cancellable = true)
	private void keyPressed(class_11908 context, CallbackInfoReturnable<Boolean> cir) {
		if (context.comp_4795() == GLFW.GLFW_KEY_PAGE_UP) {
			if (switchToPreviousPage()) {
				cir.setReturnValue(true);
			}
		} else if (context.comp_4795() == GLFW.GLFW_KEY_PAGE_DOWN) {
			if (switchToNextPage()) {
				cir.setReturnValue(true);
			}
		}
	}

	@Unique
	private boolean isGroupVisible(class_1761 itemGroup) {
		return itemGroup.method_47311() && currentPage == getPage(itemGroup);
	}

	@Override
	public int getPage(class_1761 itemGroup) {
		if (FabricCreativeGuiComponents.COMMON_GROUPS.contains(itemGroup)) {
			return currentPage;
		}

		final FabricItemGroupImpl fabricItemGroup = (FabricItemGroupImpl) itemGroup;
		return fabricItemGroup.fabric_getPage();
	}

	@Unique
	private boolean hasGroupForPage(int page) {
		return class_7706.method_47335()
				.stream()
				.anyMatch(itemGroup -> getPage(itemGroup) == page);
	}

	@Override
	public boolean switchToPage(int page) {
		if (!hasGroupForPage(page)) {
			return false;
		}

		if (currentPage == page) {
			return false;
		}

		currentPage = page;
		updateSelection();
		return true;
	}

	@Override
	public int getCurrentPage() {
		return currentPage;
	}

	@Override
	public int getPageCount() {
		return FabricCreativeGuiComponents.getPageCount();
	}

	@Override
	public List<class_1761> getItemGroupsOnPage(int page) {
		return class_7706.method_47335()
				.stream()
				.filter(itemGroup -> getPage(itemGroup) == page)
				// Thanks to isXander for the sorting
				.sorted(Comparator.comparing(class_1761::method_47309).thenComparingInt(class_1761::method_7743))
				.sorted((a, b) -> Boolean.compare(a.method_7752(), b.method_7752()))
				.toList();
	}

	@Override
	public boolean hasAdditionalPages() {
		return class_7706.method_47335().size() > (Objects.requireNonNull(class_7706.field_42466).comp_1252() ? 14 : 13);
	}

	@Override
	public class_1761 getSelectedItemGroup() {
		return selectedTab;
	}

	@Override
	public boolean setSelectedItemGroup(class_1761 itemGroup) {
		Objects.requireNonNull(itemGroup, "itemGroup");

		if (selectedTab == itemGroup) {
			return false;
		}

		if (currentPage != getPage(itemGroup)) {
			if (!switchToPage(getPage(itemGroup))) {
				return false;
			}
		}

		setSelectedTab(itemGroup);
		return true;
	}
}
