/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.mixin.client.model.loading;

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.spongepowered.asm.mixin.Final;
import org.spongepowered.asm.mixin.Mixin;
import org.spongepowered.asm.mixin.Shadow;
import org.spongepowered.asm.mixin.Unique;
import org.spongepowered.asm.mixin.injection.At;
import org.spongepowered.asm.mixin.injection.Inject;
import org.spongepowered.asm.mixin.injection.ModifyVariable;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfo;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfoReturnable;
import net.fabricmc.fabric.impl.client.model.loading.ModelLoaderHooks;
import net.fabricmc.fabric.impl.client.model.loading.ModelLoadingEventDispatcher;
import net.fabricmc.fabric.impl.client.model.loading.ModelLoadingPluginManager;
import net.minecraft.class_1088;
import net.minecraft.class_1091;
import net.minecraft.class_1100;
import net.minecraft.class_2960;
import net.minecraft.class_324;
import net.minecraft.class_3695;
import net.minecraft.class_793;

@Mixin(class_1088.class)
public abstract class ModelLoaderMixin implements ModelLoaderHooks {
	// The missing model is always loaded and added first.
	@Final
	@Shadow
	public static class_1091 MISSING_ID;
	@Final
	@Shadow
	private Set<class_2960> modelsToLoad;
	@Final
	@Shadow
	private Map<class_2960, class_1100> unbakedModels;
	@Shadow
	@Final
	private Map<class_2960, class_1100> modelsToBake;

	@Unique
	private ModelLoadingEventDispatcher fabric_eventDispatcher;
	// Explicitly not @Unique to allow mods that heavily rework model loading to reimplement the guard.
	// Note that this is an implementation detail; it can change at any time.
	private int fabric_guardGetOrLoadModel = 0;
	private boolean fabric_enableGetOrLoadModelGuard = true;

	@Shadow
	private void addModel(class_1091 id) {
	}

	@Shadow
	public abstract class_1100 getOrLoadModel(class_2960 id);

	@Shadow
	private void loadModel(class_2960 id) {
	}

	@Shadow
	private void putModel(class_2960 id, class_1100 unbakedModel) {
	}

	@Shadow
	public abstract class_793 loadModelFromJson(class_2960 id);

	@Inject(method = "<init>", at = @At(value = "INVOKE", target = "net/minecraft/util/profiler/Profiler.swap(Ljava/lang/String;)V", ordinal = 0))
	private void afterMissingModelInit(class_324 blockColors, class_3695 profiler, Map<class_2960, class_793> jsonUnbakedModels, Map<class_2960, List<class_1088.class_7777>> blockStates, CallbackInfo info) {
		// Sanity check
		if (!unbakedModels.containsKey(MISSING_ID)) {
			throw new AssertionError("Missing model not initialized. This is likely a Fabric API porting bug.");
		}

		profiler.method_15405("fabric_plugins_init");

		fabric_eventDispatcher = new ModelLoadingEventDispatcher((class_1088) (Object) this, ModelLoadingPluginManager.CURRENT_PLUGINS.get());
		ModelLoadingPluginManager.CURRENT_PLUGINS.remove();
		fabric_eventDispatcher.addExtraModels(this::addModel);
	}

	@Unique
	private void addModel(class_2960 id) {
		if (id instanceof class_1091) {
			addModel((class_1091) id);
		} else {
			// The vanilla addModel method is arbitrarily limited to ModelIdentifiers,
			// but it's useful to tell the game to just load and bake a direct model path as well.
			// Replicate the vanilla logic of addModel here.
			class_1100 unbakedModel = getOrLoadModel(id);
			this.unbakedModels.put(id, unbakedModel);
			this.modelsToBake.put(id, unbakedModel);
		}
	}

	@Inject(method = "getOrLoadModel", at = @At("HEAD"))
	private void fabric_preventNestedGetOrLoadModel(class_2960 id, CallbackInfoReturnable<class_1100> cir) {
		if (fabric_enableGetOrLoadModelGuard && fabric_guardGetOrLoadModel > 0) {
			throw new IllegalStateException("ModelLoader#getOrLoadModel called from a ModelResolver or ModelModifier.OnBake instance. This is not allowed to prevent errors during model loading. Use getOrLoadModel from the context instead.");
		}
	}

	@Inject(method = "loadModel", at = @At("HEAD"), cancellable = true)
	private void onLoadModel(class_2960 id, CallbackInfo ci) {
		// Prevent calls to getOrLoadModel from loadModel as it will cause problems.
		// Mods should call getOrLoadModel on the ModelResolver.Context instead.
		fabric_guardGetOrLoadModel++;

		try {
			if (fabric_eventDispatcher.loadModel(id)) {
				ci.cancel();
			}
		} finally {
			fabric_guardGetOrLoadModel--;
		}
	}

	@ModifyVariable(method = "putModel", at = @At("HEAD"), argsOnly = true)
	private class_1100 onPutModel(class_1100 model, class_2960 id) {
		fabric_guardGetOrLoadModel++;

		try {
			return fabric_eventDispatcher.modifyModelOnLoad(id, model);
		} finally {
			fabric_guardGetOrLoadModel--;
		}
	}

	@Override
	public ModelLoadingEventDispatcher fabric_getDispatcher() {
		return fabric_eventDispatcher;
	}

	@Override
	public class_1100 fabric_getMissingModel() {
		return unbakedModels.get(MISSING_ID);
	}

	/**
	 * Unlike getOrLoadModel, this method supports nested model loading.
	 *
	 * <p>Vanilla does not due to the iteration over modelsToLoad which causes models to be resolved multiple times,
	 * possibly leading to crashes.
	 */
	@Override
	public class_1100 fabric_getOrLoadModel(class_2960 id) {
		if (this.unbakedModels.containsKey(id)) {
			return this.unbakedModels.get(id);
		}

		if (!modelsToLoad.add(id)) {
			throw new IllegalStateException("Circular reference while loading " + id);
		}

		try {
			loadModel(id);
		} finally {
			modelsToLoad.remove(id);
		}

		return unbakedModels.get(id);
	}

	@Override
	public void fabric_putModel(class_2960 id, class_1100 model) {
		putModel(id, model);
	}

	@Override
	public void fabric_putModelDirectly(class_2960 id, class_1100 model) {
		unbakedModels.put(id, model);
	}

	@Override
	public void fabric_queueModelDependencies(class_1100 model) {
		modelsToLoad.addAll(model.method_4755());
	}

	@Override
	public class_793 fabric_loadModelFromJson(class_2960 id) {
		return loadModelFromJson(id);
	}
}
